<?php

declare(strict_types=1);

namespace Drupal\filepond\Form;

use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\filepond\FilePondConfigFormTrait;
use Drupal\media\MediaSourceManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for FilePond module settings.
 *
 * Settings include:
 * - Enable/disable Media Library widget replacement
 * - Default upload settings (max files, extensions, file size, etc.)
 * - UI customization (aspect ratio, upload prompt)
 */
class FilePondSettingsForm extends ConfigFormBase {

  use FilePondConfigFormTrait;

  /**
   * The media source plugin manager.
   */
  protected MediaSourceManager $mediaSourcePluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->mediaSourcePluginManager = $container->get('plugin.manager.media.source');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'filepond_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['filepond.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('filepond.settings');

    $form['use_cdn'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load libraries from CDN'),
      '#description' => $this->t('Load FilePond JavaScript and CSS from jsDelivr CDN instead of local library files. Useful for quick setup or when library installation is not possible.'),
      '#default_value' => $config->get('use_cdn'),
    ];

    // Check if local libraries are installed.
    $missing_libraries = $this->checkMissingLibraries();
    if (!empty($missing_libraries)) {
      $form['library_warning'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="use_cdn"]' => ['checked' => FALSE],
          ],
        ],
        'message' => [
          '#theme' => 'status_messages',
          '#message_list' => [
            'warning' => [
              $this->t('Local libraries are not installed: @missing. Either enable CDN above or <a href=":readme" target="_blank">install the libraries</a>.', [
                '@missing' => implode(', ', $missing_libraries),
                ':readme' => 'https://www.drupal.org/project/filepond',
              ]),
            ],
          ],
        ],
      ];
    }

    $form['enable_media_library_widget'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable FilePond widget in Media Library'),
      '#description' => $this->t('When enabled, the Media Library upload form will use FilePond instead of the default file upload widget. <em>Applies to: @types</em>', ['@types' => implode(', ', filepond_get_file_based_media_types())]),
      '#default_value' => $config->get('enable_media_library_widget'),
    ];

    // Get defaults, merging with install defaults to ensure all keys exist.
    $defaults = array_merge($this->getInstallDefaults(), $config->get('defaults') ?? []);

    $form['defaults'] = [
      '#type' => 'details',
      '#title' => $this->t('Default settings'),
      '#description' => $this->t('These defaults apply when the property is not set on the FilePond element. Individual elements and widgets can override these values.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    // Constraints fieldset.
    $form['defaults']['constraints'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('File constraints'),
    ] + $this->makeFieldsRequired($this->getConstraintsFields($defaults));

    // Uploader fieldset.
    $form['defaults']['uploader'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Uploader'),
    ] + $this->makeFieldsRequired($this->getUploaderFields($defaults));

    // Display fieldset.
    $form['defaults']['display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display options'),
    ] + $this->makeFieldsRequired($this->getDisplayFields($defaults));

    // Flood control settings.
    $form['flood'] = [
      '#type' => 'details',
      '#title' => $this->t('Flood control'),
      '#description' => $this->t('Rate limiting to prevent upload abuse. Users with the "Bypass FilePond flood control" permission are exempt.'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    $form['flood']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable flood control'),
      '#description' => $this->t('When enabled, users are limited to a maximum number of uploads per time window.'),
      '#default_value' => $config->get('flood.enabled') ?? FALSE,
    ];

    $form['flood']['limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Upload limit per user'),
      '#description' => $this->t('Maximum number of files a user can upload within the time window.'),
      '#default_value' => $config->get('flood.limit') ?? 500,
      '#min' => 1,
      '#states' => [
        'visible' => [
          ':input[name="flood[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['flood']['window'] = [
      '#type' => 'number',
      '#title' => $this->t('Time window'),
      '#description' => $this->t('Time window in seconds. Default 3600 = 1 hour.'),
      '#default_value' => $config->get('flood.window') ?? 3600,
      '#min' => 60,
      '#field_suffix' => $this->t('seconds'),
      '#states' => [
        'visible' => [
          ':input[name="flood[enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Collect values from fieldsets under defaults.
    $defaults_values = $form_state->getValue('defaults') ?? [];
    $values = [
      'constraints' => $defaults_values['constraints'] ?? [],
      'uploader' => $defaults_values['uploader'] ?? [],
      'display' => $defaults_values['display'] ?? [],
    ];

    // Use trait's processor to flatten fieldset structure.
    $defaults = $this->processFilePondConfigValues($values);

    // Cast numeric values.
    $defaults['max_files'] = (int) ($defaults['max_files'] ?? 0);
    $defaults['chunk_size'] = (int) ($defaults['chunk_size'] ?? 5);
    $defaults['max_parallel_uploads'] = (int) ($defaults['max_parallel_uploads'] ?? 3);

    // Get flood control values.
    $flood = $form_state->getValue('flood') ?? [];

    $this->config('filepond.settings')
      ->set('use_cdn', (bool) $form_state->getValue('use_cdn'))
      ->set('enable_media_library_widget', $form_state->getValue('enable_media_library_widget'))
      ->set('flood.enabled', (bool) ($flood['enabled'] ?? FALSE))
      ->set('flood.limit', (int) ($flood['limit'] ?? 500))
      ->set('flood.window', (int) ($flood['window'] ?? 3600))
      ->set('defaults', $defaults)
      ->save();

    // Clear media source plugin cache so the form override takes effect.
    $this->mediaSourcePluginManager->clearCachedDefinitions();

    // Clear library discovery cache so CDN setting change takes effect.
    // Uses cache tag invalidation to avoid deprecated service/method APIs.
    Cache::invalidateTags(['library_info']);

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets the default values for initial installation.
   *
   * These are sensible defaults shown on the settings form. Unlike
   * getFilePondDefaults() which returns empty values for widgets,
   * this returns actual values for the global config.
   *
   * @return array
   *   Default configuration values.
   */
  protected function getInstallDefaults(): array {
    return [
      'upload_location' => 'public://[date:custom:Y]-[date:custom:m]',
      'max_filesize' => (int) (Environment::getUploadMaxSize() / pow(Bytes::KILOBYTE, 2)) . 'M',
      'extensions' => 'jpg jpeg gif png',
      'upload_prompt' => '',
      'max_files' => 0,
      'chunk_size' => 5,
      'max_parallel_uploads' => 3,
      'item_panel_aspect_ratio' => '1:1',
      'preview_fit_mode' => 'contain',
      'columns' => 5,
      'max_width' => '',
    ];
  }

  /**
   * Makes all form fields in an array required.
   *
   * @param array $fields
   *   Array of form field definitions.
   *
   * @return array
   *   The fields with #required set to TRUE.
   */
  protected function makeFieldsRequired(array $fields): array {
    foreach ($fields as &$field) {
      if (is_array($field) && isset($field['#type'])) {
        $field['#required'] = TRUE;
      }
    }
    return $fields;
  }

  /**
   * Checks which FilePond libraries are missing.
   *
   * @return array
   *   List of missing library names.
   */
  protected function checkMissingLibraries(): array {
    $libraries = [
      'filepond' => 'FilePond',
      'filepond-plugin-file-validate-type' => 'File Validate Type',
      'filepond-plugin-file-validate-size' => 'File Validate Size',
      'filepond-plugin-image-preview' => 'Image Preview',
      'filepond-plugin-file-poster' => 'File Poster',
      'filepond-plugin-image-crop' => 'Image Crop',
    ];

    $missing = [];
    foreach ($libraries as $library_dir => $library_name) {
      $path = DRUPAL_ROOT . '/libraries/' . $library_dir . '/dist';
      if (!is_dir($path)) {
        $missing[] = $library_name;
      }
    }

    return $missing;
  }

}
