<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Core\TempStore\SharedTempStore;
use Drupal\file\FileInterface;

/**
 * Helper for efficiently getting image dimensions.
 *
 * This class provides a unified approach to image dimension handling:
 *
 * 1. During upload (while file is local): Capture dimensions and store in
 *    tempstore. This is "free" since the file hasn't moved to S3 yet.
 *
 * 2. When dimensions are needed later: Check static cache, then tempstore,
 *    then fall back to FastImage (which reads only header bytes from S3).
 *
 * Tempstore auto-expires (default 7 days), so abandoned uploads don't leave
 * orphaned data. All results are also statically cached for the request
 * lifetime to avoid redundant lookups.
 */
class ImageDimensionHelper {

  /**
   * Tempstore collection name.
   */
  protected const TEMPSTORE_COLLECTION = 'filepond.dimensions';

  /**
   * Static cache of dimensions keyed by file ID.
   *
   * @var array<int|string, array{width: int, height: int}>
   */
  protected static array $cache = [];

  /**
   * Gets dimensions for a file entity.
   *
   * Checks in order:
   * 1. Static cache (from previous call in this request)
   * 2. Tempstore (from store() during upload)
   * 3. FastImage (reads only header bytes, works for S3)
   *
   * Results are cached for subsequent calls.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return array{width: int, height: int}|array{}
   *   The dimensions array, or empty array if not determinable.
   */
  public static function getDimensions(FileInterface $file): array {
    $fid = $file->id();

    // Check static cache first.
    if (isset(self::$cache[$fid])) {
      return self::$cache[$fid];
    }

    // Check tempstore (populated during upload).
    $tempstore = self::getTempstore();
    $dimensions = $tempstore->get((string) $fid);

    if ($dimensions) {
      // Clean up tempstore after retrieval - it's a one-time use.
      $tempstore->delete((string) $fid);
      self::$cache[$fid] = $dimensions;
      return $dimensions;
    }

    // Fallback to FastImage (reads only header bytes from S3).
    $dimensions = FastImage::getDimensions($file->getFileUri());

    if (!empty($dimensions)) {
      self::$cache[$fid] = $dimensions;
    }

    return $dimensions;
  }

  /**
   * Stores dimensions directly (when already known).
   *
   * Use this when you already have dimensions and want to store them
   * for later retrieval, without reading from file.
   *
   * @param int|string $fid
   *   The file entity ID.
   * @param int $width
   *   The image width.
   * @param int $height
   *   The image height.
   */
  public static function store(int|string $fid, int $width, int $height): void {
    $dimensions = [
      'width' => $width,
      'height' => $height,
    ];

    $tempstore = self::getTempstore();
    $tempstore->set((string) $fid, $dimensions);
    self::$cache[$fid] = $dimensions;
  }

  /**
   * Gets the shared tempstore for dimensions.
   *
   * Uses shared tempstore so dimensions stored by one user (uploader) can be
   * retrieved by the system during form processing.
   *
   * @return \Drupal\Core\TempStore\SharedTempStore
   *   The tempstore.
   */
  protected static function getTempstore(): SharedTempStore {
    return \Drupal::service('tempstore.shared')->get(self::TEMPSTORE_COLLECTION);
  }

}
