<?php

declare(strict_types=1);

namespace Drupal\filepond\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\filepond\FilePondConfigFormTrait;
use Drupal\filepond\ImageDimensionHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * FilePond-based image field widget.
 *
 * Unlike core's ImageWidget which uses AJAX form rebuilds for each upload,
 * this widget uses FilePond for all upload/display/reorder functionality.
 * File entities are created asynchronously during upload, not on form submit.
 *
 * Key differences from core:
 * - No AJAX callbacks or form rebuilds needed
 * - FilePond handles thumbnails, reordering, and multiple files
 * - No alt/title text inputs (use field defaults or tokens)
 * - Simpler code since FilePond handles all UI complexity
 */
#[FieldWidget(
  id: 'filepond_image',
  label: new TranslatableMarkup('FilePond Image Uploader'),
  field_types: ['image'],
)]
class FilePondImageWidget extends WidgetBase {

  use FilePondConfigFormTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected ImageFactory $imageFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->imageFactory = $container->get('image.factory');
    $instance->currentUser = $container->get('current_user');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'upload_prompt' => '',
      'max_files' => 0,
      'chunk_size' => NULL,
      'max_parallel_uploads' => NULL,
      'item_panel_aspect_ratio' => '',
      'preview_fit_mode' => '',
      'columns' => NULL,
      'max_width' => '',
      'preview_image_style' => 'thumbnail',
      'allow_reorder' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   *
   * Override to render as a single element regardless of cardinality.
   * FilePond handles multiple files internally - no need for Drupal's
   * table wrapper, drag handles, or "Add another item" button.
   */
  protected function formMultipleElements(
    FieldItemListInterface $items,
    array &$form,
    FormStateInterface $form_state,
  ): array {
    $field_name = $this->fieldDefinition->getName();
    $parents = $form['#parents'];

    // Build a single element for delta 0 - FilePond handles all files.
    $element = [
      '#title' => $this->fieldDefinition->getLabel(),
      '#description' => $this->getFilteredDescription(),
    ];

    $element = $this->formSingleElement($items, 0, $element, $form, $form_state);

    if ($element) {
      // Set parents to match field structure.
      $element['#field_parents'] = $parents;
      $element['#field_name'] = $field_name;
      $element['#delta'] = 0;
      $element['#weight'] = 0;

      // Wrap in container with field name for proper form handling.
      $elements = [
        '#field_name' => $field_name,
        '#field_parents' => $parents,
        '#required' => $this->fieldDefinition->isRequired(),
        '#max_delta' => 0,
        0 => $element,
      ];

      return $elements;
    }

    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $element = parent::settingsForm($form, $form_state);

    $element['preview_image_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Preview image style'),
      '#options' => image_style_options(FALSE),
      '#empty_option' => $this->t('None (original image)'),
      '#default_value' => $this->getSetting('preview_image_style'),
      '#description' => $this->t('Image style for FilePond thumbnails. Use a small style for performance.'),
    ];

    // Reordering only makes sense for multi-value fields.
    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    if ($cardinality === 1) {
      $element['allow_reorder'] = [
        '#type' => 'value',
        '#value' => FALSE,
      ];
    }
    else {
      $element['allow_reorder'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Allow reordering'),
        '#default_value' => $this->getSetting('allow_reorder'),
        '#description' => $this->t('Allow users to drag and drop files to reorder them.'),
      ];
    }

    $settings = $this->getSettings();
    $element['uploader'] = [
      '#type' => 'details',
      '#title' => $this->t('Uploader settings'),
      '#process' => [[static::class, 'flattenSettingsGroup']],
    ];
    $element['uploader'] += $this->getUploaderFields($settings);
    $element['uploader'] += $this->getDisplayFields($settings);

    return $element;
  }

  /**
   * Flattens fieldset settings variables.
   */
  public static function flattenSettingsGroup(&$element, FormStateInterface $form_state, &$complete_form) {
    array_pop($element['#parents']);
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];

    $image_style = $this->getSetting('preview_image_style');
    if ($image_style) {
      $styles = image_style_options(FALSE);
      $summary[] = $this->t('Preview: @style', [
        '@style' => $styles[$image_style] ?? $image_style,
      ]);
    }
    else {
      $summary[] = $this->t('Preview: Original image');
    }

    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    if ($cardinality !== 1 && $this->getSetting('allow_reorder')) {
      $summary[] = $this->t('Reordering enabled');
    }

    $aspect_ratio = $this->getSetting('item_panel_aspect_ratio');
    if ($aspect_ratio) {
      $summary[] = $this->t('Aspect ratio: @ratio', [
        '@ratio' => $aspect_ratio,
      ]);
    }

    $fit_mode = $this->getSetting('preview_fit_mode');
    $fit_labels = [
      'contain' => $this->t('Fit'),
      'cover' => $this->t('Crop'),
    ];
    $summary[] = $this->t('Preview: @mode', [
      '@mode' => $fit_labels[$fit_mode] ?? $fit_mode,
    ]);

    $columns = $this->getSetting('columns');
    $max_width = $this->getSetting('max_width');
    $size_parts = [];
    if ($columns) {
      $size_parts[] = $this->t('@cols columns', ['@cols' => $columns]);
    }
    if ($max_width) {
      $size_parts[] = $this->t('max: @w', ['@w' => $max_width]);
    }
    if ($size_parts) {
      $summary[] = $this->t('Grid: @sizes', ['@sizes' => implode(', ', $size_parts)]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(
    FieldItemListInterface $items,
    $delta,
    array $element,
    array &$form,
    FormStateInterface $form_state,
  ): array {
    // This is called per-delta, but FilePond handles all files at once.
    // Only build the element for delta 0.
    if ($delta > 0) {
      return [];
    }

    // Check permission - show preview or message if no upload access.
    if (!$this->currentUser->hasPermission('filepond upload files')) {
      return $this->buildNoAccessElement($element, $items);
    }

    $field_settings = $this->getFieldSettings();
    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $field_name = $this->fieldDefinition->getName();

    // Get file IDs already attached to the entity (for ownership validation).
    $entity_fids = [];
    foreach ($items as $item) {
      if (!empty($item->target_id)) {
        $entity_fids[] = $item->target_id;
      }
    }

    // Check form state for validation rebuilds, fall back to entity values.
    $default_fids = [];
    $user_input = $form_state->getUserInput();
    $input_fids = $user_input[$field_name][0]['fids'] ?? '';

    if (!empty($input_fids)) {
      // Parse semicolon-separated file IDs from hidden field.
      $fids = array_filter(array_map('intval', explode(';', $input_fids)));

      // Validate: allow files owned by current user OR already on entity.
      // This lets admins edit others' content while preventing injection.
      if (!empty($fids)) {
        $file_storage = $this->entityTypeManager->getStorage('file');
        $files = $file_storage->loadMultiple($fids);
        $current_uid = $this->currentUser->id();

        foreach ($files as $file) {
          $fid = $file->id();
          $is_owner = $file->getOwnerId() == $current_uid;
          $is_on_entity = in_array($fid, $entity_fids, TRUE);
          $is_admin = $current_uid == 1;

          if ($is_owner || $is_on_entity || $is_admin) {
            $default_fids[] = $fid;
          }
        }
      }
    }
    else {
      // Initial form load - use entity field values.
      $default_fids = $entity_fids;
    }

    // Get extensions from field settings (always set for image fields).
    $extensions = $field_settings['file_extensions'];

    // Build field-based upload URLs (no tempstore).
    $entity_type = $this->fieldDefinition->getTargetEntityTypeId();
    $bundle = $this->fieldDefinition->getTargetBundle();

    $process_url = Url::fromRoute('filepond.field_process', [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
      'field_name' => $field_name,
    ])->toString();

    $patch_url = Url::fromRoute('filepond.field_patch', [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
      'field_name' => $field_name,
      'transferId' => '__TRANSFER_ID__',
    ])->toString();
    $patch_url = str_replace('/__TRANSFER_ID__', '', $patch_url);

    $revert_url = Url::fromRoute('filepond.field_revert', [
      'entity_type' => $entity_type,
      'bundle' => $bundle,
      'field_name' => $field_name,
    ])->toString();

    // Build FilePond element directly on $element (not nested).
    // This allows contrib hooks like extra_field_description to work properly,
    // as they expect $element to be the actual form element.
    $element['#type'] = 'filepond';
    $element['#default_value'] = $default_fids;
    // Server-side properties (root level).
    // Use ?: NULL so empty values fall back to filepond.settings defaults.
    // Note: #max_files uses 0 for unlimited, so don't use ?: for it.
    $element['#extensions'] = $extensions;
    $element['#max_filesize'] = $field_settings['max_filesize'] ?: NULL;
    $element['#max_files'] = $this->resolveMaxFiles((int) ($this->getSetting('max_files') ?? 0), $cardinality);
    // Get upload location from first item, create temp item if empty.
    $first_item = $items->first() ?: $items->appendItem();
    $element['#upload_location'] = $first_item->getUploadLocation() ?: NULL;
    // CSS grid sizing.
    $element['#columns'] = $this->getSetting('columns') ?: NULL;
    $element['#max_width'] = $this->getSetting('max_width') ?: NULL;
    // FilePond UI options.
    $element['#config'] = [
      'allowReorder' => (bool) $this->getSetting('allow_reorder'),
      'previewImageStyle' => $this->getSetting('preview_image_style'),
      'styleItemPanelAspectRatio' => $this->getSetting('item_panel_aspect_ratio') ?: NULL,
      'previewFitMode' => $this->getSetting('preview_fit_mode') ?: NULL,
      // Use field-based routes.
      'processUrl' => $process_url,
      'patchUrl' => $patch_url,
      'revertUrl' => $revert_url,
    ];

    // Custom prompt if set.
    $prompt = $this->getSetting('upload_prompt');
    if ($prompt) {
      $element['#config']['labelIdle'] = $prompt;
    }

    return $element;
  }

  /**
   * Builds element for users without upload permission.
   *
   * Shows rendered previews of existing images or a warning message.
   *
   * @param array $element
   *   The base element array.
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field items.
   *
   * @return array
   *   The render array.
   */
  protected function buildNoAccessElement(array $element, FieldItemListInterface $items): array {
    // Check if there are existing images to display.
    $file_ids = [];
    foreach ($items as $item) {
      if (!empty($item->target_id)) {
        $file_ids[] = $item->target_id;
      }
    }

    if (empty($file_ids)) {
      // No existing images - show warning message.
      $element['no_access'] = [
        '#theme' => 'status_messages',
        '#message_list' => [
          'warning' => [$this->t("You don't have permission to upload files.")],
        ],
      ];
      return $element;
    }

    // Load files and render previews.
    $file_storage = $this->entityTypeManager->getStorage('file');
    $files = $file_storage->loadMultiple($file_ids);
    $image_style = $this->getSetting('preview_image_style');

    $element['preview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['filepond-preview-only']],
    ];

    foreach ($file_ids as $delta => $fid) {
      if (!isset($files[$fid])) {
        continue;
      }

      /** @var \Drupal\file\FileInterface $file */
      $file = $files[$fid];

      // Build image render array.
      $image_build = [
        '#theme' => 'image_style',
        '#style_name' => $image_style ?: 'thumbnail',
        '#uri' => $file->getFileUri(),
        '#alt' => $items[$delta]->alt ?? '',
        '#title' => $items[$delta]->title ?? '',
        '#attributes' => ['class' => ['filepond-preview-image']],
      ];

      // Fall back to plain image if no style selected.
      if (empty($image_style)) {
        $image_build = [
          '#theme' => 'image',
          '#uri' => $file->getFileUri(),
          '#alt' => $items[$delta]->alt ?? '',
          '#title' => $items[$delta]->title ?? '',
          '#attributes' => ['class' => ['filepond-preview-image']],
        ];
      }

      $element['preview'][$delta] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['filepond-preview-item']],
        'image' => $image_build,
      ];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(
    array $values,
    array $form,
    FormStateInterface $form_state,
  ): array {
    $field_settings = $this->getFieldSettings();
    $result = [];

    // FilePond element returns ['fids' => [123, 456, ...]].
    $fids = $values[0]['fids'] ?? [];

    if (empty($fids)) {
      return $result;
    }

    // Load files to get dimensions.
    /** @var \Drupal\file\FileStorageInterface $file_storage */
    $file_storage = $this->entityTypeManager->getStorage('file');
    $files = $file_storage->loadMultiple($fids);

    foreach ($fids as $fid) {
      if (!isset($files[$fid])) {
        continue;
      }

      /** @var \Drupal\file\FileInterface $file */
      $file = $files[$fid];

      // Get image dimensions via helper (checks tempstore, then FastImage).
      $dimensions = ImageDimensionHelper::getDimensions($file);

      $result[] = [
        'target_id' => $fid,
        // Alt/title: empty string if field uses them, NULL if not.
        // Could be enhanced with token support later.
        'alt' => !empty($field_settings['alt_field']) ? '' : NULL,
        'title' => !empty($field_settings['title_field']) ? '' : NULL,
        'width' => $dimensions['width'] ?? NULL,
        'height' => $dimensions['height'] ?? NULL,
      ];
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition): bool {
    // Only for image fields.
    return $field_definition->getType() === 'image';
  }

}
