<?php

declare(strict_types=1);

namespace Drupal\filepond;

/**
 * Interface for resolving upload settings from various sources.
 *
 * This service consolidates the logic for extracting upload configuration
 * (extensions, max size, destination) from different sources:
 * - State API (form elements)
 * - Field definitions
 * - Media type source fields
 * - Entity Browser widget configuration
 * - Views area plugin configuration.
 *
 * Each method returns an UploadOptions value object with standardized settings.
 */
interface UploadSettingsResolverInterface {

  /**
   * Stores form element config in tempstore.
   *
   * Config is stored in SharedTempStore keyed by form_id:element_name:hash.
   * The hash ensures config changes (including form_alter) are respected.
   * Auto-expires after 7 days to prevent orphaned entries.
   *
   * @param string $formId
   *   The form ID.
   * @param string $elementName
   *   The element name.
   * @param array $config
   *   The config array with keys: extensions, max_filesize, upload_location,
   *   and optionally context.
   *
   * @return string
   *   The config hash (for use in upload URLs).
   */
  public function storeFormConfig(string $formId, string $elementName, array $config): string;

  /**
   * Resolves upload settings from tempstore (form elements).
   *
   * @param string $formId
   *   The form ID.
   * @param string $elementName
   *   The element name.
   * @param string $configHash
   *   The config hash from the URL.
   *
   * @return \Drupal\filepond\UploadOptions|null
   *   Upload options or NULL if not found.
   */
  public function resolveFromState(string $formId, string $elementName, string $configHash): ?UploadOptions;

  /**
   * Resolves upload settings from a field definition.
   *
   * @param string $entityType
   *   The entity type ID.
   * @param string $bundle
   *   The bundle name.
   * @param string $fieldName
   *   The field name.
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If entity type or field not found.
   */
  public function resolveFromField(string $entityType, string $bundle, string $fieldName): UploadOptions;

  /**
   * Resolves upload settings from a media type's source field.
   *
   * @param string $mediaTypeId
   *   The media type ID.
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If media type not found or has no source field.
   */
  public function resolveFromMediaType(string $mediaTypeId): UploadOptions;

  /**
   * Resolves upload settings from an Entity Browser widget.
   *
   * @param string $entityBrowserId
   *   The entity browser ID.
   * @param string $widgetUuid
   *   The widget UUID.
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If entity browser or widget not found.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   If widget is not a FilePond widget or media type not configured.
   */
  public function resolveFromWidget(string $entityBrowserId, string $widgetUuid): UploadOptions;

  /**
   * Resolves upload settings from a Views area plugin.
   *
   * @param string $viewId
   *   The view ID.
   * @param string $displayId
   *   The display ID.
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options with views_area context flag.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   *   If view, display, or FilePond area plugin not found.
   */
  public function resolveFromViewsArea(string $viewId, string $displayId): UploadOptions;

}
