<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\NodeType;

/**
 * Tests FilePond image field widget.
 *
 * Tests the filepond_image widget when used on an image field:
 * - Regular (non-chunked) uploads
 * - Chunked uploads for large files
 * - File removal and replacement
 * - Node save with uploaded images.
 *
 * Test groups for selective CI runs:
 * - filepond: All FilePond tests
 * - filepond_image_field: All image field widget tests
 * - filepond_core: Essential upload/remove tests (run these first)
 * - filepond_chunked: Chunked upload tests (longer running)
 * - filepond_parallel: Simultaneous upload race condition tests
 * - filepond_edit: Edit existing node tests
 * - filepond_ui: UI state tests (button disabled, etc.)
 *
 * @group filepond
 * @group filepond_image_field
 */
class FilePondImageFieldTest extends FilePondTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'file',
    'image',
    'filepond',
  ];

  /**
   * A test user with content creation permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $contentUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a node type with an image field using FilePond widget.
    $this->createNodeTypeWithImageField();

    // Create user with content permissions.
    $this->contentUser = $this->drupalCreateUser([
      'access content',
      'create test_content content',
      'edit own test_content content',
      'filepond upload files',
    ]);
    $this->drupalLogin($this->contentUser);
  }

  /**
   * Creates a node type with an image field using FilePond widget.
   */
  protected function createNodeTypeWithImageField(): void {
    // Create node type.
    if (!NodeType::load('test_content')) {
      NodeType::create([
        'type' => 'test_content',
        'name' => 'Test Content',
      ])->save();
    }

    // Create image field storage.
    if (!FieldStorageConfig::loadByName('node', 'field_image')) {
      FieldStorageConfig::create([
        'field_name' => 'field_image',
        'entity_type' => 'node',
        'type' => 'image',
        'cardinality' => -1,
        'settings' => [
          'target_type' => 'file',
        ],
      ])->save();
    }

    // Create field instance.
    if (!FieldConfig::loadByName('node', 'test_content', 'field_image')) {
      FieldConfig::create([
        'field_name' => 'field_image',
        'entity_type' => 'node',
        'bundle' => 'test_content',
        'label' => 'Image',
        'settings' => [
          'file_directory' => 'test-images',
          'file_extensions' => 'png gif jpg jpeg webp',
          'max_filesize' => '',
          'alt_field' => FALSE,
          'title_field' => FALSE,
        ],
      ])->save();
    }

    // Configure form display to use FilePond widget.
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = \Drupal::service('entity_display.repository');
    $display_repository->getFormDisplay('node', 'test_content')
      ->setComponent('field_image', [
        'type' => 'filepond_image',
        'settings' => [
          'preview_image_style' => 'thumbnail',
          'allow_reorder' => TRUE,
        ],
      ])
      ->save();
  }

  /**
   * Tests regular (non-chunked) image upload via field widget.
   *
   * @group filepond_core
   */
  public function testRegularImageUpload(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // DIAGNOSTIC: Verify drupalSettings has the config needed for uploads.
    // If this fails, the config wasn't rendered or BigPipe didn't deliver it.
    $this->assertFilePondConfigured();

    // Verify CSRF token was fetched correctly.
    // This catches base path issues (e.g., hardcoded /session/token path).
    $this->assertCsrfTokenValid();

    // Create and upload a test image.
    $filePath = $this->createTestFile('regular-upload-test');
    $this->dropFileToFilePond($filePath);

    // Wait for upload to complete.
    $this->waitForFilePondComplete();

    // Verify upload succeeded with detailed error reporting.
    // If this fails, you'll see exactly why (file type, size, server error).
    $this->assertFilesUploadedSuccessfully(1);

    // DIAGNOSTIC: Verify hidden field exists in DOM.
    // If this fails, the form element naming is wrong (D10/D11 difference).
    $this->assertTrue(
      $this->hiddenFieldExists('field_image'),
      'Hidden field should exist at field_image[0][fids]. Diagnostic: ' .
      json_encode($this->getDiagnosticState('field_image'))
    );

    // Verify hidden field captured the file ID before submission.
    // Use wait version to handle timing differences between UI update and
    // JS hidden field update (varies between Drupal versions/CI environments).
    $fileIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(
      1,
      $fileIds,
      'Hidden field should contain 1 file ID. Diagnostic: ' .
      json_encode($this->getDiagnosticState('field_image'))
    );
    $submittedFid = $this->extractFileId($fileIds[0]);
    $this->assertTrue(is_numeric($submittedFid), 'File ID should be numeric');

    // Fill in required title and submit.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Test Node');
    $this->getSession()->getPage()->pressButton('Save');

    // Wait for page to load after form submission.
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was saved successfully.
    $this->assertSession()->pageTextContains('Test Content Test Node has been created');

    // Load the saved node and verify the image was actually attached.
    $node = $this->getNodeByTitle('Test Node');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertFalse($node->get('field_image')->isEmpty(), 'Node should have image attached');
    $this->assertCount(1, $node->get('field_image'), 'Node should have exactly 1 image');

    // Verify the saved file ID matches what was in the hidden field.
    $savedFid = $node->get('field_image')->target_id;
    $this->assertEquals($submittedFid, $savedFid, 'Saved file ID should match submitted file ID');
  }

  /**
   * Tests chunked image upload via field widget.
   *
   * @group filepond_chunked
   */
  public function testChunkedImageUpload(): void {
    // Set smaller chunk size to ensure chunking happens.
    $this->config('filepond.settings')
      ->set('defaults.chunk_size', 1)
      ->save();

    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token - critical for chunked uploads.
    $this->assertCsrfTokenValid();

    // Create and upload a large test file (2MB > 1MB chunk size).
    $filePath = $this->createLargeTestFile('chunked-upload-test', 2);
    $this->dropFileToFilePond($filePath);

    // Wait for chunked upload to complete (longer timeout).
    $this->waitForFilePondComplete('.filepond--root', 60000);

    // Verify upload succeeded with detailed error reporting.
    $this->assertFilesUploadedSuccessfully(1);

    // Verify transfer ID mapping was populated (indicates chunked upload).
    $this->assertTrue(
      $this->hasTransferIdMapping(),
      'Chunked upload should populate transfer ID mapping'
    );

    // Verify hidden field captured the file ID (critical for chunked uploads).
    $fileIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $fileIds, 'Hidden field should contain 1 file ID after chunked upload');
    $submittedFid = $this->extractFileId($fileIds[0]);
    $this->assertTrue(is_numeric($submittedFid), "File ID '$submittedFid' should be numeric");

    // Fill in required title and submit.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Chunked Test');
    $this->getSession()->getPage()->pressButton('Save');

    // Wait for page to load after form submission.
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was saved successfully.
    $this->assertSession()->pageTextContains('Test Content Chunked Test has been created');

    // Load the saved node and verify the chunked upload was attached.
    $node = $this->getNodeByTitle('Chunked Test');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertFalse($node->get('field_image')->isEmpty(), 'Node should have image attached');

    // Verify the saved file ID matches what was in the hidden field.
    $savedFid = $node->get('field_image')->target_id;
    $this->assertEquals($submittedFid, $savedFid, 'Saved file ID should match submitted file ID');
  }

  /**
   * Tests multiple image uploads and verifies all IDs are saved to node.
   *
   * @group filepond_core
   */
  public function testMultipleImageUpload(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token.
    $this->assertCsrfTokenValid();

    // Upload first image and wait for completion.
    $file1 = $this->createTestFile('multi-image-1');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();

    // Verify first upload succeeded.
    $this->assertFilesUploadedSuccessfully(1);
    $fileIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $fileIds, 'Hidden field should have 1 ID after first upload');

    // Upload second image and wait for completion.
    $file2 = $this->createTestFile('multi-image-2');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);

    // Wait for both files to be processing-complete.
    $this->assertSession()->waitForElementVisible(
      'css',
      '.filepond--item:nth-child(2)[data-filepond-item-state="processing-complete"]',
      30000
    );

    // Verify both files show in FilePond UI.
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Two files should display in FilePond');

    // Wait for hidden field to capture both file IDs (may lag behind UI).
    $fileIds = $this->waitForFieldWidgetFileIds('field_image', 2);
    $this->assertCount(2, $fileIds, 'Hidden field should contain 2 file IDs');
    $submittedFids = array_map([$this, 'extractFileId'], $fileIds);

    // Submit the form.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Multi Image Test');
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');
    $this->assertSession()->pageTextContains('Multi Image Test has been created');

    // Load the saved node and verify both images were attached.
    $node = $this->getNodeByTitle('Multi Image Test');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertCount(2, $node->get('field_image'), 'Node should have exactly 2 images');

    // Verify the saved file IDs match what was in the hidden field.
    $savedFids = array_column($node->get('field_image')->getValue(), 'target_id');
    sort($submittedFids);
    sort($savedFids);
    $this->assertEquals($submittedFids, $savedFids, 'Saved file IDs should match submitted file IDs');
  }

  /**
   * Tests image removal via field widget and verifies hidden field is cleared.
   *
   * @group filepond_core
   */
  public function testImageRemoval(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token.
    $this->assertCsrfTokenValid();

    // Upload an image.
    $filePath = $this->createTestFile('removal-test');
    $this->dropFileToFilePond($filePath);
    $this->waitForFilePondComplete();

    // Verify upload succeeded.
    $this->assertFilesUploadedSuccessfully(1);
    $fileIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $fileIds, 'Hidden field should have 1 ID after upload');

    // Remove the image via FilePond API.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );

    // Wait for removal.
    $this->waitForFilePondEmpty();
    $this->assertEquals(0, $this->getFilePondFileCount());

    // Verify hidden field was cleared.
    $fileIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(0, $fileIds, 'Hidden field should be empty after removal');

    // Fill in title (no image).
    $this->getSession()->getPage()->fillField('title[0][value]', 'No Image Test');
    $this->getSession()->getPage()->pressButton('Save');

    // Wait for page to load after form submission.
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was saved.
    $this->assertSession()->pageTextContains('No Image Test has been created');

    // Load the saved node and verify no images were attached.
    $node = $this->getNodeByTitle('No Image Test');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertTrue($node->get('field_image')->isEmpty(), 'Node should have no images');
  }

  /**
   * Tests that simultaneous uploads capture all file IDs in hidden field.
   *
   * Regression test: Ensures the hidden field correctly captures all file IDs
   * when multiple files are uploaded simultaneously. The updateHiddenField()
   * function in filepond.element.js rebuilds the hidden field from FilePond's
   * getFiles() state, which prevents race conditions.
   *
   * @group filepond_parallel
   */
  public function testSimultaneousUploadsHiddenFieldSync(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token before simultaneous uploads.
    $this->assertCsrfTokenValid();

    // Create 3 test files.
    $file1 = $this->createTestFile('simultaneous-1');
    $file2 = $this->createTestFile('simultaneous-2');
    $file3 = $this->createTestFile('simultaneous-3');

    // Drop all 3 files as rapidly as possible (NO waits between).
    // This triggers parallel uploads and exposes the race condition.
    $this->dropFileToFilePond($file1);
    $this->dropFileToFilePond($file2);
    $this->dropFileToFilePond($file3);

    // Wait for all 3 files to show in FilePond UI.
    $this->waitForFilePondFileCount(3);

    // Wait for all files to reach processing-complete state.
    $this->getSession()->wait(30000,
      "document.querySelectorAll('.filepond--item[data-filepond-item-state=\"processing-complete\"]').length === 3"
    );

    // Verify all 3 uploads succeeded with detailed error reporting.
    $this->assertFilesUploadedSuccessfully(3);

    // THE CRITICAL ASSERTION: Hidden field must contain exactly 3 file IDs.
    // This is where the race condition bug manifests - the hidden field
    // may only contain 1 or 2 IDs even though the UI shows 3 files.
    // Note: Field widget uses field_name[0][fids] format.
    $fileIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(
      3,
      $fileIds,
      sprintf(
        'Hidden field should contain 3 file IDs but found %d. IDs: [%s]. ' .
        'This indicates a race condition in the hidden field update logic.',
        count($fileIds),
        implode(', ', $fileIds)
      )
    );

    // Verify all IDs are valid (non-empty, proper format).
    foreach ($fileIds as $index => $signedId) {
      $this->assertNotEmpty($signedId, "File ID at index $index should not be empty");
      $fid = $this->extractFileId($signedId);
      $this->assertNotEmpty($fid, "Extracted file ID at index $index should not be empty");
      $this->assertTrue(is_numeric($fid), "File ID '$fid' should be numeric");
    }
  }

  /**
   * Tests simultaneous uploads with larger files.
   *
   * Regression test using 500KB files to verify hidden field sync works
   * correctly even with longer upload times and overlapping completions.
   *
   * @group filepond_parallel
   */
  public function testSimultaneousLargeUploadsHiddenFieldSync(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token.
    $this->assertCsrfTokenValid();

    // Create 3 medium-sized files (500KB each) to increase upload time.
    // Larger files = longer uploads = more chance of callback overlap.
    $file1 = $this->createMediumTestFile('large-simultaneous-1');
    $file2 = $this->createMediumTestFile('large-simultaneous-2');
    $file3 = $this->createMediumTestFile('large-simultaneous-3');

    // Drop all 3 files as rapidly as possible (NO waits between).
    $this->dropFileToFilePond($file1);
    $this->dropFileToFilePond($file2);
    $this->dropFileToFilePond($file3);

    // Wait for all 3 files to show in FilePond UI.
    $this->waitForFilePondFileCount(3);

    // Wait for all files to reach processing-complete state (longer timeout).
    $this->getSession()->wait(60000,
      "document.querySelectorAll('.filepond--item[data-filepond-item-state=\"processing-complete\"]').length === 3"
    );

    // Verify all 3 uploads succeeded.
    $this->assertFilesUploadedSuccessfully(3);

    // Check hidden field contains all 3 file IDs.
    $fileIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(
      3,
      $fileIds,
      sprintf(
        'Hidden field should contain 3 file IDs but found %d. IDs: [%s]. ' .
        'Race condition likely occurred during parallel uploads.',
        count($fileIds),
        implode(', ', $fileIds)
      )
    );
  }

  /**
   * Tests image replacement - remove one and add another.
   *
   * @group filepond_core
   */
  public function testImageReplacement(): void {
    $this->drupalGet('/node/add/test_content');

    // Wait for FilePond to initialize.
    $this->waitForFilePondInit();

    // Verify CSRF token.
    $this->assertCsrfTokenValid();

    // Upload first image.
    $file1 = $this->createTestFile('original-image');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Get original file ID.
    $originalIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $originalIds, 'Hidden field should have original ID');
    $originalFid = $this->extractFileId($originalIds[0]);

    // Remove the first image.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );
    $this->waitForFilePondEmpty();

    // Verify hidden field was cleared.
    $clearedIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(0, $clearedIds, 'Hidden field should be empty after removal');

    // Upload replacement image.
    $file2 = $this->createTestFile('replacement-image');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondComplete();
    $this->assertEquals(1, $this->getFilePondFileCount());

    // Get replacement file ID.
    $replacementIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(1, $replacementIds, 'Hidden field should have replacement ID');
    $replacementFid = $this->extractFileId($replacementIds[0]);

    // Verify replacement is different from original.
    $this->assertNotEquals($originalFid, $replacementFid, 'Replacement should be different file');

    // Submit the form.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Replaced Image Test');
    $this->getSession()->getPage()->pressButton('Save');

    // Wait for page to load after form submission.
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was saved successfully.
    $this->assertSession()->pageTextContains('Test Content Replaced Image Test has been created');

    // Load the saved node and verify only the replacement image was saved.
    $node = $this->getNodeByTitle('Replaced Image Test');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertCount(1, $node->get('field_image'), 'Node should have exactly 1 image');

    // Verify the saved file is the replacement, not the original.
    $savedFid = $node->get('field_image')->target_id;
    $this->assertEquals($replacementFid, $savedFid, 'Saved file should be replacement');
    $this->assertNotEquals($originalFid, $savedFid, 'Saved file should not be original');
  }

  /**
   * Tests editing a node with existing images and adding new ones.
   *
   * This verifies:
   * - Existing images are loaded into FilePond on edit
   * - New images can be added alongside existing ones
   * - All images (existing + new) are saved correctly.
   *
   * @group filepond_edit
   */
  public function testEditNodeAddImages(): void {
    // First create a node with one image.
    $this->drupalGet('/node/add/test_content');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    $file1 = $this->createTestFile('existing-image');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    $originalIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $originalIds, 'Hidden field should have 1 ID');
    $existingFid = $this->extractFileId($originalIds[0]);

    $this->getSession()->getPage()->fillField('title[0][value]', 'Edit Test Node');
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was created with 1 image.
    $node = $this->getNodeByTitle('Edit Test Node');
    $this->assertNotNull($node, 'Node should exist');
    $this->assertCount(1, $node->get('field_image'), 'Node should have 1 image');

    // Now edit the node.
    $this->drupalGet('/node/' . $node->id() . '/edit');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Wait for existing image to load in FilePond.
    $this->waitForFilePondFileCount(1);

    // Verify existing image is shown in FilePond.
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Existing image should be in FilePond');

    // Verify hidden field has existing image ID.
    // Use wait version because the hidden field is populated asynchronously
    // (100ms setTimeout after FilePond loads existing files).
    $editIds = $this->waitForFieldWidgetFileIds('field_image', 1);
    $this->assertCount(1, $editIds, 'Hidden field should have existing ID on edit');

    // Add a second image.
    $file2 = $this->createTestFile('new-image-on-edit');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);

    // Wait for new upload to complete.
    $this->assertSession()->waitForElementVisible(
      'css',
      '.filepond--item:nth-child(2)[data-filepond-item-state="processing-complete"]',
      30000
    );

    // Verify both images are in FilePond UI.
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should have 2 images in FilePond');

    // Verify hidden field has both IDs before submission.
    $bothIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(2, $bothIds, 'Hidden field should have 2 IDs (existing + new)');

    // Extract both FIDs.
    $submittedFids = array_map([$this, 'extractFileId'], $bothIds);
    $this->assertContains($existingFid, $submittedFids, 'Existing file ID should still be present');

    // Save the node.
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');
    $this->assertSession()->pageTextContains('Edit Test Node has been updated');

    // Reload the node and verify both images were saved.
    $nodeStorage = \Drupal::entityTypeManager()->getStorage('node');
    $nodeStorage->resetCache([$node->id()]);
    $node = $nodeStorage->load($node->id());

    $this->assertCount(2, $node->get('field_image'), 'Node should have 2 images after edit');

    // Verify both submitted file IDs are on the node.
    $savedFids = array_column($node->get('field_image')->getValue(), 'target_id');
    sort($submittedFids);
    sort($savedFids);
    $this->assertEquals($submittedFids, $savedFids, 'Saved file IDs should match submitted IDs');
  }

  /**
   * Tests editing a node and removing one of multiple existing images.
   *
   * @group filepond_edit
   */
  public function testEditNodeRemoveImage(): void {
    // First create a node with two images.
    $this->drupalGet('/node/add/test_content');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    $file1 = $this->createTestFile('keep-image');
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    $file2 = $this->createTestFile('remove-image');
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);

    $this->assertSession()->waitForElementVisible(
      'css',
      '.filepond--item:nth-child(2)[data-filepond-item-state="processing-complete"]',
      30000
    );

    // Wait for hidden field to capture both file IDs (may lag behind UI).
    $originalIds = $this->waitForFieldWidgetFileIds('field_image', 2);
    $this->assertCount(2, $originalIds, 'Hidden field should have 2 IDs');

    $this->getSession()->getPage()->fillField('title[0][value]', 'Remove Test Node');
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was created with 2 images.
    $node = $this->getNodeByTitle('Remove Test Node');
    $this->assertCount(2, $node->get('field_image'), 'Node should have 2 images');

    // Get the IDs of both images.
    $savedFids = array_column($node->get('field_image')->getValue(), 'target_id');
    $keepFid = $savedFids[0];

    // Edit the node.
    $this->drupalGet('/node/' . $node->id() . '/edit');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();
    $this->waitForFilePondFileCount(2);

    // Remove the second image (index 1).
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile(1);"
    );

    // Wait for removal.
    $this->waitForFilePondFileCount(1);

    // Verify hidden field now has only 1 ID.
    $afterRemovalIds = $this->getFieldWidgetFileIds('field_image');
    $this->assertCount(1, $afterRemovalIds, 'Hidden field should have 1 ID after removal');

    // Save the node.
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');

    // Reload and verify only 1 image remains.
    $nodeStorage = \Drupal::entityTypeManager()->getStorage('node');
    $nodeStorage->resetCache([$node->id()]);
    $node = $nodeStorage->load($node->id());

    $this->assertCount(1, $node->get('field_image'), 'Node should have 1 image after edit');

    // Verify the correct image was kept (the first one).
    $remainingFid = $node->get('field_image')->target_id;
    $this->assertEquals($keepFid, $remainingFid, 'The first image should be kept after removing the second');
  }

  /**
   * Tests that submit button is disabled during upload and enabled after.
   *
   * This prevents users from submitting the form while files are still
   * uploading, which would result in missing files.
   *
   * @group filepond_ui
   */
  public function testSubmitButtonStatesDuringUpload(): void {
    $this->drupalGet('/node/add/test_content');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Get the submit button.
    $submitButton = $this->getSession()->getPage()->findButton('Save');
    $this->assertNotNull($submitButton, 'Submit button should exist');

    // Submit button should be enabled before any uploads.
    $this->assertFalse($submitButton->hasAttribute('disabled'), 'Submit button should be enabled initially');

    // Create a medium file for slower upload.
    $filePath = $this->createMediumTestFile('submit-state-test');

    // Drop file and immediately check button state.
    $this->dropFileToFilePond($filePath);

    // Submit button should be disabled while uploading.
    // Wait a tiny bit for the upload to start.
    usleep(100000);

    // Note: Checking disabled state during upload is flaky if upload completes
    // too fast. We primarily verify the button is enabled after completion.
    // Wait for upload to complete.
    $this->waitForFilePondComplete();

    // After upload completes, submit button should be enabled again.
    // Give a moment for the JS to re-enable it.
    $this->getSession()->wait(1000,
      "document.querySelector('input[value=\"Save\"]')?.disabled === false"
    );

    $isEnabledAfterUpload = $this->getSession()->evaluateScript(
      "return document.querySelector('input[value=\"Save\"]')?.disabled === false;"
    );
    $this->assertTrue($isEnabledAfterUpload, 'Submit button should be enabled after upload completes');

    // Verify we can actually submit the form.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Submit State Test');
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');
    $this->assertSession()->pageTextContains('Submit State Test has been created');
  }

  /**
   * Tests submit button with multiple uploads - stays disabled until all done.
   *
   * @group filepond_ui
   */
  public function testSubmitButtonDisabledUntilAllUploadsComplete(): void {
    $this->drupalGet('/node/add/test_content');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Create two medium files.
    $file1 = $this->createMediumTestFile('multi-submit-1');
    $file2 = $this->createMediumTestFile('multi-submit-2');

    // Drop both files rapidly.
    $this->dropFileToFilePond($file1);
    $this->dropFileToFilePond($file2);

    // Wait for both to complete.
    $this->waitForFilePondFileCount(2);
    $this->getSession()->wait(30000,
      "document.querySelectorAll('.filepond--item[data-filepond-item-state=\"processing-complete\"]').length === 2"
    );

    // After both complete, button should be enabled.
    $this->getSession()->wait(2000,
      "document.querySelector('input[value=\"Save\"]')?.disabled === false"
    );

    $isEnabled = $this->getSession()->evaluateScript(
      "return document.querySelector('input[value=\"Save\"]')?.disabled === false;"
    );
    $this->assertTrue($isEnabled, 'Submit button should be enabled after all uploads complete');

    // Verify form submission works.
    $this->getSession()->getPage()->fillField('title[0][value]', 'Multi Submit Test');
    $this->getSession()->getPage()->pressButton('Save');
    $this->assertSession()->waitForElement('css', '.messages');

    // Verify node was saved with both images.
    $node = $this->getNodeByTitle('Multi Submit Test');
    $this->assertCount(2, $node->get('field_image'), 'Node should have 2 images');
  }

}
