<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

/**
 * Tests FilePond max files (cardinality) enforcement.
 *
 * Uses the built-in test form at /admin/config/media/filepond/test
 * with max_files set to 2 via config.
 *
 * @group filepond
 * @group filepond_core
 * @group filepond_maxfiles
 */
class FilePondMaxFilesTest extends FilePondTestBase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set max files to 2 for testing limits.
    $this->config('filepond.settings')
      ->set('defaults.max_files', 2)
      ->save();
  }

  /**
   * Tests max files limit is enforced in UI.
   */
  public function testMaxFilesEnforced(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    // Create test files.
    $file1 = $this->createTestFile('image1');
    $file2 = $this->createTestFile('image2');
    $file3 = $this->createTestFile('image3');

    // Upload first two files (should succeed).
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Small delay between uploads to avoid rate limiting in CI.
    usleep(500000);
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);
    $this->assertFilesUploadedSuccessfully(2);

    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should have 2 files');

    // Try to upload a third file (should be rejected by JS).
    $this->dropFileToFilePond($file3);

    // Wait a moment for any potential processing.
    usleep(500000);

    // Should still only have 2 files.
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should still have only 2 files after limit');
  }

  /**
   * Tests file removal allows new uploads.
   */
  public function testRemovalAllowsNewUpload(): void {
    $this->drupalGet('/admin/config/media/filepond/test');
    $this->waitForFilePondInit();
    $this->assertCsrfTokenValid();

    $file1 = $this->createTestFile('image1');
    $file2 = $this->createTestFile('image2');
    $file3 = $this->createTestFile('image3');

    // Fill to limit.
    $this->dropFileToFilePond($file1);
    $this->waitForFilePondComplete();
    $this->assertFilesUploadedSuccessfully(1);

    // Small delay between uploads to avoid rate limiting in CI.
    usleep(500000);
    $this->dropFileToFilePond($file2);
    $this->waitForFilePondFileCount(2);
    $this->assertFilesUploadedSuccessfully(2);
    $this->assertEquals(2, $this->getFilePondFileCount());

    // Remove one file via API.
    $this->getSession()->executeScript(
      "FilePond.find(document.querySelector('.filepond--root')).removeFile();"
    );

    // Wait for removal.
    $start = microtime(TRUE);
    while ($this->getFilePondFileCount() > 1 && (microtime(TRUE) - $start) < 5) {
      usleep(100000);
    }
    $this->assertEquals(1, $this->getFilePondFileCount(), 'Should have 1 file after removal');

    // Now we should be able to add another file.
    $this->dropFileToFilePond($file3);
    $this->waitForFilePondFileCount(2);
    $this->assertEquals(2, $this->getFilePondFileCount(), 'Should accept new file after removal');
  }

}
