<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Drupal\filepond\Controller\UploadController;
use Drupal\filepond\UploadSettingsResolverInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\user\RoleInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the FilePond upload controller.
 *
 * @group filepond
 * @group filepond_core
 */
class FilePondUploadControllerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'filepond',
  ];

  /**
   * Test file data.
   */
  protected string $testFileData = 'FilePond test file data';

  /**
   * Temp directory for test files.
   */
  protected string $tempDir;

  /**
   * The settings resolver service.
   */
  protected UploadSettingsResolverInterface $resolver;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installConfig(['filepond']);
    $this->installSchema('file', ['file_usage']);

    // Set up temp directory.
    $this->tempDir = $this->siteDirectory . '/files';
    \Drupal::service('file_system')->prepareDirectory(
      $this->tempDir,
      FileSystemInterface::CREATE_DIRECTORY
    );
    $this->setSetting('file_temp_path', $this->tempDir);

    // Create public files directory.
    $public_dir = 'public://filepond-test';
    \Drupal::service('file_system')->prepareDirectory(
      $public_dir,
      FileSystemInterface::CREATE_DIRECTORY
    );

    // Grant permission to upload.
    $role = Role::create([
      'id' => RoleInterface::AUTHENTICATED_ID,
      'label' => 'Authenticated',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();

    // Create user 1 first (admin) - we won't use this.
    // This ensures our test user is NOT uid=1 which has super user bypass.
    $admin = User::create([
      'name' => 'admin',
      'status' => 1,
    ]);
    $admin->save();

    // Create and set current user (will be uid=2, not admin).
    $user = User::create([
      'name' => 'test_user',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    // Get the settings resolver service.
    $this->resolver = $this->container->get('filepond.settings_resolver');
  }

  /**
   * Tests that process endpoint requires valid config in State API.
   */
  public function testProcessRequiresValidConfig(): void {
    $this->container->get('router.builder')->rebuild();

    // Request with form_id/element_name that has no config stored.
    $request = Request::create('/filepond/form/nonexistent_form/upload/process', 'POST');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should return 404 when config not found in State API.
    $this->assertEquals(404, $response->getStatusCode());
  }

  /**
   * Tests chunked upload initialization.
   */
  public function testChunkedUploadInitialization(): void {
    $this->container->get('router.builder')->rebuild();

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg', 'png'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    // Initialize chunked upload.
    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(200, $response->getStatusCode());
    // Response should contain a transfer ID.
    $transferId = $response->getContent();
    $this->assertNotEmpty($transferId);
    $this->assertMatchesRegularExpression('/^[a-zA-Z0-9_-]+$/', $transferId);
  }

  /**
   * Tests that invalid upload length is rejected.
   */
  public function testInvalidUploadLength(): void {
    $this->container->get('router.builder')->rebuild();

    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '0');
    $request->headers->set('Upload-Name', 'test.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(400, $response->getStatusCode());
  }

  /**
   * Tests field-based upload routes reject invalid entity types.
   */
  public function testFieldBasedRoutesInvalidEntityType(): void {
    $this->container->get('router.builder')->rebuild();

    // Field routes should reject requests for non-existent entity types.
    $request = Request::create(
      '/filepond/field/nonexistent_entity/bundle/field_name/process',
      'POST'
    );
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should return error (404 or 500) for non-existent entity type.
    $this->assertGreaterThanOrEqual(400, $response->getStatusCode());
  }

  /**
   * Tests revert endpoint rejects plain numeric IDs (security).
   *
   * Plain numeric IDs without HMAC signature should return 200 but NOT
   * actually delete any files. This prevents attackers from guessing file IDs.
   */
  public function testRevertRejectsPlainNumericIds(): void {
    $this->container->get('router.builder')->rebuild();

    // Create a temporary file that would be deletable if allowed.
    $file = File::create([
      'uri' => 'public://filepond-test/test-file.jpg',
      'filename' => 'test-file.jpg',
      'status' => 0,
      'uid' => $this->container->get('current_user')->id(),
    ]);
    $file->save();
    $fid = $file->id();

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'destination' => 'public://filepond-test',
    ]);

    // Send plain numeric ID (not signed).
    $request = Request::create("/filepond/form/test_form/upload/$hash/revert", 'DELETE', [], [], [], [], (string) $fid);
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should return 200 with action indicating it was skipped.
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('skipped_unsigned', $response->getContent());

    // File should still exist (was not deleted).
    $file_storage = $this->container->get('entity_type.manager')->getStorage('file');
    $file_storage->resetCache([$fid]);
    $reloaded = $file_storage->load($fid);
    $this->assertNotNull($reloaded, 'File should NOT be deleted with plain numeric ID');
  }

  /**
   * Tests revert endpoint accepts valid signed tokens.
   */
  public function testRevertAcceptsValidSignedToken(): void {
    $this->container->get('router.builder')->rebuild();

    // Create a temporary file.
    $file_uri = 'public://filepond-test/deletable.jpg';
    file_put_contents($file_uri, 'test content');

    $file = File::create([
      'uri' => $file_uri,
      'filename' => 'deletable.jpg',
      'status' => 0,
      'uid' => $this->container->get('current_user')->id(),
    ]);
    $file->save();
    $fid = $file->id();

    // Create a valid signed token using same algorithm as controller.
    $signedToken = $this->signFileId($fid);

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'destination' => 'public://filepond-test',
    ]);

    // Send signed token.
    $request = Request::create("/filepond/form/test_form/upload/$hash/revert", 'DELETE', [], [], [], [], $signedToken);
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('deleted', $response->getContent());

    // File should be deleted (temporary + no usage).
    $file_storage = $this->container->get('entity_type.manager')->getStorage('file');
    $file_storage->resetCache([$fid]);
    $reloaded = $file_storage->load($fid);
    $this->assertNull($reloaded, 'Temporary file should be deleted with valid signed token');
  }

  /**
   * Tests revert endpoint rejects invalid signed tokens.
   */
  public function testRevertRejectsInvalidSignedToken(): void {
    $this->container->get('router.builder')->rebuild();

    // Create a temporary file.
    $file = File::create([
      'uri' => 'public://filepond-test/protected.jpg',
      'filename' => 'protected.jpg',
      'status' => 0,
      'uid' => $this->container->get('current_user')->id(),
    ]);
    $file->save();
    $fid = $file->id();

    // Create an INVALID signed token (wrong HMAC).
    $invalidToken = $fid . ':invalid_hmac_value123456789';

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'destination' => 'public://filepond-test',
    ]);

    // Send invalid signed token.
    $request = Request::create("/filepond/form/test_form/upload/$hash/revert", 'DELETE', [], [], [], [], $invalidToken);
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should return 403 for invalid token.
    $this->assertEquals(403, $response->getStatusCode());
    $this->assertStringContainsString('Invalid', $response->getContent());

    // File should still exist (was not deleted).
    $file_storage = $this->container->get('entity_type.manager')->getStorage('file');
    $file_storage->resetCache([$fid]);
    $reloaded = $file_storage->load($fid);
    $this->assertNotNull($reloaded, 'File should NOT be deleted with invalid token');
  }

  /**
   * Tests revert endpoint allows transfer IDs (hex strings) for cleanup.
   */
  public function testRevertAllowsTransferIds(): void {
    $this->container->get('router.builder')->rebuild();

    // Transfer IDs are 32-char hex strings (not numeric, no colon).
    // Use the actual handler method to generate the ID.
    /** @var \Drupal\filepond\FilePondUploadHandler $handler */
    $handler = $this->container->get('filepond.upload_handler');
    $transferId = $handler->generateTransferId();

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'destination' => 'public://filepond-test',
    ]);

    // Send transfer ID (should pass through to handler).
    $request = Request::create("/filepond/form/test_form/upload/$hash/revert", 'DELETE', [], [], [], [], $transferId);
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should return 200 with temp_not_found (no temp file to clean).
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('temp_not_found', $response->getContent());
  }

  /**
   * Tests that permanent files are not deleted via revert.
   */
  public function testRevertDoesNotDeletePermanentFiles(): void {
    $this->container->get('router.builder')->rebuild();

    // Create a PERMANENT file.
    $file_uri = 'public://filepond-test/permanent.jpg';
    file_put_contents($file_uri, 'permanent content');

    $file = File::create([
      'uri' => $file_uri,
      'filename' => 'permanent.jpg',
      'status' => 1,
      'uid' => $this->container->get('current_user')->id(),
    ]);
    $file->save();
    $fid = $file->id();

    // Create a valid signed token.
    $signedToken = $this->signFileId($fid);

    // Store config in State API and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'destination' => 'public://filepond-test',
    ]);

    // Send valid signed token for permanent file.
    $request = Request::create("/filepond/form/test_form/upload/$hash/revert", 'DELETE', [], [], [], [], $signedToken);
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Returns success with action indicating it was skipped.
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertEquals('skipped_permanent', $response->getContent());

    // But file should NOT be deleted (it's permanent).
    $file_storage = $this->container->get('entity_type.manager')->getStorage('file');
    $file_storage->resetCache([$fid]);
    $reloaded = $file_storage->load($fid);
    $this->assertNotNull($reloaded, 'Permanent file should NOT be deleted');
    $this->assertTrue((bool) $reloaded->isPermanent(), 'File should still be permanent');
  }

  /**
   * Stores form config using the resolver service.
   *
   * @param string $form_id
   *   The form ID.
   * @param string $element_name
   *   The element name.
   * @param array $config
   *   Upload configuration.
   *
   * @return string
   *   The config hash to use in URLs.
   */
  protected function storeFormConfig(string $form_id, string $element_name, array $config): string {
    return $this->resolver->storeFormConfig($form_id, $element_name, $config);
  }

  /**
   * Signs a file ID using the actual controller method.
   *
   * @param int|string $fileId
   *   The file ID.
   *
   * @return string
   *   The signed token in format "fid:hmac".
   */
  protected function signFileId(int|string $fileId): string {
    /** @var \Drupal\filepond\Controller\UploadController $controller */
    $controller = $this->container->get('class_resolver')
      ->getInstanceFromDefinition(UploadController::class);
    return $controller->signFileId($fileId);
  }

  /**
   * Tests that flood control is disabled by default.
   */
  public function testFloodControlDisabledByDefault(): void {
    $this->container->get('router.builder')->rebuild();

    // Store config and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    // Make many requests - should all succeed since flood control is disabled.
    for ($i = 0; $i < 10; $i++) {
      $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
      $request->headers->set('Upload-Length', '1024');
      $request->headers->set('Upload-Name', "test-image-$i.jpg");
      $request->headers->set('X-CSRF-Token', 'test');

      /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
      $http_kernel = $this->container->get('http_kernel');
      $response = $http_kernel->handle($request);

      $this->assertEquals(200, $response->getStatusCode(), "Request $i should succeed");
    }
  }

  /**
   * Tests that flood control blocks after limit is exceeded.
   */
  public function testFloodControlBlocksAfterLimit(): void {
    $this->container->get('router.builder')->rebuild();

    // Enable flood control with low limit.
    $this->config('filepond.settings')
      ->set('flood.enabled', TRUE)
      ->set('flood.limit', 3)
      ->set('flood.window', 3600)
      ->save();

    // Store config and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    // First 3 requests should succeed.
    for ($i = 0; $i < 3; $i++) {
      $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
      $request->headers->set('Upload-Length', '1024');
      $request->headers->set('Upload-Name', "test-image-$i.jpg");
      $request->headers->set('X-CSRF-Token', 'test');

      /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
      $http_kernel = $this->container->get('http_kernel');
      $response = $http_kernel->handle($request);

      $this->assertEquals(200, $response->getStatusCode(), "Request $i should succeed");
    }

    // 4th request should be blocked.
    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image-blocked.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(429, $response->getStatusCode(), 'Request should be rate limited');
  }

  /**
   * Tests that bypass permission skips flood control.
   */
  public function testFloodControlBypassPermission(): void {
    $this->container->get('router.builder')->rebuild();

    // Enable flood control with very low limit.
    $this->config('filepond.settings')
      ->set('flood.enabled', TRUE)
      ->set('flood.limit', 1)
      ->set('flood.window', 3600)
      ->save();

    // Grant bypass permission.
    $role = Role::load(RoleInterface::AUTHENTICATED_ID);
    $role->grantPermission('bypass filepond flood control');
    $role->save();

    // Store config and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    // Multiple requests should all succeed due to bypass permission.
    for ($i = 0; $i < 5; $i++) {
      $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
      $request->headers->set('Upload-Length', '1024');
      $request->headers->set('Upload-Name', "test-image-$i.jpg");
      $request->headers->set('X-CSRF-Token', 'test');

      /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
      $http_kernel = $this->container->get('http_kernel');
      $response = $http_kernel->handle($request);

      $this->assertEquals(200, $response->getStatusCode(), "Request $i should succeed with bypass");
    }
  }

  /**
   * Tests that PATCH (chunk) requests don't count toward flood limit.
   */
  public function testFloodControlDoesNotCountChunks(): void {
    $this->container->get('router.builder')->rebuild();

    // Enable flood control with low limit.
    $this->config('filepond.settings')
      ->set('flood.enabled', TRUE)
      ->set('flood.limit', 2)
      ->set('flood.window', 3600)
      ->save();

    // Store config and get hash.
    $hash = $this->storeFormConfig('test_form', 'upload', [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ]);

    // Initialize first upload (counts as 1).
    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image-1.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);
    $this->assertEquals(200, $response->getStatusCode());
    $transferId = $response->getContent();

    // Send many PATCH requests (chunks) - should not count toward limit.
    for ($i = 0; $i < 10; $i++) {
      $request = Request::create(
        "/filepond/form/test_form/upload/$hash/patch/" . $transferId,
        'PATCH',
        [],
        [],
        [],
        ['CONTENT_TYPE' => 'application/offset+octet-stream'],
        'chunk-data'
      );
      $request->headers->set('Upload-Offset', (string) ($i * 100));
      $request->headers->set('Upload-Length', '1024');
      $request->headers->set('Upload-Name', 'test-image-1.jpg');
      $request->headers->set('X-CSRF-Token', 'test');

      $response = $http_kernel->handle($request);
      // Chunks may return various codes, but not 429.
      $this->assertNotEquals(429, $response->getStatusCode(), "Chunk $i should not be rate limited");
    }

    // Second process request (counts as 2).
    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image-2.jpg');
    $request->headers->set('X-CSRF-Token', 'test');
    $response = $http_kernel->handle($request);
    $this->assertEquals(200, $response->getStatusCode(), 'Second process request should succeed');

    // Third process request should be blocked (limit is 2).
    $request = Request::create("/filepond/form/test_form/upload/$hash/process", 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image-3.jpg');
    $request->headers->set('X-CSRF-Token', 'test');
    $response = $http_kernel->handle($request);
    $this->assertEquals(429, $response->getStatusCode(), 'Third request should be rate limited');
  }

}
