<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\Core\File\FileSystemInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\user\RoleInterface;

/**
 * Tests the FilePond upload handler service.
 *
 * @group filepond
 * @group filepond_core
 */
class FilePondUploadHandlerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'filepond',
  ];

  /**
   * The upload handler service.
   *
   * @var \Drupal\filepond\FilePondUploadHandler
   */
  protected $uploadHandler;

  /**
   * Temp directory for test files.
   */
  protected string $tempDir;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installConfig(['filepond']);
    $this->installSchema('file', ['file_usage']);

    // Set up directories.
    $this->tempDir = $this->siteDirectory . '/files';
    \Drupal::service('file_system')->prepareDirectory(
      $this->tempDir,
      FileSystemInterface::CREATE_DIRECTORY
    );
    $this->setSetting('file_temp_path', $this->tempDir);

    $public_dir = 'public://filepond-test';
    \Drupal::service('file_system')->prepareDirectory(
      $public_dir,
      FileSystemInterface::CREATE_DIRECTORY
    );

    // Grant permission.
    $role = Role::create([
      'id' => RoleInterface::AUTHENTICATED_ID,
      'label' => 'Authenticated',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();

    // Create and set current user.
    $user = User::create([
      'name' => 'test_user',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    $this->uploadHandler = $this->container->get('filepond.upload_handler');
  }

  /**
   * Tests chunked upload initialization.
   */
  public function testInitializeChunkedUpload(): void {
    $options = [
      'allowed_extensions' => ['jpg', 'png', 'gif'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ];

    $result = $this->uploadHandler->initializeChunkedUpload(
      1024,
      'test-image.jpg',
      $options
    );

    $this->assertTrue($result['success']);
    $this->assertNotEmpty($result['transfer_id']);
  }

  /**
   * Tests that invalid extensions are rejected.
   */
  public function testInvalidExtensionRejected(): void {
    $options = [
      'allowed_extensions' => ['jpg', 'png'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ];

    $result = $this->uploadHandler->initializeChunkedUpload(
      1024,
      'malicious.exe',
      $options
    );

    $this->assertFalse($result['success']);
    $this->assertStringContainsString('type', strtolower($result['error']));
  }

  /**
   * Tests that file size exceeding max is rejected.
   */
  public function testFileSizeExceedsMax(): void {
    $options = [
      'allowed_extensions' => ['jpg'],
      'max_size' => 1024,
      'destination' => 'public://filepond-test',
    ];

    $result = $this->uploadHandler->initializeChunkedUpload(
      2048,
      'large-image.jpg',
      $options
    );

    $this->assertFalse($result['success']);
    $this->assertStringContainsString('large', strtolower($result['error']));
  }

  /**
   * Tests transfer ID generation is unique.
   */
  public function testTransferIdUniqueness(): void {
    $options = [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ];

    $ids = [];
    for ($i = 0; $i < 10; $i++) {
      $result = $this->uploadHandler->initializeChunkedUpload(
        1024,
        "image-$i.jpg",
        $options
      );
      $this->assertTrue($result['success']);
      $ids[] = $result['transfer_id'];
    }

    // All IDs should be unique.
    $this->assertCount(10, array_unique($ids));
  }

  /**
   * Tests filename with special characters is accepted.
   */
  public function testFilenameWithSpecialCharacters(): void {
    $options = [
      'allowed_extensions' => ['jpg'],
      'max_size' => 10 * 1024 * 1024,
      'destination' => 'public://filepond-test',
    ];

    // Test with special characters in filename.
    $result = $this->uploadHandler->initializeChunkedUpload(
      1024,
      'test image (1) & special!.jpg',
      $options
    );

    // Should succeed - sanitization happens during finalization.
    $this->assertTrue($result['success']);
    $this->assertNotEmpty($result['transfer_id']);
  }

  /**
   * Tests the isRemoteUri method for local stream wrappers.
   *
   * Uses reflection to test the protected method directly.
   */
  public function testIsRemoteUriWithLocalStreams(): void {
    $reflection = new \ReflectionMethod($this->uploadHandler, 'isRemoteUri');

    // public:// should be considered local (not remote).
    $this->assertFalse(
      $reflection->invoke($this->uploadHandler, 'public://test.jpg'),
      'public:// should be identified as local'
    );

    // temporary:// should be considered local.
    $this->assertFalse(
      $reflection->invoke($this->uploadHandler, 'temporary://test.jpg'),
      'temporary:// should be identified as local'
    );

    // Non-existent scheme should return FALSE (graceful handling).
    $this->assertFalse(
      $reflection->invoke($this->uploadHandler, 'nonexistent://test.jpg'),
      'Non-existent scheme should return FALSE'
    );

    // No scheme should return FALSE.
    $this->assertFalse(
      $reflection->invoke($this->uploadHandler, '/var/tmp/test.jpg'),
      'Path without scheme should return FALSE'
    );
  }

}
