# FilePond Module

A FilePond-based file upload element for Drupal forms with chunked uploads,
drag-and-drop, image previews, and reordering support.

This module is similar in functionality to
[DropzoneJS](https://www.drupal.org/project/dropzonejs), but uses the
[FilePond](https://pqina.nl/filepond/) JavaScript library instead. If you're
familiar with DropzoneJS, you'll find a similar architecture here: a form
element, Entity Browser widget, and Media Library integration.

## Table of Contents

- [Installation](#installation)
- [Configuration](#configuration)
  - [Global Settings](#global-settings)
  - [Element Properties](#element-properties)
  - [Reserved Options](#reserved-options-cannot-override-in-config)
  - [Image Preview Modes](#image-preview-modes)
- [Usage](#usage)
  - [Field Widget](#field-widget)
  - [Entity Browser Widget](#entity-browser-widget)
  - [Views Area Plugin](#views-area-plugin)
  - [Media Library Integration](#media-library-integration)
  - [Form Element](#form-element)
- [Events](#events)
  - [FilePondUploadPreMoveEvent](#fileponduploadpremoveevent)
  - [FilePondUploadCompleteEvent](#fileponduploadcompleteevent)
- [Permissions](#permissions)
- [Altering Elements](#altering-elements)
- [Submit Button Control](#submit-button-control)
- [Extending](#extending)
  - [Custom Upload Settings](#custom-upload-settings)
  - [Adding Context Data](#adding-context-data)
- [Submodules](#submodules)
- [Limitations](#limitations)
- [Resources](#resources)

## Installation

```bash
composer require drupal/filepond
drush en filepond
```

That's it! The module uses **jsDelivr CDN** by default, so no library
installation is required. It works out of the box.

### Self-Hosting Libraries (Optional)

If you prefer to host the libraries yourself instead of using the CDN, disable
"Load libraries from CDN" in **Configuration → Media → FilePond** and install
the libraries using one of these methods:

#### Option 1: Composer with Asset Packagist (Recommended)

```bash
composer require npm-asset/filepond \
  npm-asset/filepond-plugin-file-validate-type \
  npm-asset/filepond-plugin-file-validate-size \
  npm-asset/filepond-plugin-image-preview \
  npm-asset/filepond-plugin-file-poster \
  npm-asset/filepond-plugin-image-crop
```

**Requirements:**
- [Asset Packagist](https://asset-packagist.org/) repository configured in
  composer.json
- `oomphinc/composer-installers-extender` package
- Installer path configured for npm-asset packages to install to `libraries/`

#### Option 2: Custom Package Repository

If you're not using Asset Packagist, add the module's `composer.libraries.json`
to your project's merge configuration. First, install the
[Composer Merge Plugin](https://github.com/wikimedia/composer-merge-plugin):

```bash
composer require wikimedia/composer-merge-plugin
```

Then add to your project's `composer.json`:

```json
{
    "extra": {
        "merge-plugin": {
            "include": [
                "web/modules/contrib/filepond/composer.libraries.json"
            ]
        }
    }
}
```

Then require the libraries:

```bash
composer require pqina/filepond \
  pqina/filepond-plugin-file-validate-type \
  pqina/filepond-plugin-file-validate-size \
  pqina/filepond-plugin-image-preview \
  pqina/filepond-plugin-file-poster \
  pqina/filepond-plugin-image-crop
```

#### Option 3: Manual Download

Download each library from [unpkg.com](https://unpkg.com/) (an npm CDN) and
place in `libraries/`:

```
libraries/filepond/dist/filepond.min.js
libraries/filepond/dist/filepond.min.css
libraries/filepond-plugin-file-validate-type/dist/filepond-plugin-file-validate-type.min.js
libraries/filepond-plugin-file-validate-size/dist/filepond-plugin-file-validate-size.min.js
libraries/filepond-plugin-image-preview/dist/filepond-plugin-image-preview.min.js
libraries/filepond-plugin-image-preview/dist/filepond-plugin-image-preview.min.css
libraries/filepond-plugin-file-poster/dist/filepond-plugin-file-poster.min.js
libraries/filepond-plugin-file-poster/dist/filepond-plugin-file-poster.min.css
libraries/filepond-plugin-image-crop/dist/filepond-plugin-image-crop.min.js
```

**Direct download links:**

- [filepond](https://unpkg.com/filepond/dist/)
- [filepond-plugin-file-validate-type](https://unpkg.com/filepond-plugin-file-validate-type/dist/)
- [filepond-plugin-file-validate-size](https://unpkg.com/filepond-plugin-file-validate-size/dist/)
- [filepond-plugin-image-preview](https://unpkg.com/filepond-plugin-image-preview/dist/)
- [filepond-plugin-file-poster](https://unpkg.com/filepond-plugin-file-poster/dist/)
- [filepond-plugin-image-crop](https://unpkg.com/filepond-plugin-image-crop/dist/)

## Configuration

### Global Settings

Configure defaults at **Administration > Configuration > Media > FilePond**
(`/admin/config/media/filepond`).

### Element Properties

#### Server-Side Properties (Root Level)

These control validation and file storage:

| Property | Type | Default | Description |
|----------|------|---------|-------------|
| `#extensions` | string | `'jpg jpeg gif png'` | Space-separated allowed extensions |
| `#max_filesize` | string | PHP upload_max_filesize | Max file size (e.g., `'20M'`) |
| `#max_files` | int | From module config | Maximum files. `0` = unlimited |
| `#upload_location` | string | `'public://filepond-uploads'` | Destination URI (supports tokens) |
| `#default_value` | array | `[]` | Existing file entity IDs |

#### UI Options (#config)

Pass FilePond options via `#config`:

```php
$form['images'] = [
  '#type' => 'filepond',
  '#config' => [
    'labelIdle' => 'Drop files here or <span class="filepond--label-action">Browse</span>',
    'allowReorder' => TRUE,
    'maxParallelUploads' => 4,
  ],
];
```

**Common options:**

| Option | Type | Default | Description |
|--------|------|---------|-------------|
| `labelIdle` | string | `'Drag & drop...'` | Drop zone prompt text |
| `allowReorder` | bool | `FALSE` | Enable drag-to-reorder |
| `maxParallelUploads` | int | `3` | Simultaneous uploads |
| `styleItemPanelAspectRatio` | float | `0.5625` | Panel aspect ratio (height/width) |
| `previewFitMode` | string | `'contain'` | `'contain'` or `'cover'` |
| `previewImageStyle` | string | `NULL` | Drupal image style for thumbnails |
| `labelProcessing` | string | `'Processing...'` | Text during server processing |
| `imagePreviewMaxHeight` | int | `256` | Max preview height in pixels |

See [FilePond docs](https://pqina.nl/filepond/docs/api/instance/properties/) for
all available options.

### Reserved Options (Cannot Override in #config)

The following options are controlled by root-level properties and **cannot** be
set in `#config`. If you try to set them, they will be silently removed:

| #config key | Use this instead | Why |
|-------------|------------------|-----|
| `extensions` | `#extensions` | Server-side validation source of truth |
| `maxFilesize` | `#max_filesize` | Server-side validation |
| `maxFiles` | `#max_files` | Server-side limit enforcement |
| `uploadLocation` | `#upload_location` | Server-side file destination |
| `acceptedFileTypes` | `#extensions` | Auto-derived from extensions |

**Why this restriction?**

These options control server-side validation. Allowing them in `#config` would
risk mismatches between what the server validates and what the UI shows. For
example, if `acceptedFileTypes` could be set separately, users might select
files the UI says are allowed but the server rejects.

```php
// WRONG - these #config values will be ignored:
$form['images']['#extensions'] = 'jpg png';
$form['images']['#config']['extensions'] = 'jpg png gif exe';  // Ignored!
$form['images']['#config']['acceptedFileTypes'] = ['image/gif'];  // Ignored!

// RIGHT - use root properties:
$form['images']['#extensions'] = 'jpg png gif';
```

### Image Preview Modes

The `previewFitMode` option controls how image previews display:

- **`contain`** (default): Contains the image within the panels bounds.
- **`cover`**: Crops image to fill panel (uses Image Crop plugin)

```php
'#config' => [
  'styleItemPanelAspectRatio' => 0.5625,  // 16:9 panels
  'previewFitMode' => 'cover',             // Crop to fill
  'imagePreviewMaxHeight' => 300,          // Constrain height
],
```

When using `cover` mode, `imagePreviewMaxHeight` helps the Image Crop plugin
work correctly by constraining the preview dimensions.

### Aspect Ratio Reference

Common values for `styleItemPanelAspectRatio` (height/width):

| Ratio | Value | Description |
|-------|-------|-------------|
| 16:9 | 0.5625 | Landscape (default) |
| 4:3 | 0.75 | Landscape |
| 1:1 | 1.0 | Square |
| 3:4 | 1.333 | Portrait |
| 9:16 | 1.778 | Portrait |

## Field Widget

The module includes a FilePond widget for image fields.

### Widget Settings

Configure in **Manage form display** for any image field:

- **Preview image style**: Drupal image style for existing image thumbnails
- **Allow reorder**: Enable drag-to-reorder
- **Panel aspect ratio**: Ratio string like `16:9`, `4:3`, or `1:1`
- **Preview fit mode**: `contain` (letterbox) or `cover` (crop)

### Field-Based Routes

The widget uses field-based upload routes that derive settings from the field
configuration:

- Upload location from field's file directory setting
- Extensions from field's allowed file types
- Max filesize from field settings

No additional configuration needed - it reads directly from your field setup.

## Entity Browser Widget

The filepond_eb_widget submodule provides a FilePond widget for Entity Browser:

```bash
drush en filepond_eb_widget
```

Configure in your Entity Browser under Widgets > Add widget > "FilePond Media Upload".

## Views Area Plugin

The `filepond_views` submodule provides a Views area handler that displays a
FilePond uploader. When files are uploaded, media entities are created and the
view refreshes to show the new items.

```bash
drush en filepond_views
```

### Use Case

The primary use case is combining FilePond with Entity Browser:

1. Create a View that displays media items
2. Add the "FilePond Upload" area handler to the header
3. Configure which media type to create
4. Use the View in an Entity Browser widget

When users upload files, media entities are created automatically and appear in
the view grid for selection.

### Configuration

Add the area handler in Views UI:

1. Edit your media View
2. Add a Header (or Footer/Empty) area
3. Select "FilePond Upload"
4. Configure:
   - **Media type**: Which media type to create from uploads
   - **Inherit settings**: Use the media type's field settings for extensions/size
   - **Max files**: Limit uploads per session (0 = unlimited)

### Cardinality Enforcement

When used in Entity Browser, the uploader respects the field's cardinality. If
a field allows 3 images and 2 are already selected, FilePond will only allow 1
more upload.

## Media Library Integration

Replace Drupal's default Media Library upload widget:

1. Go to `/admin/config/media/filepond`
2. Enable "FilePond widget in Media Library"
3. Save

The image upload field in the image library will the use a filepond widget.

## Form Element

Use the `filepond` form element directly in custom forms:

```php
$form['images'] = [
  '#type' => 'filepond',
  '#title' => t('Upload Images'),
  '#extensions' => 'jpg jpeg png gif',
  '#max_filesize' => '20M',
  '#max_files' => 5,
  '#upload_location' => 'public://uploads',
];
```
On submit, the element returns file IDs:

```php
public function submitForm(array &$form, FormStateInterface $form_state) {
  $values = $form_state->getValue('images');
  $fids = $values['fids'];  // Array of file entity IDs

  $files = \Drupal\file\Entity\File::loadMultiple($fids);
  foreach ($files as $file) {
    // Create media, attach to nodes, etc.
  }
}
```

## Events

FilePond dispatches two events during the upload lifecycle, allowing you to hook
into different stages of file processing.

### Event Overview

| Event | When | Use Cases |
|-------|------|-----------|
| `FilePondUploadPreMoveEvent` | Before file moves to destination | EXIF correction, virus scanning, image optimization, watermarking |
| `FilePondUploadCompleteEvent` | After file entity is saved | Create media entities, extract metadata, update related entities |

### FilePondUploadPreMoveEvent

Fires **before** the uploaded file is moved from the local temp directory to its
final destination. The file is still on the local filesystem, making
it efficient to read and modify.

**Use this event when you need to:**
- Fix EXIF orientation issues
- Scan files for viruses
- Optimize/compress images
- Add watermarks
- Strip metadata
- Validate file contents

**Why this event exists:** Drupal core's `hook_file_presave()` fires *after* the
file is moved, which is expensive for remote storage like S3 (requires
downloading the file back to modify it).

In fact, there is a common bottleneck with many file uploaders I've tried out
when using cloud storage. Saving an image field needs the width and height. To
get this, it pulls the file back down after its uploaded which can be a slow process.

This module solves this problem by leveraging this event and extracting the
image's width and height when it's still local and passes that to the image
field before it's saved, making the process much faster.

#### Benchmark Results

Enable the `filepond_benchmark` submodule to test this yourself. Example results
for a 5MB JPEG uploaded to S3:

| Method                                            | Time | Speedup |
|---------------------------------------------------|------|---------|
| **Normal getimagesize()** (downloads entire file) | 580 ms  | baseline      |
| **FastImage Class** (reads header bytes only)     | 320 ms  | ~1.8x faster  |
| **Local Temp File** (pre-captured during upload)  | 0.12 ms | ~5000x faster |

This approach is dramatically faster because the dimensions are captured
while the file is still on the local filesystem during upload, before it's moved
to S3. No network request is needed at all.

Run the benchmark at `/admin/config/media/filepond/benchmark` after enabling the
submodule.

#### Example Event Subscriber

```php
use Drupal\filepond\Event\FilePondUploadPreMoveEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FilePondPreMoveSubscriber implements EventSubscriberInterface {

  public static function getSubscribedEvents() {
    return [
      FilePondUploadPreMoveEvent::EVENT_NAME => 'onPreMove',
    ];
  }

  public function onPreMove(FilePondUploadPreMoveEvent $event) {
    // Only process images going to remote storage.
    if (!$event->isImage() || !$event->isRemoteDestination()) {
      return;
    }

    // The local temp file.
    $path = $event->getFilePath();

    // Do whatever you want to it here.
  }

}
```

#### PreMove Event Methods

| Method | Returns | Description |
|--------|---------|-------------|
| `getFilePath()` | `string` | Local filesystem path (read/write) |
| `getMimeType()` | `string` | File MIME type (e.g., `image/jpeg`) |
| `getOriginalFilename()` | `string` | Original filename from client |
| `getDestinationUri()` | `string` | Where file will be moved (e.g., `s3://...`) |
| `getContext()` | `array` | Context identifying upload source |
| `getContextValue($key)` | `mixed` | Get specific context value |
| `isImage()` | `bool` | TRUE if MIME type starts with `image/` |
| `isRemoteDestination()` | `bool` | TRUE if destination is not local (e.g., S3) |

### FilePondUploadCompleteEvent

Fires **after** the file entity is saved. Use this for post-processing that
requires the file entity, such as creating media entities or updating related
content.

**Note:** At this point, the file entity is created and the file is at its final destination, but the file entity is still **temporary** (`status = 0`). It
typically becomes permanent when the parent form is submitted and file usage is
recorded. The Media Library, File widget, and Entity Browser widget do this
automatically. If you are using the form element on a custom form you need
to set the files as permanent in your submit handler.

```yaml
# my_module.services.yml
services:
  my_module.filepond_subscriber:
    class: Drupal\my_module\EventSubscriber\FilePondSubscriber
    tags:
      - { name: event_subscriber }
```

```php
// src/EventSubscriber/FilePondSubscriber.php
namespace Drupal\my_module\EventSubscriber;

use Drupal\filepond\Event\FilePondUploadCompleteEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FilePondSubscriber implements EventSubscriberInterface {

  public static function getSubscribedEvents() {
    return [
      FilePondUploadCompleteEvent::EVENT_NAME => 'onUploadComplete',
    ];
  }

  public function onUploadComplete(FilePondUploadCompleteEvent $event) {
    $file = $event->getFile();
    $context = $event->getContext();

    // Do something with the uploaded file.
  }

}
```

#### Complete Event Methods

| Method | Returns | Description |
|--------|---------|-------------|
| `getFile()` | `FileInterface` | The saved file entity |
| `getOriginalFilename()` | `string` | Original filename from client |
| `getContext()` | `array` | Context identifying upload source |
| `getContextValue($key)` | `mixed` | Get specific context value |
| `setResult($key, $value)` | `$this` | Pass data back to controller |

### Context by Upload Source

Both events receive context identifying where the upload originated:

**Field Widget:**
```php
['entity_type' => 'node', 'bundle' => 'article', 'field_name' => 'field_image']
```

**Entity Browser:**
```php
['entity_browser' => 'media_browser', 'widget_uuid' => '...', 'entity_type' => 'media', 'bundle' => 'image']
```

**Media Library / Custom Forms:**
```php
['element_name' => 'upload', 'element_parents' => [...], 'field_name' => '...']
```

### Identifying Upload Source

```php
public function onUploadComplete(FilePondUploadCompleteEvent $event) {
  $context = $event->getContext();

  if (!empty($context['entity_browser'])) {
    // From Entity Browser - it handles media creation.
    return;
  }

  if (!empty($context['field_name']) && $context['field_name'] === 'field_portfolio') {
    // From specific field widget.
    $this->processPortfolioImage($event->getFile());
  }
}
```

### Subscribing to Both Events

```php
use Drupal\filepond\Event\FilePondUploadCompleteEvent;
use Drupal\filepond\Event\FilePondUploadPreMoveEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FilePondSubscriber implements EventSubscriberInterface {

  public static function getSubscribedEvents() {
    return [
      FilePondUploadPreMoveEvent::EVENT_NAME => 'onPreMove',
      FilePondUploadCompleteEvent::EVENT_NAME => 'onUploadComplete',
    ];
  }

  public function onPreMove(FilePondUploadPreMoveEvent $event) {
    // Modify file before it's moved.
  }

  public function onUploadComplete(FilePondUploadCompleteEvent $event) {
    // Create media entity, extract metadata, etc.
  }

}
```
## Permissions

Users need the **FilePond: Upload files** permission to use the upload element.

## Altering Elements

### Per-Form

```php
function my_module_form_alter(&$form, $form_state, $form_id) {
  if (isset($form['images']['#type']) && $form['images']['#type'] === 'filepond') {
    $form['images']['#max_files'] = 10;
    $form['images']['#config']['allowReorder'] = TRUE;
  }
}
```

### Site-Wide

```php
function my_module_element_info_alter(array &$info) {
  if (isset($info['filepond'])) {
    $info['filepond']['#process'][] = '_my_module_process_filepond';
  }
}

function _my_module_process_filepond($element, $form_state, $form) {
  $element['#config']['labelIdle'] = 'Custom site-wide prompt';
  return $element;
}
```

**Note:** Use a `#process` callback for `#config` alterations. Setting `#config`
directly in `hook_element_info_alter()` may be overwritten by forms.

## Submit Button Control

FilePond automatically disables form submit buttons while files are uploading
or processing. This prevents form submission before file IDs are available.

### Custom Submit Validators

Modules can register additional validators to control when submit is disabled:

```javascript
// In your module's JS file (must load after filepond.base.js):
(function (Drupal) {
  // Register a validator - return TRUE to disable submit.
  Drupal.filepond.addSubmitValidator(function (pond, form) {
    // Example: disable if less than 3 files
    var files = pond.getFiles();
    return files.length < 3;
  });
})(Drupal);
```

Validators receive:
- `pond` - The FilePond instance
- `form` - The containing form element

If **any** validator returns `true`, submit buttons are disabled.

### Built-in Validators

| Validator | Source | Condition |
|-----------|--------|-----------|
| Processing | `filepond.base.js` | Any file is uploading/processing |
| No files (EB) | `filepond_eb_widget.js` | No files selected (Entity Browser only) |

### Library Dependencies

Your library must depend on `filepond/filepond.base` to ensure the validator
registry is available:

```yaml
# my_module.libraries.yml
my_module.filepond:
  js:
    js/my_module.filepond.js: { }
  dependencies:
    - filepond/filepond.base
```

## Extending

### Custom Upload Settings

The `UploadSettingsResolver` service consolidates upload configuration from
different sources (fields, media types, Entity Browser widgets, Views areas).
You can use this service to resolve settings programmatically:

```php
$resolver = \Drupal::service('filepond.settings_resolver');

// From a field definition.
$options = $resolver->resolveFromField('node', 'article', 'field_image');

// From a media type.
$options = $resolver->resolveFromMediaType('image');

// $options is an UploadOptions value object with:
// - allowedExtensions: ['jpg', 'png', 'gif']
// - allowedMimeTypes: ['image/jpeg', 'image/png', 'image/gif']
// - destination: 'public://images'
// - maxSize: 10485760 (bytes)
// - context: ['entity_type' => 'node', ...]
```

### Adding Context Data

Use the `FilePondUploadPreMoveEvent` to add custom context that will be
available in `FilePondUploadCompleteEvent`:

```php
public function onPreMove(FilePondUploadPreMoveEvent $event) {
  // Add custom data to context.
  $event->addContext('my_custom_key', 'my_value');
}

public function onUploadComplete(FilePondUploadCompleteEvent $event) {
  // Retrieve the custom data.
  $value = $event->getContextValue('my_custom_key');
}
```

### Creating Custom Upload Routes

If you need custom upload endpoints (e.g., for a specialized widget), extend
`UploadController` and use the resolver:

```php
use Drupal\filepond\Controller\UploadController;

class MyCustomUploadController extends UploadController {

  public function customProcess(Request $request, string $my_param): Response {
    // Build options using the resolver or custom logic.
    $options = $this->settingsResolver->resolveFromMediaType($my_param);
    return $this->handleProcess($request, $options->toArray());
  }

}
```

The base controller provides `handleProcess()`, `handlePatch()`, and
`handleRevert()` methods that handle chunked uploads, file validation, and
event dispatching.

## Submodules

| Module | Description |
|--------|-------------|
| `filepond_crop` | Image cropping widget using Cropper.js and Drupal Crop API |
| `filepond_eb_widget` | Entity Browser widget for media uploads |
| `filepond_views` | Views area plugin that creates media on upload |
| `filepond_benchmark` | Performance testing tools (dev only) |

Enable submodules as needed:

```bash
drush en filepond_crop filepond_eb_widget filepond_views
```

### FilePond Crop Widget

The `filepond_crop` submodule provides a field widget that combines FilePond
uploads with Cropper.js for image cropping. Crops are stored using Drupal's
Crop API (non-destructive).

**Important:** This widget only works with **single-cardinality image fields**
(fields that allow exactly 1 image). For multi-value image fields, use the
standard FilePond Image widget instead.

**Requirements:**
- Crop module (`drupal/crop`)
- At least one crop type configured
- Single-cardinality image field

**Features:**
- Cropper.js v2 integration (CDN or local)
- Non-destructive cropping via Drupal Crop API
- Circular crop mode for avatars
- Minimum width enforcement
- Preview mode with Apply/Edit workflow
- Direct mode for simpler UX

See the [FilePond Crop README](modules/filepond_crop/README.md) for detailed
configuration.

## Limitations

FilePond requires JavaScript. When JS is disabled, users see a warning message.
This is a known limitation of FilePond's async upload model.

## Resources

- [FilePond Documentation](https://pqina.nl/filepond/docs/)
- [FilePond Options Reference](https://pqina.nl/filepond/docs/api/instance/properties/)
- [Image Preview Plugin](https://pqina.nl/filepond/docs/api/plugins/image-preview/)
- [Image Crop Plugin](https://pqina.nl/filepond/docs/api/plugins/image-crop/)
