/**
 * @file
 * FilePond integration for Media Library.
 *
 * Automatically submits the media library form when all files finish uploading.
 * Works with the auto_select_handler hidden element added by FilePondMediaUploadForm.
 */

/* global FilePond */

(function (Drupal, drupalSettings, once) {
  /**
   * Attaches the media library auto-submit handler to a FilePond instance.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {HTMLElement} autoSelectHandler
   *   The auto_select_handler hidden input element.
   */
  function attachMediaLibraryHandler(pond, autoSelectHandler) {
    // Track whether we've already triggered submission for this batch.
    let submissionTriggered = false;

    /**
     * Checks if all files are processed and triggers form submission.
     */
    function checkAndSubmit() {
      // Prevent double submission.
      if (submissionTriggered) {
        return;
      }

      const files = pond.getFiles();

      // No files means nothing to submit.
      if (files.length === 0) {
        return;
      }

      // Check if any files are NOT complete yet.
      // This includes: IDLE (waiting), PROCESSING_QUEUED, PROCESSING, LOADING.
      // FilePond FileStatus values:
      // INIT=1, IDLE=2, PROCESSING_QUEUED=9, PROCESSING=3,
      // PROCESSING_COMPLETE=5, PROCESSING_ERROR=6, LOADING=7, LOAD_ERROR=8
      const pendingFiles = files.filter(function (file) {
        return (
          file.status === FilePond.FileStatus.IDLE ||
          file.status === FilePond.FileStatus.INIT ||
          file.status === FilePond.FileStatus.PROCESSING ||
          file.status === FilePond.FileStatus.PROCESSING_QUEUED ||
          file.status === FilePond.FileStatus.LOADING
        );
      });

      if (pendingFiles.length > 0) {
        // Still have files that aren't done - wait.
        return;
      }

      // Check if we have successfully processed files.
      const completedFiles = files.filter(function (file) {
        return file.status === FilePond.FileStatus.PROCESSING_COMPLETE;
      });

      if (completedFiles.length === 0) {
        return;
      }

      // All files processed - trigger the auto-select event.
      submissionTriggered = true;

      // Delay to ensure hidden field is updated with all file IDs.
      setTimeout(function () {
        // Create and dispatch the custom event.
        const event = new CustomEvent('auto_select_media_library_widget', {
          bubbles: true,
        });
        autoSelectHandler.dispatchEvent(event);

        // Reset flag after a delay so user can add more files if needed.
        setTimeout(function () {
          submissionTriggered = false;
        }, 2000);
      }, 200);
    }

    // Listen for file processing completion.
    pond.on('processfile', function () {
      checkAndSubmit();
    });

    // Also check when processing errors occur (user might have partial success).
    pond.on('processfileabort', function () {
      checkAndSubmit();
    });
  }

  /**
   * Behavior for FilePond in Media Library context.
   *
   * Watches FilePond instances and triggers the auto-select handler
   * when all files have been processed successfully.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondMediaLibrary = {
    attach(context) {
      // Find auto_select_handler - only present in media library forms.
      const autoSelectHandler = context.querySelector
        ? context.querySelector('#auto_select_handler')
        : document.getElementById('auto_select_handler');

      if (!autoSelectHandler) {
        return;
      }

      // Find FilePond inputs in the form.
      const inputs = once(
        'filepond-media-library',
        '.filepond--input',
        context,
      );

      inputs.forEach(function (input) {
        // Wait for FilePond to be initialized (done by filepond.element.js).
        const checkInstance = setInterval(function () {
          if (input._filepond) {
            clearInterval(checkInstance);
            attachMediaLibraryHandler(input._filepond, autoSelectHandler);
          }
        }, 100);

        // Stop checking after 10 seconds to prevent infinite loops.
        setTimeout(function () {
          clearInterval(checkInstance);
        }, 10000);
      });
    },
  };
})(Drupal, drupalSettings, once);
