<?php

declare(strict_types=1);

namespace Drupal\filepond_benchmark\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\ByteSizeMarkup;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\filepond\FastImage;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Benchmark form for comparing image dimension retrieval methods.
 *
 * Compares performance of:
 * 1. Tempstore (pre-captured during upload while file was local)
 * 2. FastImage (reads only header bytes from S3)
 * 3. Normal getimagesize() (downloads entire file from S3)
 *
 * Available at /admin/config/media/filepond/benchmark.
 */
class DimensionBenchmarkForm extends FormBase {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The shared tempstore factory.
   */
  protected SharedTempStoreFactory $tempStoreFactory;

  /**
   * The file URL generator.
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->tempStoreFactory = $container->get('tempstore.shared');
    $instance->fileUrlGenerator = $container->get('file_url_generator');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'filepond_dimension_benchmark_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['description'] = [
      '#markup' => '<p>' . $this->t('Upload an image to S3 and compare dimension retrieval methods. The file will be uploaded to <code>s3://filepond-benchmark/</code>.') . '</p>',
    ];

    $form['image'] = [
      '#type' => 'filepond',
      '#title' => $this->t('Upload Image'),
      '#description' => $this->t('Upload a single image to benchmark dimension retrieval.'),
      '#extensions' => 'jpg jpeg png gif webp',
      '#max_files' => 1,
      '#upload_location' => 's3://filepond-benchmark',
      '#required' => TRUE,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run Benchmark'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue('image');
    $fids = $values['fids'] ?? [];

    if (empty($fids)) {
      $this->messenger()->addWarning($this->t('No file was uploaded.'));
      return;
    }

    $fid = reset($fids);
    /** @var \Drupal\file\FileInterface $file */
    $file = $this->entityTypeManager->getStorage('file')->load($fid);

    if (!$file) {
      $this->messenger()->addError($this->t('Could not load file entity.'));
      return;
    }

    $uri = $file->getFileUri();
    $this->messenger()->addStatus($this->t('File: @name (@size)', [
      '@name' => $file->getFilename(),
      '@size' => ByteSizeMarkup::create($file->getSize()),
    ]));
    $this->messenger()->addStatus($this->t('URI: <code>@uri</code>', [
      '@uri' => $uri,
    ]));

    $results = [];

    // Method 1: Tempstore (pre-captured during upload).
    // This is always fast since it's just a database/cache lookup.
    $tempstore = $this->tempStoreFactory->get('filepond.dimensions');
    $start = microtime(TRUE);
    $tempstoreDimensions = $tempstore->get((string) $fid);
    $tempstoreTime = (microtime(TRUE) - $start) * 1000;

    if ($tempstoreDimensions) {
      $results['tempstore'] = [
        'method' => 'Tempstore (pre-captured)',
        'dimensions' => $tempstoreDimensions,
        'time_ms' => $tempstoreTime,
      ];
    }
    else {
      $results['tempstore'] = [
        'method' => 'Tempstore (pre-captured)',
        'dimensions' => NULL,
        'time_ms' => $tempstoreTime,
        'note' => 'Not found - may have been consumed already',
      ];
    }

    // Clear any PHP stat cache before S3 tests.
    clearstatcache(TRUE);

    // Method 2: Normal getimagesize() FIRST (downloads entire file).
    // Run this first so it bears the connection warmup cost.
    $start = microtime(TRUE);
    $normalDimensions = NULL;
    $imageInfo = @getimagesize($uri);
    if ($imageInfo !== FALSE) {
      $normalDimensions = [
        'width' => $imageInfo[0],
        'height' => $imageInfo[1],
      ];
    }
    $normalTime = (microtime(TRUE) - $start) * 1000;

    $results['normal'] = [
      'method' => 'Normal getimagesize() (full download)',
      'dimensions' => $normalDimensions,
      'time_ms' => $normalTime,
    ];

    // Clear caches again before FastImage test.
    clearstatcache(TRUE);

    // Method 3: FastImage SECOND (reads only header bytes).
    // Use the public URL (https://) instead of s3:// stream wrapper
    // to avoid stream wrapper overhead and allow efficient HTTP reads.
    $publicUrl = $this->fileUrlGenerator->generateAbsoluteString($uri);
    $this->messenger()->addStatus($this->t('Public URL: <code>@url</code>', [
      '@url' => $publicUrl,
    ]));

    $start = microtime(TRUE);
    $fastImage = new FastImage();
    $fastImageLoaded = $fastImage->load($publicUrl);
    $fastImageDimensions = NULL;
    if ($fastImageLoaded) {
      $size = $fastImage->getSize();
      if ($size && count($size) >= 2) {
        $fastImageDimensions = [
          'width' => (int) $size[0],
          'height' => (int) $size[1],
        ];
      }
      $fastImage->close();
    }
    $fastImageTime = (microtime(TRUE) - $start) * 1000;

    $results['fastimage'] = [
      'method' => 'FastImage (header bytes only)',
      'dimensions' => $fastImageDimensions,
      'time_ms' => $fastImageTime,
    ];

    // Display results.
    $this->messenger()->addStatus($this->t('<strong>Benchmark Results:</strong>'));

    foreach ($results as $result) {
      $dims = $result['dimensions'];
      $dimsStr = $dims ? $this->t('@w x @h', [
        '@w' => $dims['width'],
        '@h' => $dims['height'],
      ]) : $this->t('N/A');

      $note = isset($result['note']) ? ' (' . $result['note'] . ')' : '';

      $this->messenger()->addStatus($this->t('@method: @dims in <strong>@time ms</strong>@note', [
        '@method' => $result['method'],
        '@dims' => $dimsStr,
        '@time' => number_format($result['time_ms'], 2),
        '@note' => $note,
      ]));
    }

    // Summary comparison.
    if ($results['normal']['time_ms'] > 0 && $results['fastimage']['time_ms'] > 0) {
      $speedup = $results['normal']['time_ms'] / $results['fastimage']['time_ms'];
      $this->messenger()->addStatus($this->t('<strong>FastImage is @x times faster than normal getimagesize()</strong>', [
        '@x' => number_format($speedup, 1),
      ]));
    }

    if ($tempstoreDimensions && $results['normal']['time_ms'] > 0) {
      $speedup = $results['normal']['time_ms'] / $results['tempstore']['time_ms'];
      $this->messenger()->addStatus($this->t('<strong>Tempstore is @x times faster than normal getimagesize()</strong>', [
        '@x' => number_format($speedup, 1),
      ]));
    }

    // Clean up tempstore now that we're done.
    $tempstore->delete((string) $fid);
  }

}
