<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond_crop\Kernel;

use Drupal\crop\Entity\Crop;
use Drupal\crop\Entity\CropType;
use Drupal\file\Entity\File;
use Drupal\filepond_crop\CropManager;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the CropManager service.
 *
 * @group filepond
 * @group filepond_crop
 * @coversDefaultClass \Drupal\filepond_crop\CropManager
 */
class CropManagerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'image',
    'user',
    'crop',
    'filepond',
    'filepond_crop',
  ];

  /**
   * The CropManager service.
   *
   * @var \Drupal\filepond_crop\CropManager
   */
  protected CropManager $cropManager;

  /**
   * A test file entity.
   *
   * @var \Drupal\file\FileInterface
   */
  protected $testFile;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('crop');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['system', 'crop', 'filepond', 'filepond_crop']);

    // Create a crop type for testing.
    CropType::create([
      'id' => 'test_square',
      'label' => 'Test Square Crop',
      'aspect_ratio' => '1:1',
    ])->save();

    CropType::create([
      'id' => 'test_landscape',
      'label' => 'Test Landscape Crop',
      'aspect_ratio' => '16:9',
    ])->save();

    CropType::create([
      'id' => 'test_freeform',
      'label' => 'Test Freeform Crop',
      'aspect_ratio' => '',
    ])->save();

    // Create a crop type with hard and soft limits.
    CropType::create([
      'id' => 'test_with_limits',
      'label' => 'Test With Limits',
      'aspect_ratio' => '1:1',
      'hard_limit_width' => 200,
      'hard_limit_height' => 200,
      'soft_limit_width' => 400,
      'soft_limit_height' => 400,
    ])->save();

    // Create a test file.
    $this->testFile = File::create([
      'uri' => 'public://test-image.jpg',
      'filename' => 'test-image.jpg',
      'filemime' => 'image/jpeg',
      'status' => 1,
    ]);
    $this->testFile->save();

    $this->cropManager = $this->container->get('filepond_crop.crop_manager');
  }

  /**
   * Tests convertToCenter() coordinate conversion.
   *
   * @covers ::convertToCenter
   */
  public function testConvertToCenter(): void {
    // Test basic conversion: top-left (100, 50) with size 200x100.
    // Center should be at (100 + 100, 50 + 50) = (200, 100).
    $result = $this->cropManager->convertToCenter([
      'x' => 100,
      'y' => 50,
      'width' => 200,
      'height' => 100,
    ]);

    $this->assertEquals(200, $result['x'], 'Center X should be top-left X + width/2');
    $this->assertEquals(100, $result['y'], 'Center Y should be top-left Y + height/2');
    $this->assertEquals(200, $result['width'], 'Width should remain unchanged');
    $this->assertEquals(100, $result['height'], 'Height should remain unchanged');
  }

  /**
   * Tests convertToCenter() with zero origin.
   *
   * @covers ::convertToCenter
   */
  public function testConvertToCenterZeroOrigin(): void {
    // Top-left at origin (0, 0) with size 100x100.
    // Center should be at (50, 50).
    $result = $this->cropManager->convertToCenter([
      'x' => 0,
      'y' => 0,
      'width' => 100,
      'height' => 100,
    ]);

    $this->assertEquals(50, $result['x']);
    $this->assertEquals(50, $result['y']);
  }

  /**
   * Tests convertToCenter() with decimal values.
   *
   * @covers ::convertToCenter
   */
  public function testConvertToCenterDecimals(): void {
    // Cropper.js can return decimal coordinates.
    $result = $this->cropManager->convertToCenter([
      'x' => 100.7,
      'y' => 50.3,
      'width' => 200.5,
      'height' => 100.9,
    ]);

    // Should round to nearest integer.
    $this->assertEquals(201, $result['x'], 'Should round 100.7 + 100.25 = 200.95 → 201');
    $this->assertEquals(101, $result['y'], 'Should round 50.3 + 50.45 = 100.75 → 101');
    $this->assertEquals(201, $result['width'], 'Should round 200.5 → 201');
    $this->assertEquals(101, $result['height'], 'Should round 100.9 → 101');
  }

  /**
   * Tests convertToCenter() handles missing keys.
   *
   * @covers ::convertToCenter
   */
  public function testConvertToCenterMissingKeys(): void {
    // Should default missing keys to 0.
    $result = $this->cropManager->convertToCenter([]);

    $this->assertEquals(0, $result['x']);
    $this->assertEquals(0, $result['y']);
    $this->assertEquals(0, $result['width']);
    $this->assertEquals(0, $result['height']);
  }

  /**
   * Tests parseAspectRatio() with valid ratios.
   *
   * @covers ::parseAspectRatio
   * @dataProvider aspectRatioProvider
   */
  public function testParseAspectRatio(string $input, ?float $expected): void {
    $result = $this->cropManager->parseAspectRatio($input);

    if ($expected === NULL) {
      $this->assertNull($result, "Ratio '$input' should return NULL");
    }
    else {
      $this->assertEqualsWithDelta($expected, $result, 0.0001, "Ratio '$input' should parse correctly");
    }
  }

  /**
   * Data provider for aspect ratio tests.
   */
  public static function aspectRatioProvider(): array {
    return [
      'square' => ['1:1', 1.0],
      'landscape 16:9' => ['16:9', 1.7778],
      'landscape 4:3' => ['4:3', 1.3333],
      'portrait 9:16' => ['9:16', 0.5625],
      'wide banner' => ['500:180', 2.7778],
      'empty string' => ['', NULL],
      'no colon' => ['16x9', NULL],
      'single number' => ['16', NULL],
      'multiple colons' => ['16:9:4', NULL],
      'zero height' => ['16:0', NULL],
      'negative values' => ['-16:9', -1.7778],
    ];
  }

  /**
   * Tests getAspectRatio() loading from crop type.
   *
   * @covers ::getAspectRatio
   */
  public function testGetAspectRatio(): void {
    // Square crop type.
    $ratio = $this->cropManager->getAspectRatio('test_square');
    $this->assertEqualsWithDelta(1.0, $ratio, 0.0001, 'Square crop should have 1:1 ratio');

    // Landscape crop type.
    $ratio = $this->cropManager->getAspectRatio('test_landscape');
    $this->assertEqualsWithDelta(1.7778, $ratio, 0.0001, 'Landscape crop should have 16:9 ratio');

    // Freeform crop type (no ratio).
    $ratio = $this->cropManager->getAspectRatio('test_freeform');
    $this->assertNull($ratio, 'Freeform crop should have NULL ratio');

    // Non-existent crop type.
    $ratio = $this->cropManager->getAspectRatio('nonexistent');
    $this->assertNull($ratio, 'Non-existent crop type should return NULL');
  }

  /**
   * Tests saveCrop() creates a new crop entity.
   *
   * @covers ::saveCrop
   */
  public function testSaveCropNew(): void {
    $coordinates = [
      'x' => 100,
      'y' => 50,
      'width' => 200,
      'height' => 200,
    ];

    $crop = $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      $coordinates
    );

    $this->assertInstanceOf(Crop::class, $crop);
    $this->assertNotNull($crop->id(), 'Crop should be saved with an ID');
    $this->assertEquals('test_square', $crop->bundle());
    $this->assertEquals($this->testFile->getFileUri(), $crop->get('uri')->value);

    // Verify center-based coordinates were saved.
    // Top-left (100, 50) + size (200, 200) → center (200, 150).
    $this->assertEquals(200, $crop->get('x')->value);
    $this->assertEquals(150, $crop->get('y')->value);
    $this->assertEquals(200, $crop->get('width')->value);
    $this->assertEquals(200, $crop->get('height')->value);
  }

  /**
   * Tests saveCrop() updates an existing crop.
   *
   * @covers ::saveCrop
   */
  public function testSaveCropUpdate(): void {
    // Create initial crop.
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );

    // Update with new coordinates.
    $crop = $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 50, 'y' => 50, 'width' => 200, 'height' => 200]
    );

    // Should still have just one crop entity.
    $crops = \Drupal::entityTypeManager()
      ->getStorage('crop')
      ->loadByProperties([
        'uri' => $this->testFile->getFileUri(),
        'type' => 'test_square',
      ]);
    $this->assertCount(1, $crops, 'Should have exactly one crop after update');

    // Verify updated coordinates.
    // Top-left (50, 50) + size (200, 200) → center (150, 150).
    $this->assertEquals(150, $crop->get('x')->value);
    $this->assertEquals(150, $crop->get('y')->value);
  }

  /**
   * Tests saveCrop() with invalid file ID throws exception.
   *
   * @covers ::saveCrop
   */
  public function testSaveCropInvalidFile(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('File not found: 99999');

    $this->cropManager->saveCrop(
      99999,
      'test_square',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );
  }

  /**
   * Tests saveCrop() with invalid crop type throws exception.
   *
   * @covers ::saveCrop
   */
  public function testSaveCropInvalidCropType(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Crop type not found: nonexistent');

    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'nonexistent',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );
  }

  /**
   * Tests findCrop() returns existing crop.
   *
   * @covers ::findCrop
   */
  public function testFindCrop(): void {
    // Initially no crop exists.
    $crop = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $this->assertNull($crop, 'Should return NULL when no crop exists');

    // Create a crop.
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 100, 'y' => 100, 'width' => 200, 'height' => 200]
    );

    // Now should find it.
    $crop = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $this->assertNotNull($crop, 'Should find existing crop');
    $this->assertEquals('test_square', $crop->bundle());
  }

  /**
   * Tests findCrop() with different crop types.
   *
   * @covers ::findCrop
   */
  public function testFindCropDifferentTypes(): void {
    // Create crops for two different types.
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_landscape',
      ['x' => 0, 'y' => 0, 'width' => 160, 'height' => 90]
    );

    // Find each type separately.
    $square = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $landscape = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_landscape'
    );

    $this->assertNotNull($square);
    $this->assertNotNull($landscape);
    $this->assertEquals(100, $square->get('width')->value);
    $this->assertEquals(160, $landscape->get('width')->value);
  }

  /**
   * Tests deleteCrop() removes crop entity.
   *
   * @covers ::deleteCrop
   */
  public function testDeleteCrop(): void {
    // Create a crop.
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );

    // Verify it exists.
    $crop = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $this->assertNotNull($crop, 'Crop should exist before deletion');

    // Delete it.
    $this->cropManager->deleteCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );

    // Verify it's gone.
    $crop = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $this->assertNull($crop, 'Crop should be deleted');
  }

  /**
   * Tests deleteCrop() only deletes the specified type.
   *
   * @covers ::deleteCrop
   */
  public function testDeleteCropSpecificType(): void {
    // Create crops for two types.
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 0, 'y' => 0, 'width' => 100, 'height' => 100]
    );
    $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_landscape',
      ['x' => 0, 'y' => 0, 'width' => 160, 'height' => 90]
    );

    // Delete only the square crop.
    $this->cropManager->deleteCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );

    // Square should be gone, landscape should remain.
    $square = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );
    $landscape = $this->cropManager->findCrop(
      $this->testFile->getFileUri(),
      'test_landscape'
    );

    $this->assertNull($square, 'Square crop should be deleted');
    $this->assertNotNull($landscape, 'Landscape crop should remain');
  }

  /**
   * Tests deleteCrop() with non-existent crop does nothing.
   *
   * @covers ::deleteCrop
   */
  public function testDeleteCropNonExistent(): void {
    // Should not throw an exception.
    $this->cropManager->deleteCrop(
      $this->testFile->getFileUri(),
      'test_square'
    );

    // Just verify no crash occurred.
    $this->assertTrue(TRUE);
  }

  /**
   * Tests getCropTypeOptions() returns available crop types.
   *
   * @covers ::getCropTypeOptions
   */
  public function testGetCropTypeOptions(): void {
    $options = $this->cropManager->getCropTypeOptions();

    $this->assertIsArray($options);
    $this->assertArrayHasKey('test_square', $options);
    $this->assertArrayHasKey('test_landscape', $options);
    $this->assertArrayHasKey('test_freeform', $options);
    $this->assertEquals('Test Square Crop', $options['test_square']);
  }

  /**
   * Tests convertToTopLeft() coordinate conversion.
   *
   * @covers ::convertToTopLeft
   */
  public function testConvertToTopLeft(): void {
    // Create a crop with known center coordinates.
    $crop = $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      ['x' => 100, 'y' => 50, 'width' => 200, 'height' => 100]
    );

    // Convert back to top-left.
    $result = $this->cropManager->convertToTopLeft($crop);

    // Should get back original values.
    $this->assertEquals(100, $result['x'], 'Top-left X should match original');
    $this->assertEquals(50, $result['y'], 'Top-left Y should match original');
    $this->assertEquals(200, $result['width'], 'Width should match original');
    $this->assertEquals(100, $result['height'], 'Height should match original');
  }

  /**
   * Tests round-trip conversion preserves coordinates.
   *
   * @covers ::convertToCenter
   * @covers ::convertToTopLeft
   */
  public function testRoundTripConversion(): void {
    $original = [
      'x' => 150,
      'y' => 75,
      'width' => 300,
      'height' => 200,
    ];

    // Save creates center-based crop.
    $crop = $this->cropManager->saveCrop(
      $this->testFile->id(),
      'test_square',
      $original
    );

    // Convert back to top-left.
    $result = $this->cropManager->convertToTopLeft($crop);

    $this->assertEquals($original['x'], $result['x']);
    $this->assertEquals($original['y'], $result['y']);
    $this->assertEquals($original['width'], $result['width']);
    $this->assertEquals($original['height'], $result['height']);
  }

  /**
   * Tests getHardLimit() returns configured limits.
   *
   * @covers ::getHardLimit
   */
  public function testGetHardLimit(): void {
    $result = $this->cropManager->getHardLimit('test_with_limits');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('width', $result);
    $this->assertArrayHasKey('height', $result);
    $this->assertEquals(200, $result['width']);
    $this->assertEquals(200, $result['height']);
  }

  /**
   * Tests getHardLimit() returns zeros for crop type without limits.
   *
   * @covers ::getHardLimit
   */
  public function testGetHardLimitNoLimits(): void {
    $result = $this->cropManager->getHardLimit('test_square');

    $this->assertIsArray($result);
    $this->assertEquals(0, $result['width']);
    $this->assertEquals(0, $result['height']);
  }

  /**
   * Tests getHardLimit() returns defaults for non-existent crop type.
   *
   * @covers ::getHardLimit
   */
  public function testGetHardLimitNonExistent(): void {
    $result = $this->cropManager->getHardLimit('nonexistent_type');

    $this->assertIsArray($result);
    $this->assertEquals(0, $result['width']);
    $this->assertEquals(0, $result['height']);
  }

  /**
   * Tests getSoftLimit() returns configured limits.
   *
   * @covers ::getSoftLimit
   */
  public function testGetSoftLimit(): void {
    $result = $this->cropManager->getSoftLimit('test_with_limits');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('width', $result);
    $this->assertArrayHasKey('height', $result);
    $this->assertEquals(400, $result['width']);
    $this->assertEquals(400, $result['height']);
  }

  /**
   * Tests getSoftLimit() returns zeros for crop type without limits.
   *
   * @covers ::getSoftLimit
   */
  public function testGetSoftLimitNoLimits(): void {
    $result = $this->cropManager->getSoftLimit('test_freeform');

    $this->assertIsArray($result);
    $this->assertEquals(0, $result['width']);
    $this->assertEquals(0, $result['height']);
  }

}
