<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond_crop\Kernel;

use Drupal\crop\Entity\CropType;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\filepond_crop\Plugin\Field\FieldWidget\FilePondCropWidget;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\user\RoleInterface;

/**
 * Tests the FilePond Crop widget.
 *
 * @group filepond
 * @group filepond_crop
 * @coversDefaultClass \Drupal\filepond_crop\Plugin\Field\FieldWidget\FilePondCropWidget
 */
class FilePondCropWidgetTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'field',
    'file',
    'image',
    'user',
    'node',
    'text',
    'filter',
    'crop',
    'filepond',
    'filepond_crop',
  ];

  /**
   * A test user with permission to upload files.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('file');
    $this->installEntitySchema('crop');
    $this->installSchema('file', ['file_usage']);
    $this->installSchema('node', ['node_access']);
    $this->installConfig([
      'system',
      'field',
      'node',
      'filepond',
      'filepond_crop',
      'filter',
      'crop',
    ]);

    // Create authenticated role with filepond permission.
    $role = Role::create([
      'id' => RoleInterface::AUTHENTICATED_ID,
      'label' => 'Authenticated',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();

    // Create a test user.
    $this->testUser = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $this->testUser->save();
    $this->container->get('current_user')->setAccount($this->testUser);

    // Create a crop type for testing.
    CropType::create([
      'id' => 'test_crop',
      'label' => 'Test Crop',
      'aspect_ratio' => '1:1',
    ])->save();

    // Create a content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();
  }

  /**
   * Creates an image field on the article content type.
   *
   * @param int $cardinality
   *   The field cardinality (1 for single, -1 for unlimited).
   * @param string $field_name
   *   The field machine name.
   */
  protected function createImageField(int $cardinality = 1, string $field_name = 'field_image'): void {
    FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'node',
      'type' => 'image',
      'cardinality' => $cardinality,
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => $field_name,
      'label' => 'Image',
      'settings' => [
        'file_extensions' => 'png gif jpg jpeg',
        'max_filesize' => '5M',
      ],
    ])->save();
  }

  /**
   * Configures the form display to use the FilePond Crop widget.
   *
   * @param string $field_name
   *   The field machine name.
   * @param array $settings
   *   Widget settings to apply.
   */
  protected function configureWidget(string $field_name = 'field_image', array $settings = []): void {
    $default_settings = [
      'crop_type' => 'test_crop',
      'show_default_crop' => TRUE,
      'show_crop_preview' => TRUE,
      'crop_preview_image_style' => '',
      'cropper_image_style' => '',
      'circular_crop' => FALSE,
      'show_reset_button' => FALSE,
    ];

    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
    $display_repository = $this->container->get('entity_display.repository');
    $display_repository->getFormDisplay('node', 'article')
      ->setComponent($field_name, [
        'type' => 'filepond_image_crop',
        'settings' => array_merge($default_settings, $settings),
      ])
      ->save();
  }

  /**
   * Tests isApplicable() returns TRUE for single-cardinality image fields.
   *
   * @covers ::isApplicable
   */
  public function testIsApplicableSingleCardinality(): void {
    $this->createImageField(1);

    $field_definition = FieldConfig::loadByName('node', 'article', 'field_image');
    $this->assertNotNull($field_definition);

    $result = FilePondCropWidget::isApplicable($field_definition);
    $this->assertTrue($result, 'Widget should be applicable for single-cardinality image field');
  }

  /**
   * Tests isApplicable() returns FALSE for multi-cardinality image fields.
   *
   * @covers ::isApplicable
   */
  public function testIsApplicableMultiCardinality(): void {
    $this->createImageField(-1);

    $field_definition = FieldConfig::loadByName('node', 'article', 'field_image');
    $this->assertNotNull($field_definition);

    $result = FilePondCropWidget::isApplicable($field_definition);
    $this->assertFalse($result, 'Widget should NOT be applicable for unlimited cardinality');
  }

  /**
   * Tests isApplicable() returns FALSE for fixed multi-cardinality fields.
   *
   * @covers ::isApplicable
   */
  public function testIsApplicableCardinalityFive(): void {
    $this->createImageField(5);

    $field_definition = FieldConfig::loadByName('node', 'article', 'field_image');
    $this->assertNotNull($field_definition);

    $result = FilePondCropWidget::isApplicable($field_definition);
    $this->assertFalse($result, 'Widget should NOT be applicable for cardinality > 1');
  }

  /**
   * Tests isApplicable() returns FALSE for non-image fields.
   *
   * @covers ::isApplicable
   */
  public function testIsApplicableNonImageField(): void {
    // Create a text field instead.
    FieldStorageConfig::create([
      'field_name' => 'field_text',
      'entity_type' => 'node',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'bundle' => 'article',
      'field_name' => 'field_text',
      'label' => 'Text',
    ])->save();

    $field_definition = FieldConfig::loadByName('node', 'article', 'field_text');
    $this->assertNotNull($field_definition);

    $result = FilePondCropWidget::isApplicable($field_definition);
    $this->assertFalse($result, 'Widget should NOT be applicable for non-image fields');
  }

  /**
   * Tests that the widget renders with required elements.
   *
   * @covers ::formElement
   */
  public function testWidgetRendersWithCropElements(): void {
    $this->createImageField(1);
    $this->configureWidget();
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Check for the fieldset wrapper with correct ID.
    $this->assertNotEmpty(
      $this->xpath("//fieldset[@id='filepond-crop-field_image']"),
      'Fieldset wrapper should have correct ID'
    );

    // Check for filepond-crop-widget class.
    $this->assertNotEmpty(
      $this->xpath("//fieldset[contains(@class, 'filepond-crop-widget')]"),
      'Fieldset should have filepond-crop-widget class'
    );

    // Check for crop container.
    $this->assertNotEmpty(
      $this->xpath("//*[@data-filepond-crop='container']"),
      'Crop container element should be present'
    );

    // Check for hidden coordinate fields.
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='x']"),
      'Hidden X coordinate field should be present'
    );
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='y']"),
      'Hidden Y coordinate field should be present'
    );
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='width']"),
      'Hidden width field should be present'
    );
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='height']"),
      'Hidden height field should be present'
    );
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='applied']"),
      'Hidden applied field should be present'
    );
    $this->assertNotEmpty(
      $this->xpath("//input[@type='hidden'][@data-filepond-crop='fid']"),
      'Hidden file ID field should be present'
    );
  }

  /**
   * Tests that action buttons render correctly.
   *
   * @covers ::formElement
   */
  public function testActionButtonsRender(): void {
    $this->createImageField(1);
    $this->configureWidget();
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Check for Apply button.
    $this->assertNotEmpty(
      $this->xpath("//button[@data-filepond-crop='apply']"),
      'Apply Crop button should be present'
    );

    // Check for Edit button.
    $this->assertNotEmpty(
      $this->xpath("//button[@data-filepond-crop='edit']"),
      'Edit Crop button should be present'
    );

    // Check for Remove button.
    $this->assertNotEmpty(
      $this->xpath("//button[@data-filepond-crop='remove']"),
      'Remove Image button should be present'
    );
  }

  /**
   * Tests that reset button only appears when enabled.
   *
   * @covers ::formElement
   */
  public function testResetButtonConditional(): void {
    $this->createImageField(1);

    // First, configure without reset button.
    $this->configureWidget('field_image', ['show_reset_button' => FALSE]);
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Reset button should NOT be present.
    $this->assertEmpty(
      $this->xpath("//button[@data-filepond-crop='reset']"),
      'Reset button should NOT be present when disabled'
    );

    // Now enable reset button.
    $this->configureWidget('field_image', ['show_reset_button' => TRUE]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Reset button should be present.
    $this->assertNotEmpty(
      $this->xpath("//button[@data-filepond-crop='reset']"),
      'Reset button should be present when enabled'
    );
  }

  /**
   * Tests circular crop adds the correct class.
   *
   * @covers ::formElement
   */
  public function testCircularCropClass(): void {
    $this->createImageField(1);

    // Configure with circular crop disabled.
    $this->configureWidget('field_image', ['circular_crop' => FALSE]);
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Circular class should NOT be present.
    $this->assertEmpty(
      $this->xpath("//fieldset[contains(@class, 'filepond-crop--circular')]"),
      'Circular class should NOT be present when disabled'
    );

    // Enable circular crop.
    $this->configureWidget('field_image', ['circular_crop' => TRUE]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Circular class should be present.
    $this->assertNotEmpty(
      $this->xpath("//fieldset[contains(@class, 'filepond-crop--circular')]"),
      'Circular class should be present when enabled'
    );
  }

  /**
   * Tests direct mode hides Apply/Edit buttons.
   *
   * @covers ::formElement
   */
  public function testDirectModeHidesPreviewButtons(): void {
    $this->createImageField(1);

    // Configure with preview mode disabled (direct mode).
    $this->configureWidget('field_image', ['show_crop_preview' => FALSE]);
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Apply and Edit buttons should NOT be present in direct mode.
    $this->assertEmpty(
      $this->xpath("//button[@data-filepond-crop='apply']"),
      'Apply button should NOT be present in direct mode'
    );
    $this->assertEmpty(
      $this->xpath("//button[@data-filepond-crop='edit']"),
      'Edit button should NOT be present in direct mode'
    );

    // Remove button should still be present.
    $this->assertNotEmpty(
      $this->xpath("//button[@data-filepond-crop='remove']"),
      'Remove button should still be present in direct mode'
    );
  }

  /**
   * Tests default settings values.
   *
   * @covers ::defaultSettings
   */
  public function testDefaultSettings(): void {
    $defaults = FilePondCropWidget::defaultSettings();

    $this->assertArrayHasKey('crop_type', $defaults);
    $this->assertArrayHasKey('show_default_crop', $defaults);
    $this->assertArrayHasKey('show_crop_preview', $defaults);
    $this->assertArrayHasKey('crop_preview_image_style', $defaults);
    $this->assertArrayHasKey('cropper_image_style', $defaults);
    $this->assertArrayHasKey('circular_crop', $defaults);
    $this->assertArrayHasKey('show_reset_button', $defaults);

    // Check default values.
    $this->assertEquals('', $defaults['crop_type']);
    $this->assertTrue($defaults['show_default_crop']);
    $this->assertTrue($defaults['show_crop_preview']);
    $this->assertFalse($defaults['circular_crop']);
    $this->assertFalse($defaults['show_reset_button']);
  }

  /**
   * Tests drupalSettings are attached with correct values.
   *
   * @covers ::formElement
   */
  public function testDrupalSettingsAttached(): void {
    $this->createImageField(1);
    $this->configureWidget('field_image', [
      'crop_type' => 'test_crop',
      'circular_crop' => TRUE,
      'show_default_crop' => TRUE,
      'show_crop_preview' => TRUE,
    ]);
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);

    // The widget element is nested under field_image > widget > 0.
    $widget_element = $form['field_image']['widget'][0] ?? NULL;
    $this->assertNotNull($widget_element, 'Widget element should exist');

    // Check that drupalSettings are attached to the widget element.
    $this->assertArrayHasKey('#attached', $widget_element);
    $this->assertArrayHasKey('drupalSettings', $widget_element['#attached']);
    $this->assertArrayHasKey('filepondCrop', $widget_element['#attached']['drupalSettings']);

    $element_id = 'filepond-crop-field_image';
    $this->assertArrayHasKey($element_id, $widget_element['#attached']['drupalSettings']['filepondCrop']);

    $settings = $widget_element['#attached']['drupalSettings']['filepondCrop'][$element_id];

    $this->assertEquals($element_id, $settings['elementId']);
    $this->assertEquals('test_crop', $settings['cropType']);
    $this->assertTrue($settings['circularCrop']);
    $this->assertTrue($settings['showDefaultCrop']);
    $this->assertTrue($settings['showCropPreview']);
    // Circular crop forces 1:1 aspect ratio.
    $this->assertEquals(1, $settings['aspectRatio']);
    // Hard/soft limits come from crop type config.
    $this->assertArrayHasKey('hardLimit', $settings);
    $this->assertArrayHasKey('softLimit', $settings);
    $this->assertIsArray($settings['hardLimit']);
    $this->assertIsArray($settings['softLimit']);
  }

  /**
   * Tests empty state class is applied.
   *
   * @covers ::formElement
   */
  public function testEmptyStateClass(): void {
    $this->createImageField(1);
    $this->configureWidget();
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);
    $this->render($form);

    // Empty state class should be present for new node.
    $this->assertNotEmpty(
      $this->xpath("//fieldset[contains(@class, 'filepond-crop--empty')]"),
      'Empty state class should be present for new node without image'
    );
  }

  /**
   * Tests library is attached.
   *
   * @covers ::formElement
   */
  public function testLibraryAttached(): void {
    $this->createImageField(1);
    $this->configureWidget();
    $this->container->get('router.builder')->rebuild();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);

    $form = $this->container->get('entity.form_builder')->getForm($node);

    // The widget element is nested under field_image > widget > 0.
    $widget_element = $form['field_image']['widget'][0] ?? NULL;
    $this->assertNotNull($widget_element, 'Widget element should exist');

    // Check library is attached to the widget element.
    $this->assertArrayHasKey('#attached', $widget_element);
    $this->assertContains(
      'filepond_crop/filepond_crop',
      $widget_element['#attached']['library'],
      'filepond_crop library should be attached'
    );
  }

}
