/**
 * @file
 * FilePond Views integration.
 *
 * Refreshes the view when all uploads are complete and optionally
 * auto-selects newly created media in Entity Browser.
 */

/* global FilePond */

(function (Drupal, drupalSettings, once) {
  /**
   * Extracts media ID from FilePond server response.
   *
   * For regular uploads, serverId is JSON with mediaId.
   * For chunked uploads, serverId is the transfer ID, and mediaId
   * is stored in the global Drupal.filepond.transferToMediaId map.
   *
   * @param {string} serverId
   *   The server ID returned by FilePond.
   *
   * @return {string|null}
   *   The media ID or null if not present.
   */
  function extractMediaId(serverId) {
    if (!serverId) {
      return null;
    }

    // Try parsing as JSON first (regular uploads).
    try {
      const data = JSON.parse(serverId);
      if (data.mediaId) {
        return String(data.mediaId);
      }
    } catch (e) {
      // Not JSON - likely a transfer ID from chunked upload.
      // Check the global mapping.
      if (Drupal.filepond?.transferToMediaId?.[serverId]) {
        return String(Drupal.filepond.transferToMediaId[serverId]);
      }
    }

    return null;
  }

  /**
   * Auto-selects media items in Entity Browser view.
   *
   * @param {Array} mediaIds
   *   Array of media IDs to select.
   * @param {string} viewDomId
   *   The view's DOM ID.
   */
  function autoSelectMedia(mediaIds, viewDomId) {
    if (!mediaIds.length) {
      return;
    }

    const viewContainer = document.querySelector(
      `.js-view-dom-id-${viewDomId}`,
    );
    if (!viewContainer) {
      return;
    }

    // Check if we're inside an Entity Browser form.
    const form = viewContainer.closest('form.entity-browser-form');

    // Set auto-select mode flag so eb_extras knows not to show alerts.
    // It will silently skip items that exceed cardinality.
    if (form) {
      form.dataset.autoSelectMode = 'true';
    }

    // Find and click each media item's checkbox/selection element.
    mediaIds.forEach((mediaId) => {
      // Entity Browser uses data-entity-id on selectable items.
      const selector = `[data-entity-id="media:${mediaId}"]`;
      const item = viewContainer.querySelector(selector);

      if (item) {
        // Trigger click on the item to select it.
        item.click();
      } else {
        // Try alternate selector - some views use input checkboxes.
        const checkbox = viewContainer.querySelector(
          `input[name*="entity_browser_select"][value="media:${mediaId}"]`,
        );
        if (checkbox && !checkbox.checked) {
          checkbox.click();
        }
      }
    });

    // Clear auto-select mode flag.
    if (form) {
      delete form.dataset.autoSelectMode;
    }
  }

  /**
   * Handles the processfiles event - refreshes view after uploads complete.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {string} viewDomId
   *   The view's DOM ID for refresh targeting.
   * @param {HTMLElement} element
   *   The uploader element for message context.
   * @param {Object} settings
   *   View settings from drupalSettings.
   */
  function handleProcessFiles(pond, viewDomId, element, settings) {
    const files = pond.getFiles();
    const processedFiles = files.filter(
      (file) => file.status === FilePond.FileStatus.PROCESSING_COMPLETE,
    );

    if (processedFiles.length === 0) {
      return;
    }

    // Collect media IDs from processed files for auto-select.
    const mediaIds = [];
    if (settings.autoSelect) {
      processedFiles.forEach((file) => {
        const mediaId = extractMediaId(file.serverId);
        if (mediaId) {
          mediaIds.push(mediaId);
        }
      });
    }

    // Trigger view refresh.
    // Note: jQuery is required here - Drupal Views AJAX uses jQuery events.
    const viewContainer = document.querySelector(
      `.js-view-dom-id-${viewDomId}`,
    );
    if (viewContainer && typeof jQuery !== 'undefined') {
      const $view = jQuery(viewContainer);

      // If auto-select is enabled, store media IDs in drupalSettings.
      // The behavior will pick these up when it re-attaches to the refreshed view.
      if (settings.autoSelect && mediaIds.length) {
        drupalSettings.filepondViews = drupalSettings.filepondViews || {};
        drupalSettings.filepondViews.pendingAutoSelect =
          drupalSettings.filepondViews.pendingAutoSelect || {};
        drupalSettings.filepondViews.pendingAutoSelect[viewDomId] = mediaIds;
      }

      $view.trigger('RefreshView');
    }

    // Show success message.
    // Message wrapper is placed outside the view, so it persists through refresh.
    const message = Drupal.t('@count file(s) uploaded successfully.', {
      '@count': processedFiles.length,
    });
    Drupal.filepond.showMessage(message, {
      type: 'status',
      context: element,
    });
  }

  /**
   * Sets up Views-specific FilePond handlers.
   *
   * @param {Object} pond
   *   The FilePond instance.
   * @param {HTMLElement} element
   *   The uploader wrapper element.
   */
  function initializeViewsHandler(pond, element) {
    // Get settings for this uploader instance by its element ID.
    const elementId = element.id;
    const settings =
      Drupal.settings?.filepondViews?.[elementId] ||
      drupalSettings?.filepondViews?.[elementId] ||
      {};

    // viewsDomId is needed for view refresh targeting.
    const viewDomId = settings.viewsDomId || element.dataset.viewsDomId;

    // Apply max files limit from view settings.
    // This reflects both configured max_files and EB cardinality limits
    // (calculated server-side in FilePondUploadArea::render).
    const maxFiles = settings.maxFiles || 0;
    if (maxFiles > 0) {
      Drupal.filepond.setMaxFiles(pond, maxFiles);
    }

    pond.on('processfiles', () =>
      handleProcessFiles(pond, viewDomId, element, settings),
    );
  }

  /**
   * Handles FilePond initialization event.
   *
   * @param {CustomEvent} e
   *   The filepond:init event.
   * @param {HTMLElement} element
   *   The uploader element.
   */
  function handleFilePondInit(e, element) {
    const { pond } = e.detail;
    if (!pond) {
      return;
    }
    initializeViewsHandler(pond, element);
  }

  /**
   * Behavior for FilePond views area.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondViews = {
    attach(context) {
      // Find all FilePond views areas, checking both context and document.
      // Entity Browser loads content dynamically, so the element might not
      // be in the passed context but could be in the document.
      let uploaders = once(
        'filepond-views',
        '.filepond-views-area[data-views-dom-id]',
        context,
      );

      // If nothing found in context, check the full document.
      // This handles cases where EB content loads after initial behavior run.
      if (uploaders.length === 0 && context !== document) {
        uploaders = once(
          'filepond-views',
          '.filepond-views-area[data-views-dom-id]',
          document,
        );
      }

      uploaders.forEach((element) => {
        // Try multiple ways to find the FilePond instance:
        // 1. Original input with _filepond property
        // 2. FilePond.find() on the root element
        // 3. Fall back to event listener
        let pond = null;

        // Method 1: Check original input (might be hidden/replaced by FilePond)
        const input = element.querySelector('[data-filepond-id]');
        if (input && input._filepond) {
          pond = input._filepond;
        }

        // Method 2: Use FilePond.find() on the filepond root element
        if (!pond && typeof FilePond !== 'undefined') {
          const pondRoot = element.querySelector('.filepond--root');
          if (pondRoot) {
            pond = FilePond.find(pondRoot);
          }
        }

        if (pond) {
          initializeViewsHandler(pond, element);
        } else {
          // Not yet initialized - listen for the event.
          element.addEventListener('filepond:init', (e) =>
            handleFilePondInit(e, element),
          );
        }
      });
    },
  };

  /**
   * Behavior for auto-selecting media after view refresh.
   *
   * When views refresh via AJAX, behaviors re-attach. This behavior checks
   * for pending auto-select media IDs and selects them.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.filepondViewsAutoSelect = {
    attach(context) {
      // Check if we have any pending auto-select.
      const pending = drupalSettings.filepondViews?.pendingAutoSelect;

      if (!pending || !Object.keys(pending).length) {
        return;
      }

      // Check if context itself is a view (AJAX refresh passes view as context).
      // Also check for views inside context (initial page load).
      let views = [];
      if (context.className?.includes('js-view-dom-id-')) {
        views = [context];
      } else if (context.querySelectorAll) {
        views = Array.from(
          context.querySelectorAll('[class*="js-view-dom-id-"]'),
        );
      }

      views.forEach((viewContainer) => {
        // Extract view DOM ID from class name.
        const classMatch = viewContainer.className.match(
          /js-view-dom-id-([a-f0-9]+)/,
        );
        if (!classMatch) {
          return;
        }

        const viewDomId = classMatch[1];

        // Check if we have pending media IDs to select for this view.
        if (pending[viewDomId]?.length) {
          const mediaIds = pending[viewDomId];

          // Clear pending before selecting to avoid re-triggering.
          delete pending[viewDomId];

          // Small delay to ensure DOM is fully ready.
          setTimeout(() => autoSelectMedia(mediaIds, viewDomId), 100);
        }
      });
    },
  };
})(Drupal, drupalSettings, once);
