<?php

declare(strict_types=1);

namespace Drupal\filepond_views\Controller;

use Drupal\filepond\Controller\UploadController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Handles FilePond upload requests for Views area context.
 *
 * Extends the main UploadController and uses the settings resolver for
 * views area option resolution. The resolver sets widget_type to 'views_area'
 * so the event subscriber knows to create media entities on upload.
 *
 * Routes:
 * - POST /filepond/views/{view_id}/{display_id}/process
 * - PATCH /filepond/views/{view_id}/{display_id}/patch/{transferId}
 * - DELETE /filepond/views/{view_id}/{display_id}/revert
 */
class ViewsUploadController extends UploadController {

  /**
   * Handles Views process requests (POST).
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsProcess(Request $request, string $view_id, string $display_id): Response {
    $options = $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);

    // Check media create access if uploading to a media type.
    $media_type = $options->context['bundle'] ?? NULL;
    if ($media_type) {
      $access_error = $this->checkMediaCreateAccess($media_type);
      if ($access_error) {
        return $access_error;
      }
    }

    // Store Entity Browser UUID in context for chunked uploads.
    // This allows PATCH requests to access EB context without query strings.
    $optionsArray = $options->toArray();
    $uuid = $request->query->get('uuid');
    if ($uuid) {
      $optionsArray['context']['uuid'] = $uuid;
    }

    return $this->handleProcess($request, $optionsArray);
  }

  /**
   * Handles Views PATCH requests for chunked uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   * @param string $transferId
   *   The transfer ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsPatch(Request $request, string $view_id, string $display_id, string $transferId): Response {
    $options = $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);

    // Check media create access if uploading to a media type.
    $media_type = $options->context['bundle'] ?? NULL;
    if ($media_type) {
      $access_error = $this->checkMediaCreateAccess($media_type);
      if ($access_error) {
        return $access_error;
      }
    }

    // Retrieve full context stored during POST (includes Entity Browser data).
    // This restores entity_browser.widget_context which may contain upload_type
    // set via form_alter (used by PortfolioSourceSubscriber).
    $optionsArray = $options->toArray();
    $transferContext = $this->uploadHandler->getTransferContext($transferId);
    if ($transferContext) {
      // Merge stored context into options context. This restores:
      // - entity_browser.uuid
      // - entity_browser.widget_context (with upload_type, etc.)
      $optionsArray['context'] = array_merge($optionsArray['context'], $transferContext);
    }

    return $this->handlePatch($request, $transferId, $optionsArray);
  }

  /**
   * Handles Views DELETE requests to revert/cancel uploads.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function viewsRevert(Request $request, string $view_id, string $display_id): Response {
    // Validate view and display exist. Revert access handled by parent.
    $this->settingsResolver->resolveFromViewsArea($view_id, $display_id);
    return $this->handleRevert($request);
  }

}
