<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond_views\Kernel;

use Drupal\filepond\UploadOptions;
use Drupal\filepond\UploadSettingsResolverInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\Entity\MediaType;
use Drupal\views\Entity\View;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Tests the FilePond Views area plugin integration.
 *
 * @group filepond_views
 */
class ViewsAreaTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'field',
    'image',
    'media',
    'views',
    'filepond',
    'filepond_views',
  ];

  /**
   * The settings resolver service.
   */
  protected UploadSettingsResolverInterface $resolver;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installConfig(['filepond', 'media']);

    $this->resolver = $this->container->get('filepond.settings_resolver');
  }

  /**
   * Creates an image media type for testing.
   *
   * @param string $id
   *   The media type ID.
   * @param array $field_settings
   *   Optional field settings overrides.
   */
  protected function createImageMediaType(string $id, array $field_settings = []): void {
    $media_type = MediaType::create([
      'id' => $id,
      'label' => ucfirst($id),
      'source' => 'image',
      'source_configuration' => [],
    ]);
    $media_type->save();

    // Create the source field.
    $source = $media_type->getSource();
    $source_field = $source->createSourceField($media_type);

    // Apply any custom field settings.
    if (!empty($field_settings)) {
      foreach ($field_settings as $key => $value) {
        $source_field->setSetting($key, $value);
      }
    }

    $source_field->getFieldStorageDefinition()->save();
    $source_field->save();

    // Set source field in media type config.
    $media_type->set('source_configuration', [
      'source_field' => $source_field->getName(),
    ]);
    $media_type->save();
  }

  /**
   * Creates a view with a FilePond upload area plugin.
   *
   * @param string $view_id
   *   The view ID.
   * @param string $display_id
   *   The display ID.
   * @param array $area_options
   *   Options for the filepond_upload area plugin.
   */
  protected function createViewWithFilePondArea(string $view_id, string $display_id, array $area_options): void {
    $view = View::create([
      'id' => $view_id,
      'label' => 'Test View',
      'base_table' => 'media_field_data',
      'display' => [
        $display_id => [
          'display_plugin' => 'default',
          'id' => $display_id,
          'display_title' => 'Default',
          'position' => 0,
          'display_options' => [
            'header' => [
              'filepond_upload' => array_merge([
                'id' => 'filepond_upload',
                'table' => 'views',
                'field' => 'filepond_upload',
                'plugin_id' => 'filepond_upload',
              ], $area_options),
            ],
          ],
        ],
      ],
    ]);
    $view->save();
  }

  /**
   * Tests resolveFromViewsArea throws for non-existent view.
   */
  public function testResolveFromViewsAreaThrowsForNonExistentView(): void {
    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('View not found');

    $this->resolver->resolveFromViewsArea('nonexistent_view', 'default');
  }

  /**
   * Tests resolveFromViewsArea throws for non-existent display.
   */
  public function testResolveFromViewsAreaThrowsForNonExistentDisplay(): void {
    // Create a view without FilePond area.
    $view = View::create([
      'id' => 'test_view',
      'label' => 'Test View',
      'base_table' => 'media_field_data',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 0,
          'display_options' => [],
        ],
      ],
    ]);
    $view->save();

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Display not found');

    $this->resolver->resolveFromViewsArea('test_view', 'nonexistent_display');
  }

  /**
   * Tests resolveFromViewsArea throws when FilePond area plugin is missing.
   */
  public function testResolveFromViewsAreaThrowsWhenPluginMissing(): void {
    // Create a view without FilePond area.
    $view = View::create([
      'id' => 'test_view_no_plugin',
      'label' => 'Test View',
      'base_table' => 'media_field_data',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 0,
          'display_options' => [
            'header' => [],
          ],
        ],
      ],
    ]);
    $view->save();

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('FilePond area plugin not found in view');

    $this->resolver->resolveFromViewsArea('test_view_no_plugin', 'default');
  }

  /**
   * Tests resolveFromViewsArea throws when media type not configured.
   */
  public function testResolveFromViewsAreaThrowsWhenMediaTypeNotConfigured(): void {
    $this->createViewWithFilePondArea('test_view_no_media', 'default', [
      'media_type' => '',
    ]);

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Media type not configured in views area');

    $this->resolver->resolveFromViewsArea('test_view_no_media', 'default');
  }

  /**
   * Tests resolveFromViewsArea throws when media type not found.
   */
  public function testResolveFromViewsAreaThrowsWhenMediaTypeNotFound(): void {
    $this->createViewWithFilePondArea('test_view_bad_media', 'default', [
      'media_type' => 'nonexistent_media_type',
    ]);

    $this->expectException(NotFoundHttpException::class);
    $this->expectExceptionMessage('Media type not found');

    $this->resolver->resolveFromViewsArea('test_view_bad_media', 'default');
  }

  /**
   * Tests resolveFromViewsArea returns options for valid configuration.
   */
  public function testResolveFromViewsAreaReturnsOptions(): void {
    // Create media type with custom field settings.
    $this->createImageMediaType('test_image', [
      'file_extensions' => 'jpg png webp',
      'max_filesize' => '20M',
      'file_directory' => 'test-uploads',
    ]);

    // Create view with FilePond area pointing to this media type.
    $this->createViewWithFilePondArea('test_view', 'default', [
      'media_type' => 'test_image',
    ]);

    $result = $this->resolver->resolveFromViewsArea('test_view', 'default');

    $this->assertInstanceOf(UploadOptions::class, $result);
    $this->assertEquals(['jpg', 'png', 'webp'], $result->allowedExtensions);
    $this->assertStringStartsWith('public://', $result->destination);
    $this->assertStringContainsString('test-uploads', $result->destination);
    $this->assertEquals(20 * 1024 * 1024, $result->maxSize);

    // Check context includes widget_type.
    $this->assertEquals('views_area', $result->context['widget_type']);
    $this->assertEquals('test_view', $result->context['view_id']);
    $this->assertEquals('default', $result->context['display_id']);
    $this->assertEquals('test_image', $result->context['bundle']);
  }

  /**
   * Tests resolveFromViewsArea works with footer area.
   */
  public function testResolveFromViewsAreaWorksWithFooter(): void {
    $this->createImageMediaType('footer_test_image');

    // Create view with FilePond area in footer instead of header.
    $view = View::create([
      'id' => 'test_view_footer',
      'label' => 'Test View Footer',
      'base_table' => 'media_field_data',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 0,
          'display_options' => [
            'footer' => [
              'filepond_upload' => [
                'id' => 'filepond_upload',
                'table' => 'views',
                'field' => 'filepond_upload',
                'plugin_id' => 'filepond_upload',
                'media_type' => 'footer_test_image',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $result = $this->resolver->resolveFromViewsArea('test_view_footer', 'default');

    $this->assertInstanceOf(UploadOptions::class, $result);
    $this->assertEquals('footer_test_image', $result->context['bundle']);
  }

  /**
   * Tests resolveFromViewsArea works with empty area.
   */
  public function testResolveFromViewsAreaWorksWithEmptyArea(): void {
    $this->createImageMediaType('empty_test_image');

    // Create view with FilePond area in empty area.
    $view = View::create([
      'id' => 'test_view_empty',
      'label' => 'Test View Empty',
      'base_table' => 'media_field_data',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_title' => 'Default',
          'position' => 0,
          'display_options' => [
            'empty' => [
              'filepond_upload' => [
                'id' => 'filepond_upload',
                'table' => 'views',
                'field' => 'filepond_upload',
                'plugin_id' => 'filepond_upload',
                'media_type' => 'empty_test_image',
              ],
            ],
          ],
        ],
      ],
    ]);
    $view->save();

    $result = $this->resolver->resolveFromViewsArea('test_view_empty', 'default');

    $this->assertInstanceOf(UploadOptions::class, $result);
    $this->assertEquals('empty_test_image', $result->context['bundle']);
  }

  /**
   * Tests MIME types are correctly derived from extensions.
   */
  public function testMimeTypesAreDerivedFromExtensions(): void {
    $this->createImageMediaType('mime_test_image', [
      'file_extensions' => 'jpg png gif',
    ]);

    $this->createViewWithFilePondArea('mime_test_view', 'default', [
      'media_type' => 'mime_test_image',
    ]);

    $result = $this->resolver->resolveFromViewsArea('mime_test_view', 'default');

    $this->assertContains('image/jpeg', $result->allowedMimeTypes);
    $this->assertContains('image/png', $result->allowedMimeTypes);
    $this->assertContains('image/gif', $result->allowedMimeTypes);
  }

  /**
   * Tests that different views can have different configurations.
   */
  public function testDifferentViewsHaveDifferentConfigs(): void {
    $this->createImageMediaType('view_test_a', [
      'file_extensions' => 'jpg',
    ]);
    $this->createImageMediaType('view_test_b', [
      'file_extensions' => 'png gif',
    ]);

    // Create two separate views with different media types.
    $this->createViewWithFilePondArea('test_view_a', 'default', [
      'media_type' => 'view_test_a',
    ]);
    $this->createViewWithFilePondArea('test_view_b', 'default', [
      'media_type' => 'view_test_b',
    ]);

    $resultA = $this->resolver->resolveFromViewsArea('test_view_a', 'default');
    $resultB = $this->resolver->resolveFromViewsArea('test_view_b', 'default');

    $this->assertEquals(['jpg'], $resultA->allowedExtensions);
    $this->assertEquals('view_test_a', $resultA->context['bundle']);

    $this->assertEquals(['png', 'gif'], $resultB->allowedExtensions);
    $this->assertEquals('view_test_b', $resultB->context['bundle']);
  }

}
