<?php

declare(strict_types=1);

namespace Drupal\filepond\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\ByteSizeMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Test form for the FilePond element.
 *
 * This form provides a simple interface for testing the FilePond upload
 * element functionality. Available at /admin/config/media/filepond/test.
 *
 * Uses default settings from filepond.settings configuration.
 */
class FilePondTestForm extends FormBase {

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'filepond_test_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('filepond.settings');
    $defaults = $config->get('defaults') ?? [];

    $form['description'] = [
      '#theme' => 'status_messages',
      '#message_list' => [
        'warning' => [
          $this->t("This form doesn't do much. It's just here to test that uploads are working. Temporary file entities will be created from whatever you upload here."),
        ],
      ],
    ];

    // Use defaults from filepond.settings configuration.
    // The element will use these defaults automatically, but we set them
    // explicitly here so users can see what values are being used.
    $form['images'] = [
      '#type' => 'filepond',
      '#title' => $this->t('Upload Images'),
      '#description' => $this->t('Upload one or more images to test the FilePond element. Using settings from configuration.'),
      '#extensions' => $defaults['extensions'] ?? NULL,
      '#max_filesize' => $defaults['max_filesize'] ?? NULL,
      '#max_files' => $defaults['max_files'] ?? NULL,
      '#upload_location' => $defaults['upload_location'] ?? NULL,
      '#config' => [
        'allowReorder' => TRUE,
        'labelIdle' => $defaults['upload_prompt'] ?? NULL,
        'maxParallelUploads' => $defaults['max_parallel_uploads'] ?? NULL,
        'styleItemPanelAspectRatio' => $defaults['item_panel_aspect_ratio'] ?? NULL,
        'previewFitMode' => $defaults['preview_fit_mode'] ?? NULL,
      ],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue('images');
    $fids = $values['fids'] ?? [];

    if (empty($fids)) {
      $this->messenger()->addWarning($this->t('No files were uploaded.'));
      return;
    }

    $this->messenger()->addStatus($this->t('Uploaded @count file(s):', [
      '@count' => count($fids),
    ]));

    $files = $this->entityTypeManager->getStorage('file')->loadMultiple($fids);
    foreach ($files as $file) {
      $this->messenger()->addStatus($this->t('- @name (@size) - ID: @id', [
        '@name' => $file->getFilename(),
        '@size' => ByteSizeMarkup::create($file->getSize()),
        '@id' => $file->id(),
      ]));
    }
  }

}
