<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\file\FileInterface;
use Drupal\media\MediaTypeInterface;

/**
 * Provides helper methods for building media source field values.
 */
trait MediaSourceFieldTrait {

  /**
   * Builds the source field value for a media entity.
   *
   * For image fields, includes dimensions to avoid ImageItem::preSave()
   * downloading the entire file from S3 just to read header bytes.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type.
   *
   * @return array|\Drupal\file\FileInterface
   *   Array with target_id/width/height for image fields, or file entity.
   */
  protected function buildSourceFieldValue(FileInterface $file, MediaTypeInterface $media_type): array|FileInterface {
    $source = $media_type->getSource();
    $field_definition = $source->getSourceFieldDefinition($media_type);

    if ($field_definition && $field_definition->getType() === 'image') {
      $dimensions = ImageDimensionHelper::getDimensions($file);
      return [
        'target_id' => $file->id(),
        'width' => $dimensions['width'] ?? NULL,
        'height' => $dimensions['height'] ?? NULL,
      ];
    }

    return $file;
  }

  /**
   * Builds thumbnail field value for image media.
   *
   * For image media, the thumbnail IS the source image. Pre-populating this
   * avoids Media::loadThumbnail() doing a redundant DB query to find a file
   * we already have.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type.
   *
   * @return array|null
   *   Thumbnail field value array, or NULL if not an image type.
   */
  protected function buildThumbnailValue(FileInterface $file, MediaTypeInterface $media_type): ?array {
    $source = $media_type->getSource();
    $field_definition = $source->getSourceFieldDefinition($media_type);

    // Only for image source fields where thumbnail = source file.
    if (!$field_definition || $field_definition->getType() !== 'image') {
      return NULL;
    }

    $dimensions = ImageDimensionHelper::getDimensions($file);

    return [
      'target_id' => $file->id(),
      'width' => $dimensions['width'] ?? NULL,
      'height' => $dimensions['height'] ?? NULL,
      'alt' => $file->getFilename(),
    ];
  }

  /**
   * Builds complete media entity values including optimized thumbnail.
   *
   * Use this instead of manually building the values array. It handles:
   * - Source field with dimensions (avoids S3 downloads)
   * - Thumbnail field pre-populated (avoids redundant DB query)
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   * @param \Drupal\media\MediaTypeInterface $media_type
   *   The media type.
   *
   * @return array
   *   Entity values array ready for Media::create().
   */
  protected function buildMediaValues(FileInterface $file, MediaTypeInterface $media_type): array {
    $source = $media_type->getSource();
    $source_field = $source->getConfiguration()['source_field'];

    $values = [
      'bundle' => $media_type->id(),
      $source_field => $this->buildSourceFieldValue($file, $media_type),
    ];

    // Pre-populate thumbnail for image types.
    $thumbnail = $this->buildThumbnailValue($file, $media_type);
    if ($thumbnail !== NULL) {
      $values['thumbnail'] = $thumbnail;
    }

    return $values;
  }

}
