<?php

declare(strict_types=1);

namespace Drupal\filepond\Upload;

use Drupal\file\Upload\UploadedFileInterface;
use Symfony\Component\Validator\ConstraintViolationList;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Represents an uploaded file from a temp location (not $_FILES).
 *
 * Unlike FormUploadedFile, this doesn't wrap a Symfony UploadedFile.
 * Core's FileUploadHandler checks instanceof FormUploadedFile to decide
 * whether to use moveUploadedFile() (which fails for non-$_FILES uploads)
 * or regular move(). This class uses the move() path.
 */
class TempUploadedFile implements UploadedFileInterface {

  /**
   * The absolute path to the temp file.
   */
  protected string $path;

  /**
   * The original filename.
   */
  protected string $originalName;

  /**
   * The file size in bytes.
   */
  protected int $size;

  /**
   * Constructs a TempUploadedFile.
   *
   * @param string $path
   *   The absolute path to the temp file.
   * @param string $original_name
   *   The original filename.
   */
  public function __construct(string $path, string $original_name) {
    $this->path = $path;
    $this->originalName = $original_name;
    $this->size = (int) @filesize($path);
  }

  /**
   * {@inheritdoc}
   */
  public function getClientOriginalName(): string {
    return $this->originalName;
  }

  /**
   * {@inheritdoc}
   */
  public function isValid(): bool {
    @trigger_error(__METHOD__ . '() is deprecated in drupal:10.3.0 and is removed from drupal:11.0.0. Use \Drupal\file\Validation\UploadedFileValidatorInterface::validate() instead. See https://www.drupal.org/node/3375456', E_USER_DEPRECATED);
    return file_exists($this->path);
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessage(): string {
    @trigger_error(__METHOD__ . '() is deprecated in drupal:10.3.0 and is removed from drupal:11.0.0. Use \Drupal\file\Validation\UploadedFileValidatorInterface::validate() instead. See https://www.drupal.org/node/3375456', E_USER_DEPRECATED);
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getError(): int {
    @trigger_error(__METHOD__ . '() is deprecated in drupal:10.3.0 and is removed from drupal:11.0.0. Use \Drupal\file\Validation\UploadedFileValidatorInterface::validate() instead. See https://www.drupal.org/node/3375456', E_USER_DEPRECATED);
    return UPLOAD_ERR_OK;
  }

  /**
   * {@inheritdoc}
   */
  public function getSize(): int {
    return $this->size;
  }

  /**
   * {@inheritdoc}
   */
  public function getRealPath(): string|false {
    return $this->path;
  }

  /**
   * {@inheritdoc}
   */
  public function getPathname(): string {
    return $this->path;
  }

  /**
   * {@inheritdoc}
   */
  public function getFilename(): string {
    return basename($this->path);
  }

  /**
   * {@inheritdoc}
   */
  public function validate(ValidatorInterface $validator, array $options = []): ConstraintViolationListInterface {
    // Return empty list - temp files don't have PHP upload errors.
    // Core's UploadedFileConstraint expects Symfony's UploadedFile and
    // validates UPLOAD_ERR_* codes which don't apply to our temp files.
    return new ConstraintViolationList();
  }

}
