<?php

declare(strict_types=1);

namespace Drupal\filepond;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Bytes;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\Core\Utility\Token;
use Drupal\filepond\Element\FilePondUploader;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Resolves upload settings from various configuration sources.
 *
 * This service consolidates the common pattern of extracting upload
 * configuration (extensions, max size, destination) from different sources.
 * Each source (State API, field definitions, media types, etc.) stores
 * settings differently, but they all need to be converted to the same
 * standardized format for FilePondUploadHandler.
 */
class UploadSettingsResolver implements UploadSettingsResolverInterface {

  /**
   * Default file extensions if none specified.
   */
  protected const DEFAULT_EXTENSIONS = 'png gif jpg jpeg';

  /**
   * Tempstore collection for form config.
   */
  protected const TEMPSTORE_COLLECTION = 'filepond.form_config';

  /**
   * Constructs an UploadSettingsResolver.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\TempStore\SharedTempStoreFactory $tempStoreFactory
   *   The shared tempstore factory.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container (for optional EB service).
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected SharedTempStoreFactory $tempStoreFactory,
    protected Token $token,
    protected RequestStack $requestStack,
    protected ContainerInterface $container,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function storeFormConfig(string $formId, string $elementName, array $config): string {
    $config_hash = $this->hashConfig($config);
    $key = $formId . ':' . $elementName . ':' . $config_hash;
    $tempstore = $this->tempStoreFactory->get(self::TEMPSTORE_COLLECTION);

    // Only write if not already stored (hash ensures uniqueness).
    if ($tempstore->get($key) === NULL) {
      $tempstore->set($key, $config);
    }

    return $config_hash;
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromState(string $formId, string $elementName, string $configHash): ?UploadOptions {
    $key = $formId . ':' . $elementName . ':' . $configHash;
    $config = $this->tempStoreFactory->get(self::TEMPSTORE_COLLECTION)->get($key);

    if (!is_array($config)) {
      return NULL;
    }

    // Extract extensions.
    $extensionsString = $config['extensions'] ?? self::DEFAULT_EXTENSIONS;
    $extensions = $this->parseExtensions($extensionsString);

    // Convert max filesize to bytes.
    $maxSize = $this->parseMaxSize($config['max_filesize'] ?? NULL);

    // Replace tokens in upload location at request time.
    $destination = $config['upload_location'] ?? 'public://';
    $destination = $this->token->replace($destination, [], ['clear' => TRUE]);

    return new UploadOptions(
      allowedExtensions: $extensions,
      allowedMimeTypes: $this->extensionsToMimeTypes($extensions),
      destination: $destination,
      maxSize: $maxSize,
      context: $config['context'] ?? [],
    );
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromField(string $entityType, string $bundle, string $fieldName): UploadOptions {
    // Validate entity type exists.
    if (!$this->entityTypeManager->hasDefinition($entityType)) {
      throw new NotFoundHttpException('Entity type not found');
    }

    $fieldDefinitions = $this->entityFieldManager->getFieldDefinitions($entityType, $bundle);

    if (!isset($fieldDefinitions[$fieldName])) {
      throw new NotFoundHttpException('Field not found');
    }

    $fieldDefinition = $fieldDefinitions[$fieldName];
    $source_field_settings = $fieldDefinition->getSettings();
    $source_field_storage = $fieldDefinition->getFieldStorageDefinition()->getSettings();

    return $this->buildOptionsFromFieldSettings($source_field_settings, $source_field_storage, [
      'widget_type' => 'field_widget',
      'entity_type' => $entityType,
      'bundle' => $bundle,
      'field_name' => $fieldName,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromMediaType(string $mediaTypeId): UploadOptions {
    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager->getStorage('media_type')->load($mediaTypeId);
    if (!$mediaType) {
      throw new NotFoundHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new NotFoundHttpException('Media type has no source field');
    }

    $source_field_settings = $sourceField->getSettings();
    $source_field_storage = $sourceField->getFieldStorageDefinition()->getSettings();

    return $this->buildOptionsFromFieldSettings($source_field_settings, $source_field_storage, [
      'widget_type' => 'media_library',
      'entity_type' => 'media',
      'bundle' => $mediaTypeId,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromWidget(string $entityBrowserId, string $widgetUuid): UploadOptions {
    /** @var \Drupal\entity_browser\EntityBrowserInterface|null $entityBrowser */
    $entityBrowser = $this->entityTypeManager
      ->getStorage('entity_browser')
      ->load($entityBrowserId);

    if (!$entityBrowser) {
      throw new NotFoundHttpException('Entity browser not found');
    }

    try {
      $widget = $entityBrowser->getWidgets()->get($widgetUuid);
    }
    catch (PluginNotFoundException $e) {
      throw new NotFoundHttpException('Widget not found');
    }
    if (!$widget) {
      throw new NotFoundHttpException('Widget not found');
    }

    if ($widget->getPluginId() !== 'filepond_media') {
      throw new AccessDeniedHttpException('Invalid widget type');
    }

    $config = $widget->getConfiguration();
    $widgetSettings = $config['settings'] ?? [];

    $mediaTypeId = $widgetSettings['media_type'] ?? '';
    if (empty($mediaTypeId)) {
      throw new AccessDeniedHttpException('Media type not configured');
    }

    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager
      ->getStorage('media_type')
      ->load($mediaTypeId);

    if (!$mediaType) {
      throw new AccessDeniedHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new AccessDeniedHttpException('Source field not found');
    }

    $source_field_settings = $sourceField->getSettings();
    $source_field_storage = $sourceField->getFieldStorageDefinition()->getSettings();

    // Build overrides only if NOT inheriting from media type.
    $overrides = [];
    if (empty($widgetSettings['inherit_settings'])) {
      if (!empty($widgetSettings['upload_location'])) {
        $overrides['upload_location'] = $widgetSettings['upload_location'];
      }
      if (!empty($widgetSettings['max_filesize'])) {
        $overrides['max_filesize'] = $widgetSettings['max_filesize'];
      }
    }

    return $this->buildOptionsFromFieldSettings($source_field_settings, $source_field_storage, [
      'widget_type' => 'eb_widget',
      'entity_browser' => $entityBrowserId,
      'widget_uuid' => $widgetUuid,
      'entity_type' => 'media',
      'bundle' => $mediaTypeId,
    ], $overrides);
  }

  /**
   * {@inheritdoc}
   */
  public function resolveFromViewsArea(string $viewId, string $displayId): UploadOptions {
    $view = Views::getView($viewId);
    if (!$view) {
      throw new NotFoundHttpException('View not found');
    }

    if (!$view->setDisplay($displayId)) {
      throw new NotFoundHttpException('Display not found');
    }

    // Find the FilePond area plugin in this display.
    $areaOptions = $this->findViewsAreaPluginOptions($view, $displayId);
    if (!$areaOptions) {
      throw new NotFoundHttpException('FilePond area plugin not found in view');
    }

    // Get media type from area options.
    $mediaTypeId = $areaOptions['media_type'] ?? NULL;
    if (empty($mediaTypeId)) {
      throw new NotFoundHttpException('Media type not configured in views area');
    }

    /** @var \Drupal\media\MediaTypeInterface|null $mediaType */
    $mediaType = $this->entityTypeManager->getStorage('media_type')->load($mediaTypeId);
    if (!$mediaType) {
      throw new NotFoundHttpException('Media type not found');
    }

    $source = $mediaType->getSource();
    $sourceField = $source->getSourceFieldDefinition($mediaType);
    if (!$sourceField) {
      throw new NotFoundHttpException('Media type has no source field');
    }

    $source_field_settings = $sourceField->getSettings();
    $source_field_storage = $sourceField->getFieldStorageDefinition()->getSettings();

    // Build overrides only if NOT inheriting from media type.
    $overrides = [];
    if (empty($areaOptions['inherit_settings'])) {
      if (!empty($areaOptions['upload_location'])) {
        $overrides['upload_location'] = $areaOptions['upload_location'];
      }
      if (!empty($areaOptions['max_filesize'])) {
        $overrides['max_filesize'] = $areaOptions['max_filesize'];
      }
    }

    // Build base context.
    $context = [
      'widget_type' => 'views_area',
      'view_id' => $viewId,
      'display_id' => $displayId,
      'entity_type' => 'media',
      'bundle' => $mediaTypeId,
    ];

    // Include Entity Browser context if available.
    $ebContext = $this->getEntityBrowserContext();
    if ($ebContext) {
      $context['entity_browser'] = $ebContext;
    }

    return $this->buildOptionsFromFieldSettings($source_field_settings, $source_field_storage, $context, $overrides);
  }

  /**
   * Builds upload options from source field settings with optional overrides.
   *
   * This is the core method that converts field settings to standardized
   * UploadOptions. Extensions always come from the source field (media type),
   * while destination and max size can be overridden by the widget.
   *
   * @param array $source_field_settings
   *   Source field instance settings containing:
   *   - file_extensions: Allowed extensions (e.g., 'jpg png gif').
   *   - max_filesize: Max file size (e.g., '10M').
   *   - file_directory: Upload subdirectory with tokens.
   * @param array $source_field_storage
   *   Source field storage settings containing:
   *   - uri_scheme: File system scheme ('public' or 'private').
   * @param array $context
   *   Additional context for event subscribers.
   * @param array $overrides
   *   Optional overrides from widget settings:
   *   - upload_location: Custom upload path (with tokens).
   *   - max_filesize: Custom max file size (e.g., '10M').
   *
   * @return \Drupal\filepond\UploadOptions
   *   Upload options.
   */
  protected function buildOptionsFromFieldSettings(array $source_field_settings, array $source_field_storage, array $context, array $overrides = []): UploadOptions {
    // Extensions ALWAYS from media type source field - no override allowed.
    $extensions_string = $source_field_settings['file_extensions'] ?? self::DEFAULT_EXTENSIONS;
    $extensions = $this->parseExtensions($extensions_string);

    // Destination: use widget override if provided, otherwise source field.
    if (!empty($overrides['upload_location'])) {
      $destination = $this->token->replace($overrides['upload_location'], [], ['clear' => TRUE]);
    }
    else {
      $file_directory = $this->token->replace($source_field_settings['file_directory'] ?? '', [], ['clear' => TRUE]);
      $uri_scheme = $source_field_storage['uri_scheme'] ?? 'public';
      $destination = $uri_scheme . '://' . $file_directory;
    }

    // Max size: use widget override if provided, otherwise source field.
    $max_size = !empty($overrides['max_filesize'])
      ? $this->parseMaxSize($overrides['max_filesize'])
      : $this->parseMaxSize($source_field_settings['max_filesize'] ?? NULL);

    return new UploadOptions(
      allowedExtensions: $extensions,
      allowedMimeTypes: $this->extensionsToMimeTypes($extensions),
      destination: $destination,
      maxSize: $max_size,
      context: $context,
    );
  }

  /**
   * Parses an extensions string into an array.
   *
   * @param string $extensionsString
   *   Space-separated extensions (e.g., 'jpg jpeg png gif').
   *
   * @return array
   *   Array of lowercase extensions.
   */
  protected function parseExtensions(string $extensionsString): array {
    return array_filter(
      array_map('trim', explode(' ', strtolower($extensionsString)))
    );
  }

  /**
   * Parses max filesize to bytes.
   *
   * @param string|int|null $maxFilesize
   *   Max filesize as string (e.g., '10M') or bytes.
   *
   * @return int|null
   *   Max size in bytes, or NULL if not set.
   */
  protected function parseMaxSize(string|int|null $maxFilesize): ?int {
    if (empty($maxFilesize)) {
      return NULL;
    }

    if (is_numeric($maxFilesize)) {
      return (int) $maxFilesize;
    }

    // Bytes::toNumber returns float, cast to int.
    return (int) Bytes::toNumber($maxFilesize);
  }

  /**
   * Converts file extensions to MIME types.
   *
   * @param array $extensions
   *   Array of file extensions.
   *
   * @return array
   *   Array of MIME types.
   */
  protected function extensionsToMimeTypes(array $extensions): array {
    return FilePondUploader::extensionsToMimeTypes(implode(' ', $extensions));
  }

  /**
   * Finds the FilePond area plugin options in a view display.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view.
   * @param string $displayId
   *   The display ID.
   *
   * @return array|null
   *   The area plugin options, or NULL if not found.
   */
  protected function findViewsAreaPluginOptions($view, string $displayId): ?array {
    $display = $view->displayHandlers->get($displayId);
    if (!$display) {
      return NULL;
    }

    // Check all area types.
    foreach (['header', 'footer', 'empty'] as $areaType) {
      $areas = $display->getOption($areaType);
      if (empty($areas)) {
        continue;
      }

      foreach ($areas as $area) {
        if (($area['plugin_id'] ?? '') === 'filepond_upload') {
          return $area;
        }
      }
    }

    return NULL;
  }

  /**
   * Generates a short hash of the config array.
   *
   * Used to create unique tempstore keys that change when config changes.
   * This ensures form_alter modifications and config updates are respected.
   *
   * @param array $config
   *   The config array to hash.
   *
   * @return string
   *   A short URL-safe hash (8 characters).
   */
  protected function hashConfig(array $config): string {
    // Sort to ensure consistent hash regardless of key order.
    ksort($config);
    $serialized = serialize($config);
    // Use first 8 chars of base64-encoded hash (URL-safe).
    return substr(strtr(base64_encode(hash('sha256', $serialized, TRUE)), '+/', '-_'), 0, 8);
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityBrowserContext(): ?array {
    $request = $this->requestStack->getCurrentRequest();
    if (!$request || !$request->query->has('uuid')) {
      return NULL;
    }

    $uuid = $request->query->get('uuid');

    // Entity Browser is an optional module.
    if (!$this->container->has('entity_browser.selection_storage')) {
      return NULL;
    }

    $storage = $this->container->get('entity_browser.selection_storage')->get($uuid);
    if (!$storage) {
      return NULL;
    }

    $widget_context = $storage['widget_context'] ?? [];

    // Return UUID, cardinality info, and full widget_context.
    // Custom keys set via form_alter (like upload_type) are accessible here.
    return [
      'uuid' => $uuid,
      'widget_context' => $widget_context,
    ];
  }

}
