<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Base class for FilePond JavaScript tests.
 *
 * Provides helper methods for:
 * - Creating test files
 * - Dropping files into FilePond elements
 * - Waiting for uploads to complete
 * - Reading file IDs from hidden inputs.
 */
abstract class FilePondTestBase extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'file',
    'image',
    'filepond',
  ];

  /**
   * Test image as base64 (tiny valid 1x1 PNG).
   *
   * @var string
   * @cspell:disable-next-line
   */
  protected string $testImageData = 'iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==';

  /**
   * A test user with upload permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Enable CDN mode so FilePond libraries load without local files.
    $this->config('filepond.settings')->set('use_cdn', TRUE)->save();

    // Create user with permissions for the built-in test form.
    $this->testUser = $this->drupalCreateUser([
      'access content',
      'administer site configuration',
      'filepond upload files',
    ]);
    $this->drupalLogin($this->testUser);
  }

  /**
   * Waits for FilePond to be initialized on the page.
   *
   * This includes waiting for:
   * - The FilePond wrapper element (created by Drupal)
   * - The FilePond root element (created by JS initialization)
   * - The FilePond JS global
   * - The drupalSettings config (may be delayed by BigPipe)
   *
   * @param int $timeout
   *   Timeout in milliseconds.
   */
  protected function waitForFilePondInit(int $timeout = 10000): void {
    // First check that the page actually loaded and has a form.
    // Note: statusCodeEquals() not available with WebDriver.
    $this->assertSession()->elementExists('css', 'form');

    // Wait for the FilePond wrapper element (created by Drupal).
    $wrapper = $this->assertSession()->waitForElement('css', '.filepond-wrapper', $timeout);
    $this->assertNotNull($wrapper, 'FilePond wrapper element should exist');

    // Wait for FilePond JS to initialize (creates .filepond--root).
    $root = $this->assertSession()->waitForElement('css', '.filepond--root', $timeout);
    if ($root === NULL) {
      // Debug: output page content to see what's happening.
      $content = $this->getSession()->getPage()->getContent();
      // Check for all script tags.
      preg_match_all('/<script[^>]*src="([^"]*)"/', $content, $matches);
      $allScripts = $matches[1] ?? [];
      // Look for filepond wrapper content.
      preg_match('/<div[^>]*class="[^"]*filepond-wrapper[^"]*"[^>]*>(.{0,500})/s', $content, $wrapperMatch);
      $wrapperContent = $wrapperMatch[0] ?? 'not found';
      $this->fail('FilePond root element missing. Scripts: ' . implode(', ', array_slice($allScripts, 0, 10)) . '. Wrapper HTML: ' . $wrapperContent);
    }

    // Wait for FilePond JS global to be available.
    $jsReady = $this->getSession()->wait($timeout, 'typeof FilePond !== "undefined"');
    $this->assertTrue($jsReady, 'FilePond JS global should be available');

    // Wait for drupalSettings config to be available.
    // BigPipe may stream this after initial page load, so we need to wait.
    // The config is needed for the server config object (URLs, etc).
    $configReady = $this->getSession()->wait($timeout,
      'typeof drupalSettings !== "undefined" && ' .
      'drupalSettings.filepond && ' .
      'drupalSettings.filepond.instances && ' .
      'Object.keys(drupalSettings.filepond.instances).length > 0'
    );
    if (!$configReady) {
      $settingsState = $this->getSession()->evaluateScript(
        'return { ' .
        'hasDrupalSettings: typeof drupalSettings !== "undefined", ' .
        'hasFilepond: !!(drupalSettings && drupalSettings.filepond), ' .
        'hasInstances: !!(drupalSettings && drupalSettings.filepond && drupalSettings.filepond.instances), ' .
        'instanceKeys: drupalSettings && drupalSettings.filepond && drupalSettings.filepond.instances ? Object.keys(drupalSettings.filepond.instances) : [] ' .
        '};'
      );
      $this->fail(
        'drupalSettings.filepond.instances not ready. BigPipe may not have completed. State: ' .
        json_encode($settingsState)
      );
    }
  }

  /**
   * Creates a test file in public://.
   *
   * @param string $name
   *   The filename (without extension).
   * @param string $extension
   *   The file extension.
   *
   * @return string
   *   The real path to the created file.
   */
  protected function createTestFile(string $name, string $extension = 'png'): string {
    $path = 'public://' . $name . '.' . $extension;
    file_put_contents($path, base64_decode($this->testImageData));
    return \Drupal::service('file_system')->realpath($path);
  }

  /**
   * Creates a medium test file (500KB) to increase upload time.
   *
   * Larger files take longer to upload, increasing the likelihood that
   * parallel uploads will have overlapping completion callbacks.
   *
   * @param string $name
   *   The filename (without extension).
   *
   * @return string
   *   The real path to the created file.
   */
  protected function createMediumTestFile(string $name): string {
    $path = 'public://' . $name . '.jpg';
    // Create minimal JPEG header followed by padding to reach 500KB.
    // cspell:disable
    $jpegHeader = base64_decode(
      '/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkS' .
      'Ew8UHRofHh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJ' .
      'CQwLDBgNDRgyIRwhMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIy' .
      'MjIyMjIyMjIyMjIyMjL/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAn/' .
      'xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/' .
      'xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwCwAB//2Q=='
    );
    // cspell:enable
    // Pad to 500KB.
    $targetSize = 500 * 1024;
    $padding = str_repeat("\x00", $targetSize - strlen($jpegHeader));
    file_put_contents($path, $jpegHeader . $padding);
    return \Drupal::service('file_system')->realpath($path);
  }

  /**
   * Creates a large test file to force chunked uploads.
   *
   * Creates a valid JPEG file of the specified size. Chunked uploads are
   * triggered when the file exceeds the chunk size threshold (default 1MB).
   *
   * @param string $name
   *   The filename (without extension).
   * @param int $sizeInMb
   *   Size in megabytes.
   *
   * @return string
   *   The real path to the created file.
   */
  protected function createLargeTestFile(string $name, int $sizeInMb = 2): string {
    $path = 'public://' . $name . '.jpg';
    // Create minimal JPEG header followed by padding to reach target size.
    // This is a valid enough JPEG for file validation purposes.
    // cspell:disable
    $jpegHeader = base64_decode(
      '/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkS' .
      'Ew8UHRofHh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJ' .
      'CQwLDBgNDRgyIRwhMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIy' .
      'MjIyMjIyMjIyMjIyMjL/wAARCAABAAEDASIAAhEBAxEB/8QAFQABAQAAAAAAAAAAAAAAAAAAAAn/' .
      'xAAUEAEAAAAAAAAAAAAAAAAAAAAA/8QAFQEBAQAAAAAAAAAAAAAAAAAAAAX/' .
      'xAAUEQEAAAAAAAAAAAAAAAAAAAAA/9oADAMBAAIRAxEAPwCwAB//2Q=='
    );
    // cspell:enable
    // Pad to target size.
    $targetSize = $sizeInMb * 1024 * 1024;
    $padding = str_repeat("\x00", $targetSize - strlen($jpegHeader));
    file_put_contents($path, $jpegHeader . $padding);
    return \Drupal::service('file_system')->realpath($path);
  }

  /**
   * Drops a file into a FilePond element.
   *
   * Uses the pattern from dropzonejs tests: inject a fake file input,
   * attach the file, then trigger FilePond's addFile method.
   *
   * @param string $filePath
   *   The real path to the file.
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function dropFileToFilePond(string $filePath, string $selector = '.filepond--root'): void {
    $session = $this->getSession();

    // Generate unique ID for the temporary input.
    $inputId = 'filepond-test-input-' . uniqid();

    // Add temporary file input inside the FilePond element.
    $js = "document.querySelector('$selector').insertAdjacentHTML('beforeend', '<input type=\"file\" id=\"$inputId\" style=\"display:none\">');";
    $session->executeScript($js);

    // Attach file to the input.
    $session->getPage()->attachFileToField($inputId, $filePath);

    // Trigger FilePond addFile from the input.
    $drop = <<<JS
      (function() {
        var input = document.getElementById('$inputId');
        var pond = FilePond.find(document.querySelector('$selector'));
        if (pond && input.files[0]) {
          pond.addFile(input.files[0]);
        }
        input.remove();
      })();
    JS;
    $session->executeScript($drop);
  }

  /**
   * Waits for FilePond to finish processing all files.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   * @param int $timeout
   *   Timeout in milliseconds.
   */
  protected function waitForFilePondComplete(string $selector = '.filepond--root', int $timeout = 30000): void {
    $completeSelector = $selector . ' .filepond--item[data-filepond-item-state="processing-complete"]';
    $element = $this->assertSession()->waitForElementVisible('css', $completeSelector, $timeout);

    // If element wasn't found, get diagnostic info about why.
    if ($element === NULL) {
      $fileStates = $this->getFilePondFileState($selector);
      $itemStates = $this->getSession()->evaluateScript(
        "return Array.from(document.querySelectorAll('$selector .filepond--item')).map(el => el.getAttribute('data-filepond-item-state'));"
      );
      $csrfStatus = $this->getCsrfTokenStatus();
      $jsStatus = $this->getFilePondJsStatus($selector);

      $this->fail(sprintf(
        "Timed out waiting for processing-complete.\n" .
        "FilePond states: %s\n" .
        "DOM item states: %s\n" .
        "CSRF token: %s\n" .
        "JS status: %s",
        json_encode($fileStates),
        json_encode($itemStates),
        json_encode($csrfStatus),
        json_encode($jsStatus)
      ));
    }
  }

  /**
   * Waits for a specific number of files to be ready in FilePond.
   *
   * This waits for files to reach a "ready" state, which includes:
   * - Newly uploaded files with "processing-complete" state
   * - Existing/local files with "idle" state (loaded from server)
   *
   * @param int $count
   *   Expected number of ready files.
   * @param string $selector
   *   CSS selector for the FilePond root element.
   * @param int $timeout
   *   Timeout in milliseconds.
   */
  protected function waitForFilePondFileCount(int $count, string $selector = '.filepond--root', int $timeout = 30000): void {
    $start = microtime(TRUE);
    $timeoutSec = $timeout / 1000;

    while ((microtime(TRUE) - $start) < $timeoutSec) {
      $actual = $this->getFilePondFileCount($selector);
      if ($actual >= $count) {
        // Wait for ALL files to be ready (processing-complete OR idle).
        $readyCount = $this->getCompletedFileCount($selector);
        if ($readyCount >= $count) {
          return;
        }
      }
      usleep(100000);
    }

    // Timeout - provide diagnostic info.
    $fileStates = $this->getFilePondFileState($selector);
    $this->fail(sprintf(
      "Timed out waiting for %d ready files in FilePond. Current state: %s",
      $count,
      json_encode($fileStates)
    ));
  }

  /**
   * Gets the count of files that are ready (uploaded or existing).
   *
   * This counts files with either:
   * - "processing-complete" state: Newly uploaded files
   * - "idle" state: Existing/local files loaded from the server.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return int
   *   Number of files in a ready state.
   */
  protected function getCompletedFileCount(string $selector = '.filepond--root'): int {
    // Count both processing-complete (newly uploaded) and idle (existing/local)
    // files. Existing files loaded via the 'files' option have 'idle' state
    // because they didn't go through the upload process.
    return (int) $this->getSession()->evaluateScript(
      "return document.querySelectorAll('$selector .filepond--item[data-filepond-item-state=\"processing-complete\"], $selector .filepond--item[data-filepond-item-state=\"idle\"]').length;"
    );
  }

  /**
   * Gets signed file IDs from FilePond hidden input.
   *
   * Use this for standalone FilePond form elements (e.g., test forms).
   *
   * @param string $fieldName
   *   The form element name (e.g., 'filepond' or 'upload').
   *
   * @return array
   *   Array of signed file IDs (format: "fid:hmac").
   */
  protected function getFilePondFileIds(string $fieldName): array {
    $input = $this->getSession()->getPage()->find('css', "input[name='{$fieldName}[fids]']");
    if (!$input) {
      return [];
    }
    $value = $input->getValue();
    // Hidden field uses semicolon separator.
    return $value ? array_filter(explode(';', $value)) : [];
  }

  /**
   * Gets file IDs from a field widget's hidden input.
   *
   * Field widgets use a different naming pattern: field_name[0][fids].
   * Use this for testing FilePond on entity field widgets.
   *
   * @param string $fieldName
   *   The field name (e.g., 'field_image').
   *
   * @return array
   *   Array of signed file IDs (format: "fid:hmac").
   */
  protected function getFieldWidgetFileIds(string $fieldName): array {
    $input = $this->getSession()->getPage()->find('css', "input[name='{$fieldName}[0][fids]']");
    if (!$input) {
      return [];
    }
    $value = $input->getValue();
    // Hidden field uses semicolon separator.
    return $value ? array_filter(explode(';', $value)) : [];
  }

  /**
   * Waits for the hidden field to have a specific number of file IDs.
   *
   * This is important because the hidden field update may happen slightly
   * after the UI updates. Use this instead of immediately checking count.
   *
   * @param string $fieldName
   *   The field name (e.g., 'field_image').
   * @param int $expectedCount
   *   The expected number of file IDs.
   * @param int $timeout
   *   Timeout in milliseconds.
   *
   * @return array
   *   Array of file IDs once the expected count is reached.
   */
  protected function waitForFieldWidgetFileIds(string $fieldName, int $expectedCount, int $timeout = 10000): array {
    $start = microtime(TRUE);
    $timeoutSec = $timeout / 1000;

    while ((microtime(TRUE) - $start) < $timeoutSec) {
      $ids = $this->getFieldWidgetFileIds($fieldName);
      if (count($ids) >= $expectedCount) {
        return $ids;
      }
      usleep(200000);
    }

    // Return whatever we have, let the assertion fail with actual count.
    return $this->getFieldWidgetFileIds($fieldName);
  }

  /**
   * Extracts numeric file ID from a signed token.
   *
   * The server returns signed tokens in format "fid:hmac" to prevent
   * guessing attacks. This extracts just the file ID for entity loading.
   *
   * @param string $signedToken
   *   The signed token (e.g., "123:abc...").
   *
   * @return string
   *   The numeric file ID.
   */
  protected function extractFileId(string $signedToken): string {
    if (str_contains($signedToken, ':')) {
      [$fileId] = explode(':', $signedToken, 2);
      return $fileId;
    }
    return $signedToken;
  }

  /**
   * Gets the number of files currently in FilePond.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return int
   *   The number of files.
   */
  protected function getFilePondFileCount(string $selector = '.filepond--root'): int {
    $result = $this->getSession()->evaluateScript(
      "return FilePond.find(document.querySelector('$selector'))?.getFiles().length || 0;"
    );
    return (int) $result;
  }

  /**
   * Checks if the transfer ID to file ID mapping exists in JavaScript.
   *
   * This verifies that chunked uploads correctly capture file IDs.
   *
   * @return bool
   *   TRUE if at least one mapping exists.
   */
  protected function hasTransferIdMapping(): bool {
    $result = $this->getSession()->evaluateScript(
      "return Object.keys(Drupal.filepond?.transferToFileId || {}).length > 0;"
    );
    return (bool) $result;
  }

  /**
   * Clicks the remove button on the first FilePond item.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function removeFirstFilePondItem(string $selector = '.filepond--root'): void {
    // Wait for remove button to be visible (appears on hover/complete).
    $buttonSelector = $selector . ' .filepond--action-remove-item';
    $removeButton = $this->assertSession()->waitForElementVisible('css', $buttonSelector, 5000);

    if ($removeButton) {
      // Scroll element into view and click.
      $this->getSession()->executeScript(
        "document.querySelector('$buttonSelector').scrollIntoView({block: 'center'});"
      );
      // Brief pause for scroll.
      usleep(100000);
      $removeButton->click();
    }
  }

  /**
   * Waits for all FilePond items to be removed.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   * @param int $timeout
   *   Timeout in milliseconds.
   */
  protected function waitForFilePondEmpty(string $selector = '.filepond--root', int $timeout = 10000): void {
    $itemSelector = $selector . ' .filepond--item';
    $this->assertSession()->waitForElementRemoved('css', $itemSelector, $timeout);
  }

  /**
   * Gets detailed FilePond file state for debugging.
   *
   * Returns an array of file objects with status, serverId, origin, etc.
   * Use this to diagnose where in the upload chain things are failing.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Array of file state objects with keys: id, serverId, status, origin.
   */
  protected function getFilePondFileState(string $selector = '.filepond--root'): array {
    $result = $this->getSession()->evaluateScript(
      "return (FilePond.find(document.querySelector('$selector'))?.getFiles() || []).map(f => ({
        id: f.id,
        serverId: f.serverId,
        status: f.status,
        origin: f.origin,
        filename: f.filename
      }));"
    );
    return is_array($result) ? $result : [];
  }

  /**
   * Gets detailed file state including error information.
   *
   * Use this when uploads fail to get comprehensive error details.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Array of file objects with error details.
   */
  protected function getFilePondFileErrors(string $selector = '.filepond--root'): array {
    $result = $this->getSession()->evaluateScript(
      "return (FilePond.find(document.querySelector('$selector'))?.getFiles() || []).map(f => ({
        filename: f.filename,
        fileSize: f.fileSize,
        fileType: f.fileType,
        status: f.status,
        statusName: (function(s) {
          var names = {
            1: 'INIT',
            2: 'IDLE',
            3: 'PROCESSING_QUEUED',
            4: 'PROCESSING',
            5: 'PROCESSING_COMPLETE',
            6: 'PROCESSING_ERROR',
            7: 'PROCESSING_REVERT_ERROR',
            8: 'LOADING',
            9: 'LOAD_ERROR'
          };
          return names[s] || 'UNKNOWN(' + s + ')';
        })(f.status),
        serverId: f.serverId,
        serverError: f.serverError || null,
        processingError: f.processingMessage || null
      }));"
    );
    return is_array($result) ? $result : [];
  }

  /**
   * Asserts file upload succeeded with detailed error reporting.
   *
   * Use this instead of just checking file count to get clear error messages
   * when uploads fail due to validation, server errors, etc.
   *
   * @param int $expectedCount
   *   Expected number of successfully uploaded files.
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function assertFilesUploadedSuccessfully(int $expectedCount, string $selector = '.filepond--root'): void {
    $files = $this->getFilePondFileErrors($selector);
    $successful = [];
    $failed = [];

    foreach ($files as $file) {
      // Status 5 = PROCESSING_COMPLETE.
      if ($file['status'] === 5) {
        $successful[] = $file;
      }
      else {
        $failed[] = $file;
      }
    }

    if (count($failed) > 0) {
      $errorDetails = [];
      foreach ($failed as $file) {
        $detail = sprintf(
          '%s (%s, %s bytes): %s',
          $file['filename'],
          $file['fileType'],
          $file['fileSize'],
          $file['statusName']
        );
        if ($file['serverError']) {
          $detail .= ' - Server: ' . $file['serverError'];
        }
        if ($file['processingError']) {
          $detail .= ' - Processing: ' . $file['processingError'];
        }
        $errorDetails[] = $detail;
      }

      $this->fail(sprintf(
        "Upload failed for %d file(s):\n%s\n\n" .
        "Common causes:\n" .
        "- File too large (check max_filesize in field settings)\n" .
        "- Wrong file type (check allowed extensions)\n" .
        "- CSRF token invalid (check Drupal.url() usage)\n" .
        "- Server error (check Drupal logs)",
        count($failed),
        implode("\n", $errorDetails)
      ));
    }

    $this->assertCount(
      $expectedCount,
      $successful,
      sprintf(
        'Expected %d successful uploads, got %d. Files: %s',
        $expectedCount,
        count($successful),
        json_encode($files, JSON_PRETTY_PRINT)
      )
    );
  }

  /**
   * Gets validation error messages from FilePond UI.
   *
   * FilePond displays validation errors (wrong type, too large) in the UI.
   * This extracts those messages for test diagnostics.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Array of error message strings.
   */
  protected function getFilePondValidationErrors(string $selector = '.filepond--root'): array {
    $result = $this->getSession()->evaluateScript(
      "return Array.from(document.querySelectorAll('$selector .filepond--file-status-sub')).map(el => el.textContent.trim()).filter(t => t);"
    );
    return is_array($result) ? $result : [];
  }

  /**
   * Gets the FilePond instance configuration for debugging.
   *
   * Useful to verify what validation settings are in effect.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Configuration array with validation settings.
   */
  protected function getFilePondValidationConfig(string $selector = '.filepond--root'): array {
    $result = $this->getSession()->evaluateScript(
      "return (function() {
        var pond = FilePond.find(document.querySelector('$selector'));
        if (!pond) return null;
        return {
          maxFileSize: pond.maxFileSize,
          maxTotalFileSize: pond.maxTotalFileSize,
          maxFiles: pond.maxFiles,
          acceptedFileTypes: pond.acceptedFileTypes,
          allowMultiple: pond.allowMultiple,
          chunkUploads: pond.chunkUploads,
          chunkSize: pond.chunkSize
        };
      })();"
    );
    return is_array($result) ? $result : [];
  }

  /**
   * Asserts that FilePond files have server IDs (upload succeeded).
   *
   * This checks that the server responded with valid file IDs. If this
   * fails but waitForFilePondComplete passes, the server returned an
   * error or unexpected response format.
   *
   * @param int $expectedCount
   *   Expected number of files with server IDs.
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function assertFilePondServerIds(int $expectedCount, string $selector = '.filepond--root'): void {
    $files = $this->getFilePondFileState($selector);
    $withServerIds = array_filter($files, fn($f) => !empty($f['serverId']));

    $this->assertCount(
      $expectedCount,
      $withServerIds,
      sprintf(
        'Expected %d files with serverId, found %d. File states: %s',
        $expectedCount,
        count($withServerIds),
        json_encode($files, JSON_PRETTY_PRINT)
      )
    );
  }

  /**
   * Checks if the hidden field element exists in the DOM.
   *
   * @param string $fieldName
   *   The field name (e.g., 'field_image').
   *
   * @return bool
   *   TRUE if the hidden field exists.
   */
  protected function hiddenFieldExists(string $fieldName): bool {
    $input = $this->getSession()->getPage()->find(
      'css',
      "input[name='{$fieldName}[0][fids]']"
    );
    return $input !== NULL;
  }

  /**
   * Gets the raw hidden field value for debugging.
   *
   * @param string $fieldName
   *   The field name (e.g., 'field_image').
   *
   * @return string|null
   *   The raw value, or NULL if field not found.
   */
  protected function getHiddenFieldRawValue(string $fieldName): ?string {
    $input = $this->getSession()->getPage()->find(
      'css',
      "input[name='{$fieldName}[0][fids]']"
    );
    return $input ? $input->getValue() : NULL;
  }

  /**
   * Gets diagnostic info about the hidden field from JavaScript.
   *
   * Checks what the JS sees - useful when PHP finds the element but
   * JS might be looking at a different selector.
   *
   * @param string $filepond_key
   *   The filepond key (data-filepond-fids value).
   *
   * @return array
   *   Array with 'exists', 'value', 'name' keys.
   */
  protected function getHiddenFieldJsState(string $filepond_key): array {
    $result = $this->getSession()->evaluateScript(
      "return (function() {
        var el = document.querySelector('input[data-filepond-fids=\"$filepond_key\"]');
        if (!el) return { exists: false };
        return { exists: true, value: el.value, name: el.name };
      })();"
    );
    return is_array($result) ? $result : ['exists' => FALSE];
  }

  /**
   * Gets the filepond key that links input, hidden field, and drupalSettings.
   *
   * The key is stored in multiple places for reliability:
   * - data-filepond-key on the wrapper element (survives FilePond init)
   * - data-filepond-fids on the hidden field
   * - data-filepond-id on the file input (lost after FilePond init)
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return string|null
   *   The filepond key, or NULL if not found.
   */
  protected function getFilePondKey(string $selector = '.filepond--root'): ?string {
    $result = $this->getSession()->evaluateScript(
      "return (function() {
        // Primary: get from wrapper's data-filepond-key attribute.
        var wrapper = document.querySelector('[data-filepond-key]');
        if (wrapper) return wrapper.getAttribute('data-filepond-key');

        // Fallback 1: get from hidden field's data-filepond-fids attribute.
        var hidden = document.querySelector('input[data-filepond-fids]');
        if (hidden) return hidden.getAttribute('data-filepond-fids');

        // Fallback 2: try to find original input before FilePond initializes.
        var input = document.querySelector('input[data-filepond-id]');
        if (input) return input.getAttribute('data-filepond-id');

        return null;
      })();"
    );
    return $result ?: NULL;
  }

  /**
   * Dumps full diagnostic state for debugging test failures.
   *
   * Call this when a test fails to get comprehensive state information.
   *
   * @param string $fieldName
   *   The field name (e.g., 'field_image').
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Diagnostic information array.
   */
  protected function getDiagnosticState(string $fieldName, string $selector = '.filepond--root'): array {
    $filepond_key = $this->getFilePondKey($selector);

    return [
      'filepond_key' => $filepond_key,
      'file_count' => $this->getFilePondFileCount($selector),
      'file_states' => $this->getFilePondFileState($selector),
      'hidden_field_exists_php' => $this->hiddenFieldExists($fieldName),
      'hidden_field_value_php' => $this->getHiddenFieldRawValue($fieldName),
      'hidden_field_js_state' => $filepond_key ? $this->getHiddenFieldJsState($filepond_key) : NULL,
      'transfer_id_mapping' => $this->hasTransferIdMapping(),
      'drupal_settings_config' => $filepond_key ? $this->getDrupalSettingsConfig($filepond_key) : NULL,
    ];
  }

  /**
   * Gets the FilePond config from drupalSettings for debugging.
   *
   * @param string $filepond_key
   *   The filepond key.
   *
   * @return array|null
   *   The config array, or NULL if not found.
   */
  protected function getDrupalSettingsConfig(string $filepond_key): ?array {
    $result = $this->getSession()->evaluateScript(
      "return drupalSettings?.filepond?.instances?.['$filepond_key'] || null;"
    );
    return $result;
  }

  /**
   * Asserts that FilePond is properly configured in drupalSettings.
   *
   * Call this after waitForFilePondInit() to verify JS has the config
   * needed for uploads. If this fails, there's a server-side rendering
   * or BigPipe delivery issue.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function assertFilePondConfigured(string $selector = '.filepond--root'): void {
    $filepond_key = $this->getFilePondKey($selector);
    $this->assertNotNull($filepond_key, 'FilePond input should have data-filepond-id attribute');

    $config = $this->getDrupalSettingsConfig($filepond_key);
    $this->assertNotNull(
      $config,
      "drupalSettings.filepond.instances['$filepond_key'] should exist"
    );

    $this->assertNotEmpty(
      $config['processUrl'] ?? NULL,
      'Config should have processUrl. Config: ' . json_encode($config)
    );
  }

  /**
   * Gets any JavaScript errors from the browser console.
   *
   * Note: This may not work with all WebDriver configurations.
   *
   * @return array
   *   Array of error messages.
   */
  protected function getJsErrors(): array {
    try {
      $logs = $this->getSession()->getDriver()->getWebDriverSession()->log('browser');
      return array_filter($logs, fn($log) => ($log['level'] ?? '') === 'SEVERE');
    }
    catch (\Exception $e) {
      // Log access may not be available in all configurations.
      return [];
    }
  }

  /**
   * Verifies the upload endpoint is accessible.
   *
   * Makes a synchronous XHR request to the processUrl to check if the route
   * exists and returns an expected response. Helps diagnose routing issues.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Array with 'accessible', 'status', 'response' keys.
   */
  protected function verifyUploadEndpoint(string $selector = '.filepond--root'): array {
    $filepond_key = $this->getFilePondKey($selector);
    if (!$filepond_key) {
      return ['accessible' => FALSE, 'error' => 'Could not find filepond key'];
    }

    $config = $this->getDrupalSettingsConfig($filepond_key);
    if (!$config || empty($config['processUrl'])) {
      return ['accessible' => FALSE, 'error' => 'No processUrl'];
    }

    // Make a synchronous XHR test request (without file - should get 400).
    $result = $this->getSession()->evaluateScript(
      "return (function() {
        try {
          var xhr = new XMLHttpRequest();
          xhr.open('POST', '{$config['processUrl']}', false);
          xhr.setRequestHeader('X-CSRF-Token', Drupal.filepond?.csrfToken || '');
          xhr.send(null);
          return {
            accessible: true,
            status: xhr.status,
            statusText: xhr.statusText,
            response: xhr.responseText.substring(0, 500)
          };
        } catch (e) {
          return { accessible: false, error: e.message };
        }
      })();"
    );

    return is_array($result) ? $result : ['accessible' => FALSE, 'error' => 'Unknown'];
  }

  /**
   * Gets the CSRF token status for FilePond.
   *
   * @return array
   *   Array with 'available', 'token' keys.
   */
  protected function getCsrfTokenStatus(): array {
    $result = $this->getSession()->evaluateScript(
      "return {
        available: !!Drupal.filepond?.csrfToken,
        token: Drupal.filepond?.csrfToken ? 'present' : 'missing',
        drupalFilepond: typeof Drupal.filepond
      };"
    );
    return is_array($result) ? $result : ['available' => FALSE, 'error' => 'Unknown'];
  }

  /**
   * Gets the status of the FilePond JavaScript objects.
   *
   * @param string $selector
   *   CSS selector for the FilePond root element.
   *
   * @return array
   *   Diagnostic info about JS state.
   */
  protected function getFilePondJsStatus(string $selector = '.filepond--root'): array {
    $result = $this->getSession()->evaluateScript(
      "return (function() {
        var root = document.querySelector('$selector');
        var pond = root ? FilePond.find(root) : null;
        return {
          filepond_global: typeof FilePond,
          drupal_filepond: typeof Drupal.filepond,
          root_element: !!root,
          pond_instance: !!pond,
          pond_server_config: pond ? JSON.stringify(pond.server || null) : null,
          drupal_settings_keys: Object.keys(drupalSettings.filepond?.instances || {})
        };
      })();"
    );
    return is_array($result) ? $result : ['error' => 'Could not get JS status'];
  }

  /**
   * Waits for the CSRF token to be fetched.
   *
   * The token is fetched asynchronously, so we need to wait for it.
   *
   * @param int $timeout
   *   Timeout in milliseconds.
   *
   * @return bool
   *   TRUE if token became available.
   */
  protected function waitForCsrfToken(int $timeout = 5000): bool {
    return (bool) $this->getSession()->wait(
      $timeout,
      "typeof Drupal.filepond !== 'undefined' && Drupal.filepond.csrfToken !== null"
    );
  }

  /**
   * Asserts that the CSRF token was fetched correctly.
   *
   * This is a critical check - if the CSRF token fetch fails (e.g., due to
   * base path issues), all uploads will fail silently. This test provides
   * a clear error message identifying the root cause.
   *
   * Common failure causes:
   * - Using hardcoded '/session/token' instead of Drupal.url('session/token')
   * - Base path misconfiguration
   * - Session/cookie issues
   */
  protected function assertCsrfTokenValid(): void {
    // First ensure token was fetched.
    $tokenReady = $this->waitForCsrfToken();
    $this->assertTrue(
      $tokenReady,
      'CSRF token fetch timed out. Check that Drupal.filepond is initialized.'
    );

    // Get the actual token value.
    $token = $this->getSession()->evaluateScript(
      'return Drupal.filepond.csrfToken;'
    );

    // Token should not be empty.
    $this->assertNotEmpty(
      $token,
      'CSRF token is empty. The /session/token endpoint may not be accessible.'
    );

    // Token should be a short alphanumeric string, not HTML.
    // Valid tokens are ~43 characters (base64-encoded hash).
    $this->assertLessThan(
      100,
      strlen($token),
      sprintf(
        'CSRF token is too long (%d chars). Got HTML instead of token? ' .
        'This usually means /session/token returned 404. ' .
        'Check that Drupal.url() is used instead of hardcoded path. ' .
        'Token starts with: %s',
        strlen($token),
        substr($token, 0, 100)
      )
    );

    // Token should not contain HTML.
    $this->assertStringNotContainsString(
      '<',
      $token,
      sprintf(
        'CSRF token contains HTML. The /session/token endpoint returned ' .
        'an error page instead of a token. This happens when using ' .
        'hardcoded "/session/token" path instead of Drupal.url("session/token") ' .
        'in subdirectory installations. Token: %s',
        substr($token, 0, 200)
      )
    );

    // Token should not be an error message.
    // Note: We check for error phrases, not just "404" because a valid
    // base64-like token could randomly contain "404" in its characters.
    $this->assertStringNotContainsString(
      'Not Found',
      $token,
      sprintf(
        'CSRF token contains "Not Found". The session/token endpoint returned ' .
        'a 404 error. Ensure Drupal.url() is used for the fetch path. Token: %s',
        substr($token, 0, 200)
      )
    );
    // Verify it looks like a valid token (43 char alphanumeric with - and _).
    $this->assertMatchesRegularExpression(
      '/^[a-zA-Z0-9_-]{43}$/',
      $token,
      sprintf(
        'CSRF token does not match expected format (43 alphanumeric chars). ' .
        'The session/token endpoint may have returned an error. Token: %s',
        substr($token, 0, 200)
      )
    );
  }

  /**
   * Asserts that chunked upload is configured with the expected chunk size.
   *
   * Use this in chunked upload tests to verify the config change in setUp()
   * actually reached the JavaScript. If this fails, the FilePond instance
   * may be using default settings instead of the test configuration.
   *
   * @param int $expectedBytes
   *   Expected chunk size in bytes (e.g., 1048576 for 1MB).
   * @param string $selector
   *   CSS selector for the FilePond root element.
   */
  protected function assertChunkSizeConfigured(int $expectedBytes, string $selector = '.filepond--root'): void {
    $config = $this->getFilePondValidationConfig($selector);

    // First verify chunked uploads are enabled.
    $this->assertTrue(
      $config['chunkUploads'] ?? FALSE,
      sprintf(
        'Chunked uploads should be enabled. Config: %s',
        json_encode($config)
      )
    );

    // Then verify the chunk size.
    $actualSize = $config['chunkSize'] ?? 0;
    $this->assertEquals(
      $expectedBytes,
      $actualSize,
      sprintf(
        'Expected chunkSize of %d bytes (%s MB), got %d bytes (%s MB). ' .
        'This may indicate the config change in setUp() did not reach JS. ' .
        'Full config: %s',
        $expectedBytes,
        number_format($expectedBytes / 1024 / 1024, 2),
        $actualSize,
        number_format($actualSize / 1024 / 1024, 2),
        json_encode($config)
      )
    );
  }

}
