# FilePond FunctionalJavascript Tests

These tests run in a real browser (via Selenium/ChromeDriver) to verify that FilePond's JavaScript functionality works correctly end-to-end.

## Why These Tests Exist

These tests were created after a CDN version mismatch bug (4.31.4 vs 4.32.10) broke chunked uploads without being detected. The Kernel tests only test backend PHP logic - they can't catch JavaScript issues.

## Test Files

### FilePondTestBase.php
**Base class with shared helper methods.**

| Method | Purpose |
|--------|---------|
| `createTestFile($name)` | Creates small test PNG (1x1 pixel) |
| `createMediumTestFile($name)` | Creates 500KB file for longer uploads |
| `createLargeTestFile($name, $sizeMB)` | Creates large file to force chunked uploads |
| `dropFileToFilePond($path, $selector)` | Simulates file drop via FilePond API |
| `waitForFilePondInit($timeout)` | Waits for FilePond JS to initialize |
| `waitForFilePondComplete($selector, $timeout)` | Waits for `processing-complete` state |
| `waitForFilePondFileCount($count)` | Waits for specific file count |
| `waitForFilePondEmpty($selector)` | Waits for all files removed |
| `waitForCsrfToken($timeout)` | Waits for async CSRF token fetch |
| `getFilePondFileCount($selector)` | Returns current file count via JS |
| `getFilePondFileIds($fieldName)` | Gets IDs from standalone FilePond element |
| `getFieldWidgetFileIds($fieldName)` | Gets IDs from field widget (different naming) |
| `waitForFieldWidgetFileIds($field, $count)` | Waits for expected file count in hidden field |
| `extractFileId($signedToken)` | Extracts numeric FID from signed token |
| `hasTransferIdMapping()` | Checks if chunked upload mapping exists |
| `getFilePondFileState($selector)` | Gets detailed file status, serverId, origin |
| `assertFilePondServerIds($count)` | Asserts files have server IDs |
| `assertFilePondConfigured($selector)` | Asserts drupalSettings has config |
| `getCsrfTokenStatus()` | Returns CSRF token availability info |
| `getFilePondJsStatus($selector)` | Returns FilePond JS object status |
| `verifyUploadEndpoint($selector)` | Tests upload endpoint (sync XHR - don't use in tests, crashes WebDriver) |
| `getDiagnosticState($fieldName)` | Full diagnostic dump for debugging |
| `assertCsrfTokenValid()` | **Validates CSRF token** - catches base path issues with clear error |
| `assertFilesUploadedSuccessfully($count)` | **Validates uploads** - shows file type, size, server errors on failure |
| `getFilePondFileErrors($selector)` | Gets detailed file state with error messages |
| `getFilePondValidationErrors($selector)` | Gets validation error text from UI |
| `getFilePondValidationConfig($selector)` | Gets maxFileSize, acceptedFileTypes, etc. |

### FilePondImageFieldTest.php
**Tests the `filepond_image` widget on entity image fields.**

| Test | What It Verifies |
|------|-----------------|
| `testRegularImageUpload` | Single file upload: UI shows file, hidden field has ID, form submits, node has correct image |
| `testChunkedImageUpload` | Large file chunked upload: transfer ID mapping works, hidden field has ID, node saves correctly |
| `testMultipleImageUpload` | Two files uploaded: both show in UI, hidden field has both IDs, node saves with both images |
| `testImageRemoval` | Upload then remove: hidden field clears, node saves with no images |
| `testImageReplacement` | Upload, remove, upload different: hidden field changes, node has only replacement |
| `testSimultaneousUploadsHiddenFieldSync` | 3 files dropped at once: regression test for hidden field race condition |
| `testSimultaneousLargeUploadsHiddenFieldSync` | Same as above with larger (500KB) files |
| `testEditNodeAddImages` | Edit existing node, add new image: existing + new both saved |
| `testEditNodeRemoveImage` | Edit node with 2 images, remove one: only 1 remains after save |
| `testSubmitButtonStatesDuringUpload` | Submit button disabled during upload, enabled after |
| `testSubmitButtonDisabledUntilAllUploadsComplete` | Multiple uploads: button enabled only after all complete |

### FilePondChunkedUploadTest.php
**Tests the chunked upload protocol (POST transfer ID, PATCH chunks, file ID response).**

Uses the built-in test form at `/admin/config/media/filepond/test`.

| Test | What It Verifies |
|------|-----------------|
| `testChunkedUploadComplete` | Large file uploads via chunks, file ID captured from PATCH response |
| `testTransferIdMapping` | JavaScript `transferToFileId` map populated after chunked upload |
| `testChunkedUploadUi` | Chunked file displays correctly in FilePond UI |
| `testChunkedUploadRemoval` | Remove chunked file before submit, form shows no files |

### FilePondMediaLibraryTest.php
**Tests FilePond integration with Drupal's Media Library modal.**

Requires `enable_media_library_widget` config to be TRUE.

| Test | What It Verifies |
|------|-----------------|
| `testFilePondInMediaLibrary` | FilePond appears in Media Library modal instead of default upload |
| `testMediaLibraryUpload` | File uploads via FilePond in modal, auto-select triggers |
| `testMediaLibrarySaveAndInsert` | Full flow: upload in modal, save, media appears in field |
| `testMediaLibraryFileRemoval` | Remove file in modal before saving |

### FilePondEntityBrowserTest.php
**Tests FilePond `filepond_media` widget in Entity Browser iframe.**

| Test | What It Verifies |
|------|-----------------|
| `testEntityBrowserOpensWithFilePond` | EB iframe opens with FilePond widget initialized |
| `testEntityBrowserUpload` | File uploads via FilePond inside EB iframe |
| `testEntityBrowserSelectMedia` | Upload, click select, media appears in node field |
| `testEntityBrowserMultipleUploads` | Two files uploaded in EB display correctly |
| `testEntityBrowserFileRemoval` | Remove file in EB before selection |
| `testEntityBrowserChunkedUpload` | Chunked upload works in EB iframe context |

## Key Assertions Pattern

Every test that involves form submission should verify the **full round-trip**:

1. **UI State**: File shows in FilePond (`getFilePondFileCount()`)
2. **Hidden Field**: File ID captured before submit (`getFieldWidgetFileIds()`)
3. **Form Submit**: Page shows success message
4. **Entity State**: Loaded entity has correct files attached

```php
// UI shows file
$this->assertEquals(1, $this->getFilePondFileCount());

// Hidden field has ID
$fileIds = $this->getFieldWidgetFileIds('field_image');
$this->assertCount(1, $fileIds);
$submittedFid = $this->extractFileId($fileIds[0]);

// Submit form
$this->getSession()->getPage()->pressButton('Save');
$this->assertSession()->waitForElement('css', '.messages');

// Entity has correct file
$node = $this->getNodeByTitle('Test Node');
$savedFid = $node->get('field_image')->target_id;
$this->assertEquals($submittedFid, $savedFid);
```

## Test Gaps (Known Missing Coverage)

- [ ] File type validation (reject non-images)
- [ ] File size limit enforcement
- [ ] Max files cardinality limit
- [ ] Reorder functionality (drag and drop order change)
- [ ] Edit form with different user (permission testing)
- [ ] Error handling (server errors, network failures)
- [ ] Alt/title field population for images

## Running Tests

```bash
# All FilePond JS tests
./vendor/bin/phpunit web/modules/custom/filepond/tests/src/FunctionalJavascript/

# Specific test file
./vendor/bin/phpunit web/modules/custom/filepond/tests/src/FunctionalJavascript/FilePondImageFieldTest.php

# Specific test method
./vendor/bin/phpunit --filter testRegularImageUpload web/modules/custom/filepond/tests/src/FunctionalJavascript/FilePondImageFieldTest.php
```

## Drupal.org CI Configuration

The `.gitlab-ci.yml` controls which tests run on drupal.org CI using PHPUnit groups:

```yaml
variables:
  # Run core tests (4 tests, ~2 min)
  _PHPUNIT_EXTRA: '--group filepond_core'

  # Run multiple groups
  # _PHPUNIT_EXTRA: '--group filepond_core --group filepond_chunked'

  # Run all tests (remove this line entirely)
```

### Test Groups

Tests are organized into **type groups** (by integration) and **functional groups** (by feature).

#### Type Groups (by integration)

| Group | Test Class | Description |
|-------|------------|-------------|
| `filepond_core` | FilePondRegularUploadTest, FilePondChunkedUploadTest, FilePondMaxFilesTest | Built-in test form (standalone element) |
| `filepond_image_field` | FilePondImageFieldTest | Image field widget on node forms |
| `filepond_media_library` | FilePondMediaLibraryTest | Media Library modal integration |
| `filepond_entity_browser` | FilePondEntityBrowserTest | Entity Browser iframe integration |

#### Functional Groups (by feature)

| Group | Tests | Description |
|-------|-------|-------------|
| `filepond_chunked` | 5 | Large file chunked uploads (across test form and image field) |
| `filepond_maxfiles` | 2 | Max files cardinality enforcement |
| `filepond_parallel` | 2 | Simultaneous upload race condition tests |
| `filepond_edit` | 2 | Edit existing node tests |
| `filepond_ui` | 2 | Submit button state tests |

### Tests by Group

**filepond_core** (built-in test form):
- FilePondRegularUploadTest: `testSingleFileUpload`, `testMultipleFileUpload`, `testFileRemoval`, `testFilePondItemCount`
- FilePondChunkedUploadTest: `testChunkedUploadComplete`, `testTransferIdMapping`, `testChunkedUploadUi`, `testChunkedUploadRemoval`
- FilePondMaxFilesTest: `testMaxFilesEnforced`, `testRemovalAllowsNewUpload`
- FilePondImageFieldTest (method-level): `testRegularImageUpload`, `testMultipleImageUpload`, `testImageRemoval`, `testImageReplacement`

**filepond_image_field** (image field widget):
- `testRegularImageUpload` - Basic upload, CSRF, hidden field, node save
- `testChunkedImageUpload` - Large file chunked upload
- `testMultipleImageUpload` - 2 files uploaded sequentially
- `testImageRemoval` - Upload then remove
- `testImageReplacement` - Remove and add different file
- `testSimultaneousUploadsHiddenFieldSync` - 3 files dropped at once
- `testSimultaneousLargeUploadsHiddenFieldSync` - Same with 500KB files
- `testEditNodeAddImages` - Edit existing node, add images
- `testEditNodeRemoveImage` - Edit node, remove one
- `testSubmitButtonStatesDuringUpload` - Button disabled during upload
- `testSubmitButtonDisabledUntilAllUploadsComplete` - Button enabled after all done

**filepond_media_library** (Media Library modal):
- `testFilePondInMediaLibrary` - FilePond appears in modal
- `testMediaLibraryUpload` - Upload via FilePond in modal
- `testMediaLibrarySaveAndInsert` - Full save and insert flow
- `testMediaLibraryFileRemoval` - Remove before saving

**filepond_entity_browser** (Entity Browser iframe):
- `testEntityBrowserOpensWithFilePond` - EB opens with FilePond
- `testEntityBrowserUpload` - Upload in EB iframe
- `testEntityBrowserSelectMedia` - Select uploaded media
- `testEntityBrowserMultipleUploads` - Multiple files in EB
- `testEntityBrowserFileRemoval` - Remove before selection
- `testEntityBrowserChunkedUpload` - Chunked in EB context

**filepond_chunked** (large file uploads):
- FilePondChunkedUploadTest: all 4 tests
- FilePondImageFieldTest: `testChunkedImageUpload`

**filepond_parallel** (race condition tests):
- `testSimultaneousUploadsHiddenFieldSync`
- `testSimultaneousLargeUploadsHiddenFieldSync`

**filepond_edit** (edit node tests):
- `testEditNodeAddImages`
- `testEditNodeRemoveImage`

**filepond_ui** (UI state tests):
- `testSubmitButtonStatesDuringUpload`
- `testSubmitButtonDisabledUntilAllUploadsComplete`

### Running Groups Locally

```bash
# Run all FilePond tests
./vendor/bin/phpunit --group filepond

# Run by integration type
./vendor/bin/phpunit --group filepond_core           # Test form only
./vendor/bin/phpunit --group filepond_image_field    # Image field widget
./vendor/bin/phpunit --group filepond_media_library  # Media Library
./vendor/bin/phpunit --group filepond_entity_browser # Entity Browser

# Run by feature
./vendor/bin/phpunit --group filepond_chunked        # Chunked uploads
./vendor/bin/phpunit --group filepond_maxfiles       # Cardinality limits

# Combine groups
./vendor/bin/phpunit --group filepond_core --group filepond_image_field
```

## Requirements

- Chrome or Chromium browser
- ChromeDriver matching browser version
- `SIMPLETEST_BASE_URL` environment variable set
- Database configured in `phpunit.xml`

## Debugging Failed Tests

1. **Screenshots**: Tests automatically save screenshots on failure to `sites/simpletest/browser_output/`

2. **Verbose output**: Add `--debug` flag to see what's happening

3. **Check hidden field**: The most common failure is hidden field not capturing IDs. Add debug logging:
   ```php
   $value = $this->getSession()->getPage()->find('css', "input[name='field_image[0][fids]']")->getValue();
   error_log("Hidden field value: $value");
   ```

4. **Check JS console**: Use browser dev tools to watch for JavaScript errors

5. **Timing issues**: If tests pass sometimes and fail others, add more wait time or check for race conditions

## Drupal 11 CI Issue (RESOLVED)

### The Problem

Tests passed locally but failed on Drupal.org CI with uploads stuck at status 3.

### Root Cause

The CSRF token fetch used a hardcoded path that broke in subdirectory installs:
```javascript
// BROKEN - doesn't work when Drupal is in /web/ subdirectory
fetch('/session/token')

// FIXED - respects base path
fetch(Drupal.url('session/token'))
```

### Prevention

Tests now include assertions that catch this issue immediately:

```php
// Validates token is not HTML/404 - catches base path issues
$this->assertCsrfTokenValid();

// Shows file type, size, server error on failure
$this->assertFilesUploadedSuccessfully(1);
```

### Lessons Learned

1. **Always use `Drupal.url()`** for Drupal endpoints in JavaScript
2. **Validate token format**, not just presence (HTML != valid token)
3. **Include error context** in failure messages (file size, type, server response)
4. **CI runs in subdirectory** - local root installs won't catch path issues

## Chunked Upload CI Issue (RESOLVED - Dec 2025)

### The Problem

Chunked upload tests passed locally but failed on Drupal.org CI (headless Chrome) with:
- `serverId: null`
- `status: 6` (PROCESSING_ERROR)

Apache logs showed the upload actually completes, but then extra PATCH requests arrive:
```
PATCH → 204 (chunk 0 OK)
PATCH → 200 (chunk 1 - COMPLETE, returns file ID)
PATCH → 400 (extra request - temp file already deleted)
PATCH → 400 (retry)
PATCH → 400 (retry)
```

The 200 response indicates success, but FilePond sends duplicate PATCH requests which fail because the temp file was deleted after finalization. FilePond sees these 400 errors and marks the upload as failed.

### Root Cause

Unknown. The duplicate PATCH requests only occur in headless Chrome CI environment, not locally. Possibly related to:
- Event loop timing differences in headless mode
- Network layer behavior
- FilePond's internal retry logic

### The Fix

Two fixes were applied:

**1. JS Fix (`filepond.element.js`)** - Return plain fileId instead of JSON:
```javascript
onload: (xhr) => {
  // ... parse JSON, store mappings ...
  return data.fileId;  // FilePond expects plain string, not JSON
},
```

**2. Server-side Cache (`FilePondUploadHandler.php`)** - Tolerate late PATCH requests:

The server now caches completed transfers for 5 minutes. When a late PATCH request arrives for an already-completed transfer, it returns success with the cached file ID instead of a 400 error. This makes the endpoint idempotent.

```php
// In processChunk(): Check cache before validation
if ($cachedResult = $this->getCompletedTransfer($transferId)) {
  return new JsonResponse($cachedResult);
}

// In finalizeUpload(): Cache the result
$this->cacheCompletedTransfer($transferId, $fileId, $mediaId);
```

### Test Fixes

Additionally, test helpers were updated to handle existing files correctly:

**`FilePondTestBase::getCompletedFileCount()`** - Now counts both states:
- `processing-complete` for newly uploaded files
- `idle` for existing files loaded from server

**`FilePondImageFieldTest::testEditNodeAddImages()`** - Uses `waitForFieldWidgetFileIds()` instead of `getFieldWidgetFileIds()` when checking existing files, since the hidden field is populated asynchronously.

### Status

- [x] JS fix implemented (return `data.fileId` instead of JSON)
- [x] Server-side cache enabled (tolerates late PATCH requests)
- [x] Test helpers updated for existing file states
- [x] All 21 tests pass on Drupal.org CI

### Known Limitations

The server-side cache solution is a workaround, not a root cause fix. The cache:
- Uses Drupal's cache API with 5-minute expiry
- Adds slight overhead to chunked uploads
- May mask other issues if the duplicate requests have a different cause

**TODO**: Investigate why headless Chrome sends duplicate PATCH requests. The cache works but understanding the root cause would be better.

### Files Modified

- `js/filepond.element.js` - `patch.onload` returns plain fileId
- `src/FilePondUploadHandler.php` - Server-side cache for completed transfers
- `tests/src/FunctionalJavascript/FilePondTestBase.php` - Handle `idle` state for existing files
- `tests/src/FunctionalJavascript/FilePondImageFieldTest.php` - Use wait helper for existing files
