<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests FilePond configuration.
 *
 * @group filepond
 * @group filepond_core
 */
class FilePondSettingsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'user',
    'filepond',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['filepond']);
  }

  /**
   * Tests that default configuration is installed.
   */
  public function testDefaultConfigInstalled(): void {
    $config = $this->config('filepond.settings');

    // Check top-level settings.
    $this->assertFalse($config->get('enable_media_library_widget'));

    // Check defaults.
    $defaults = $config->get('defaults');
    $this->assertIsArray($defaults);
    $this->assertEquals(10, $defaults['max_files']);
    $this->assertEquals('jpg jpeg gif png webp', $defaults['extensions']);
    $this->assertEquals(5, $defaults['chunk_size']);
    $this->assertEquals(3, $defaults['max_parallel_uploads']);
    $this->assertNotEmpty($defaults['upload_location']);
    $this->assertNotEmpty($defaults['upload_prompt']);
    $this->assertEquals('1:1', $defaults['item_panel_aspect_ratio']);
  }

  /**
   * Tests configuration schema compliance.
   */
  public function testConfigSchemaCompliance(): void {
    // This will throw an exception if schema is invalid.
    $config = $this->config('filepond.settings');

    // Verify we can set and get all expected values.
    $config->set('enable_media_library_widget', TRUE);
    $config->set('defaults.max_files', 20);
    $config->set('defaults.extensions', 'jpg png');
    $config->set('defaults.max_filesize', '50M');
    $config->set('defaults.chunk_size', 10);
    $config->set('defaults.upload_location', 'public://custom-path');
    $config->set('defaults.upload_prompt', 'Custom prompt');
    $config->set('defaults.max_parallel_uploads', 5);
    $config->set('defaults.item_panel_aspect_ratio', '16:9');
    $config->save();

    // Re-load and verify.
    $config = $this->config('filepond.settings');
    $this->assertTrue($config->get('enable_media_library_widget'));
    $this->assertEquals(20, $config->get('defaults.max_files'));
    $this->assertEquals('16:9', $config->get('defaults.item_panel_aspect_ratio'));
  }

  /**
   * Tests that element info includes filepond type.
   */
  public function testElementInfoRegistered(): void {
    /** @var \Drupal\Core\Render\ElementInfoManagerInterface $element_info */
    $element_info = $this->container->get('plugin.manager.element_info');

    $info = $element_info->getInfo('filepond');

    $this->assertNotEmpty($info);
    $this->assertArrayHasKey('#extensions', $info);
    $this->assertArrayHasKey('#max_filesize', $info);
    $this->assertArrayHasKey('#max_files', $info);
    $this->assertArrayHasKey('#upload_location', $info);
    $this->assertArrayHasKey('#config', $info);
  }

}
