<?php

declare(strict_types=1);

namespace Drupal\Tests\filepond\Kernel;

use Drupal\Core\File\FileSystemInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\media\Entity\MediaType;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;
use Drupal\user\RoleInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests media create access checks on FilePond upload routes.
 *
 * @group filepond
 * @group filepond_core
 */
class MediaAccessTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'file',
    'field',
    'image',
    'user',
    'media',
    'filepond',
  ];

  /**
   * Temp directory for test files.
   */
  protected string $tempDir;

  /**
   * The current test user.
   */
  protected User $testUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installConfig(['filepond', 'media']);
    $this->installSchema('file', ['file_usage']);

    // Set up temp directory.
    $this->tempDir = $this->siteDirectory . '/files';
    \Drupal::service('file_system')->prepareDirectory(
      $this->tempDir,
      FileSystemInterface::CREATE_DIRECTORY
    );
    $this->setSetting('file_temp_path', $this->tempDir);

    // Create public files directory.
    $public_dir = 'public://filepond-test';
    \Drupal::service('file_system')->prepareDirectory(
      $public_dir,
      FileSystemInterface::CREATE_DIRECTORY
    );

    // Create a test media type.
    $this->createTestMediaType();

    // Create role with filepond permission but NOT media create.
    $role = Role::create([
      'id' => RoleInterface::AUTHENTICATED_ID,
      'label' => 'Authenticated',
    ]);
    $role->grantPermission('filepond upload files');
    $role->save();

    // Create user 1 first (admin) - we won't use this.
    $admin = User::create([
      'name' => 'admin',
      'status' => 1,
    ]);
    $admin->save();

    // Create and set current user (will be uid=2, not admin).
    $this->testUser = User::create([
      'name' => 'test_user',
      'status' => 1,
    ]);
    $this->testUser->save();
    $this->container->get('current_user')->setAccount($this->testUser);
  }

  /**
   * Creates a test media type for image uploads.
   */
  protected function createTestMediaType(): void {
    // Create a simple image media type.
    $media_type = MediaType::create([
      'id' => 'test_image',
      'label' => 'Test Image',
      'source' => 'image',
      'source_configuration' => [
        'source_field' => 'field_media_image',
      ],
    ]);
    $media_type->save();

    // Create the source field.
    $field_storage = \Drupal::entityTypeManager()
      ->getStorage('field_storage_config')
      ->create([
        'field_name' => 'field_media_image',
        'entity_type' => 'media',
        'type' => 'image',
      ]);
    $field_storage->save();

    $field = \Drupal::entityTypeManager()
      ->getStorage('field_config')
      ->create([
        'field_storage' => $field_storage,
        'bundle' => 'test_image',
        'label' => 'Image',
        'settings' => [
          'file_directory' => 'filepond-test',
          'file_extensions' => 'jpg jpeg png gif',
          'max_filesize' => '10 MB',
        ],
      ]);
    $field->save();
  }

  /**
   * Tests that media route returns 403 when user lacks media create access.
   */
  public function testMediaRouteRequiresCreateAccess(): void {
    $this->container->get('router.builder')->rebuild();

    // User has filepond permission but NOT media create permission.
    // Try to upload via media route.
    $request = Request::create('/filepond/media/test_image/process', 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(403, $response->getStatusCode());
    $this->assertStringContainsString('permission', $response->getContent());
  }

  /**
   * Tests that media route allows upload when user has media create access.
   */
  public function testMediaRouteAllowsWithCreateAccess(): void {
    $this->container->get('router.builder')->rebuild();

    // Grant media create permission.
    $role = Role::load(RoleInterface::AUTHENTICATED_ID);
    $role->grantPermission('create media');
    $role->save();

    // Try to upload via media route.
    $request = Request::create('/filepond/media/test_image/process', 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    // Should succeed (200) with transfer ID.
    $this->assertEquals(200, $response->getStatusCode());
    $this->assertNotEmpty($response->getContent());
  }

  /**
   * Tests that media route returns 404 for non-existent media type.
   */
  public function testMediaRouteReturns404ForInvalidType(): void {
    $this->container->get('router.builder')->rebuild();

    // Grant media create permission.
    $role = Role::load(RoleInterface::AUTHENTICATED_ID);
    $role->grantPermission('create media');
    $role->save();

    // Try to upload to non-existent media type.
    $request = Request::create('/filepond/media/nonexistent_type/process', 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(404, $response->getStatusCode());
  }

  /**
   * Tests that PATCH route also checks media create access.
   */
  public function testMediaPatchRouteRequiresCreateAccess(): void {
    $this->container->get('router.builder')->rebuild();

    // User has filepond permission but NOT media create permission.
    // Try to send chunk via media PATCH route.
    $request = Request::create(
      '/filepond/media/test_image/patch/some-transfer-id',
      'PATCH',
      [],
      [],
      [],
      ['CONTENT_TYPE' => 'application/offset+octet-stream'],
      'chunk-data'
    );
    $request->headers->set('Upload-Offset', '0');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);

    $this->assertEquals(403, $response->getStatusCode());
  }

  /**
   * Tests per-bundle media create access.
   */
  public function testMediaRouteRespectsPerBundleAccess(): void {
    $this->container->get('router.builder')->rebuild();

    // Create a second media type.
    $media_type = MediaType::create([
      'id' => 'restricted_image',
      'label' => 'Restricted Image',
      'source' => 'image',
      'source_configuration' => [
        'source_field' => 'field_media_image_2',
      ],
    ]);
    $media_type->save();

    // Create the source field for restricted type.
    $field_storage = \Drupal::entityTypeManager()
      ->getStorage('field_storage_config')
      ->create([
        'field_name' => 'field_media_image_2',
        'entity_type' => 'media',
        'type' => 'image',
      ]);
    $field_storage->save();

    $field = \Drupal::entityTypeManager()
      ->getStorage('field_config')
      ->create([
        'field_storage' => $field_storage,
        'bundle' => 'restricted_image',
        'label' => 'Image',
        'settings' => [
          'file_directory' => 'restricted',
          'file_extensions' => 'jpg jpeg png gif',
        ],
      ]);
    $field->save();

    // Grant permission to create test_image but NOT restricted_image.
    $role = Role::load(RoleInterface::AUTHENTICATED_ID);
    $role->grantPermission('create test_image media');
    $role->save();

    // Upload to test_image should succeed.
    $request = Request::create('/filepond/media/test_image/process', 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'test-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    /** @var \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel */
    $http_kernel = $this->container->get('http_kernel');
    $response = $http_kernel->handle($request);
    $this->assertEquals(200, $response->getStatusCode(), 'Should allow test_image upload');

    // Upload to restricted_image should fail.
    $request = Request::create('/filepond/media/restricted_image/process', 'POST');
    $request->headers->set('Upload-Length', '1024');
    $request->headers->set('Upload-Name', 'restricted-image.jpg');
    $request->headers->set('X-CSRF-Token', 'test');

    $response = $http_kernel->handle($request);
    $this->assertEquals(403, $response->getStatusCode(), 'Should deny restricted_image upload');
  }

}
