<?php

namespace Drupal\fillpdf_media\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceLabelFormatter;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\file\Plugin\Field\FieldFormatter\FileFormatterBase;
use Drupal\file\FileInterface;

/*
 * Plugin implementation of the 'fillpdf_file_url' formatter.
 *
 * Based on the 'file_url_plain' formatter.
 */
#[FieldFormatter(
  id: 'fillpdf_media',
  label: new TranslatableMarkup('fillPDF for media'),
  field_types: [
    'entity_reference',
  ],
)]
class FillPdfMediaFormatter extends EntityReferenceLabelFormatter {

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $output_as_link = $this->getSetting('link');

    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $entity) {

      $elements[$delta] = [
        '#entity' => $entity,
      ];
      $label = $entity->label();
      $cacheability = CacheableMetadata::createFromObject($entity);

      // If the link is to be displayed and the entity has a uri, display a
      // link.
      if ($output_as_link && !$entity->isNew()) {
        // If this is a media type that has a FillPDF form, prefer that.
        $uri = NULL;
        // @TODO do not hard code 'pdf'.
        if (isset($entity) && $entity->bundle() === 'pdf') {
          // Not sure there's a practical way to avoid hardcoding the field.
          $file_id = $entity->field_media_file->target_id;
          if ($file_id) {
            $fillpdf_admin_form_helper = \Drupal::service('fillpdf.admin_form_helper');
            $form_id = $fillpdf_admin_form_helper->getFormIdByFileId($file_id);
            if ($form_id) {
              $uri = Url::fromRoute('entity.fillpdf_form.canonical', ['fillpdf_form' => $form_id]);
            }
          }
        }

        if (!$uri) {
          try {
            $uri = $entity->toUrl();
          } catch (UndefinedLinkTemplateException) {
            // This exception is thrown by
            // \Drupal\Core\Entity\EntityInterface::toUrl() and it means that the
            // entity type doesn't have a link template nor a valid
            // "uri_callback", so don't bother trying to output a link for the
            // rest of the referenced entities.
            $elements[$delta]['#plain_text'] = $label;
            $cacheability->applyTo($elements[$delta]);
            continue;
          }
        }
        $uri_access = $uri->access(return_as_object: TRUE);
        $cacheability->addCacheableDependency($uri_access);
        if ($uri_access->isAllowed()) {
          $elements[$delta] += [
            '#type' => 'link',
            '#title' => $label,
            '#url' => $uri,
            '#options' => $uri->getOptions(),
          ];
          if (!empty($items[$delta]->_attributes)) {
            $elements[$delta]['#options'] += [
              'attributes' => [],
            ];
            $elements[$delta]['#options']['attributes'] += $items[$delta]->_attributes;
            // Unset field item attributes since they have been included in the
            // formatter output and shouldn't be rendered in the field template.
            unset($items[$delta]->_attributes);
          }
        }
        else {
          $elements[$delta]['#plain_text'] = $label;
        }
      }
      else {
        $elements[$delta]['#plain_text'] = $label;
      }
      $cacheability->applyTo($elements[$delta]);
    }
    return $elements;
  }

  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return ($field_definition->getFieldStorageDefinition()->getSetting('target_type') == 'media');
  }

}
