<?php

namespace Drupal\find_text\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\find_text\TextSearchService;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Provides a search form for finding text across entities.
 */
class SearchForm extends FormBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cacheManager;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected CacheTagsInvalidatorInterface $cacheTagsInvalidator;

  /**
   * The text search service.
   *
   * @var \Drupal\find_text\TextSearchService
   */
  protected TextSearchService $textSearchService;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a SearchForm object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_manager
   *   The cache backend.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   The cache tags invalidator.
   * @param \Drupal\find_text\TextSearchService $text_search_service
   *   The text search service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(RendererInterface $renderer, EntityTypeManagerInterface $entity_type_manager, CacheBackendInterface $cache_manager, CacheTagsInvalidatorInterface $cache_tags_invalidator, TextSearchService $text_search_service, ModuleHandlerInterface $module_handler, LanguageManagerInterface $language_manager, ConfigFactoryInterface $config_factory) {
    $this->renderer = $renderer;
    $this->entityTypeManager = $entity_type_manager;
    $this->cacheManager = $cache_manager;
    $this->cacheTagsInvalidator = $cache_tags_invalidator;
    $this->textSearchService = $text_search_service;
    $this->moduleHandler = $module_handler;
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('entity_type.manager'),
      $container->get('cache.data'),
      $container->get('cache_tags.invalidator'),
      $container->get('find_text.search'),
      $container->get('module_handler'),
      $container->get('language_manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'find_text_search_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $request = $this->getRequest();
    $query_search_text = $request->query->get('text') ? urldecode($request->query->get('text')) : '';
    $query_render_as_markup = $request->query->get('markup') === 'true';
    $query_regex = $request->query->get('regex') === 'true';

    $wrapper_id = $this->getFormId() . '-wrapper';
    $results_markup = $form_state->get('results_markup') ?? '';
    $has_results = (bool) $form_state->get('results');
    $settings = $this->configFactory->get('find_text.settings');

    $form = [
      '#prefix' => '<div id="' . $wrapper_id . '" aria-live="polite">',
      '#suffix' => '</div>',
      '#attached' => ['library' => ['find_text/results']],
    ];

    $form['intro'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<p>Search text fields for a provided string. The search is not case-sensitive.</p><p>Pre-rendered text area markup is searched, so some characters may be different; for instance, <code>&amp;amp;</code> may match ampersands where <code>&amp;</code> will not. Node fields and content blocks are included, but some areas such as menu links will not be searched.</p>'),
    ];

    $form['needle'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Text'),
      '#default_value' => $query_search_text,
      '#description' => $this->t('The string to search against. Wildcards % will match any number of characters and _ will match any single character. % wildcards are prepended and appended automatically when not using regex.'),
      '#required' => TRUE,
    ];

    $form['render'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Render markup'),
      '#description' => $this->t('Display the results as rendered HTML. This may hide parts of the results, such as text matched within HTML tags.'),
      '#default_value' => $query_render_as_markup ? 1 : 0,
    ];

    $form['regexed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use REGEXP'),
      '#description' => $this->t('% wildcards will not be prepended or appended, and the search will be performed as a REGEXP search rather than LIKE. Use only if a full regular expression is required.'),
      '#default_value' => $query_regex ? 1 : 0,
    ];

    if ($settings->get('enable_search_results_cache')) {
      $form['invalidate_cache'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Invalidate cache'),
        '#description' => $this->t('If checked, the cache for search results will be cleared and the search will be performed again.'),
        '#default_value' => 0,
      ];
    }

    $form['filter_container'] = [
      '#type' => 'container',
      '#attributes' => ['style' => 'display: flex; gap: 1.5rem; flex-wrap: wrap;'],
    ];

    // Language filter.
    if ($this->moduleHandler->moduleExists('language')) {
      $form['filter_container']['language_selection'] = $this->buildLanguageFilter();
    }
    // Workflow filter.
    $has_workflows = $this->moduleHandler->moduleExists('workflows');
    if ($has_workflows) {
      $form['filter_container']['workflow_selection'] = $this->buildWorkflowFilter();
    }

    // Published status filter.
    // Disable if workflow filter is being used (not set to 'All').
    $workflow_value = $form_state->getValue('workflow_selection') ?? 'All';
    $is_workflow_active = $has_workflows && $workflow_value !== 'All';

    $form['filter_container']['published_state'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#options' => [
        'All' => $this->t('All'),
        'published' => $this->t('Published'),
        'unpublished' => $this->t('Unpublished'),
      ],
      '#default_value' => 'All',
      '#disabled' => $is_workflow_active,
      '#description' => $is_workflow_active
        ? $this->t('Status filter is disabled when using workflow filter.')
        : NULL,
    ];

    // Content type filter.
    $form['filter_container']['node_type'] = $this->buildContentTypeFilter();

    $form['actions'] = [
      '#type' => 'actions',
      '#weight' => 99,
    ];

    $form['actions']['search'] = [
      '#type' => 'submit',
      '#value' => $this->t('Search'),
      '#button_type' => 'primary',
      '#name' => 'search',
      '#submit' => ['::generateResults'],
      '#ajax' => [
        'callback' => '::ajaxCallback',
        'wrapper' => $wrapper_id,
        'method' => 'replaceWith',
        'disable-refocus' => TRUE,
        'effect' => 'fade',
        'event' => 'click',
      ],
    ];

    if ($settings->get('save_as_csv')) {
      $form['actions']['generate_csv'] = [
        '#type' => 'submit',
        '#value' => $this->t('Download CSV'),
        '#name' => 'generate_csv',
        '#submit' => ['::generateCsv'],
        '#disabled' => !$has_results,
      ];
    }

    $form['actions']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset'),
      '#name' => 'reset',
      '#submit' => ['::resetButton'],
    ];

    $form['results'] = [
      '#type' => 'markup',
      '#markup' => $results_markup,
      '#weight' => 100,
    ];

    return $form;
  }

  /**
   * Build language filter element.
   *
   * @return array
   *   Form element array.
   */
  protected function buildLanguageFilter(): array {
    $language_options = ['All' => $this->t('All')];
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $langcode => $language) {
      $language_options[$langcode] = $language->getName();
    }

    return [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $language_options,
      '#default_value' => 'All',
    ];
  }

  /**
   * Build workflow filter element.
   *
   * @return array
   *   Form element array.
   */
  protected function buildWorkflowFilter() {
    $workflows = $this->entityTypeManager->getStorage('workflow')->loadMultiple();
    $workflow_options = ['All' => $this->t('All')];

    foreach ($workflows as $workflow) {
      $states = $workflow->getTypePlugin()->getStates();
      foreach ($states as $state) {
        if (!isset($workflow_options[$state->id()])) {
          $workflow_options[$state->id()] = [$state->label()];
        }
        else {
          $workflow_options[$state->id()][] = $state->label();
        }
      }
    }

    foreach ($workflow_options as $id => $label) {
      if (is_array($label)) {
        $workflow_options[$id] = implode('/', array_unique($label));
      }
    }

    return [
      '#type' => 'select',
      '#title' => $this->t('Workflow'),
      '#options' => $workflow_options,
      '#default_value' => 'All',
      '#ajax' => [
        'callback' => '::ajaxCallback',
        'wrapper' => $this->getFormId() . '-wrapper',
        'event' => 'change',
      ],
    ];
  }

  /**
   * Build content type filter element.
   *
   * @return array
   *   Form element array.
   */
  protected function buildContentTypeFilter(): array {
    $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $content_type_options = ['All' => $this->t('All')];

    foreach ($node_types as $node_type) {
      $content_type_options[$node_type->id()] = $node_type->label();
    }

    $content_type_options['menu'] = $this->t('Menu');
    $content_type_options['taxonomy'] = $this->t('Taxonomy Terms');

    return [
      '#type' => 'select',
      '#title' => $this->t('Content Type'),
      '#options' => $content_type_options,
      '#default_value' => 'All',
    ];
  }

  /**
   * AJAX callback for form rebuild.
   */
  public function ajaxCallback(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * Generate search results.
   */
  public function generateResults(array &$form, FormStateInterface $form_state): void {
    $needle = $form_state->getValue('needle');

    if (strlen($needle) < 3) {
      return;
    }

    $regexed = (bool) $form_state->getValue('regexed');
    $render = (bool) $form_state->getValue('render');
    $langcode = $form_state->getValue('language_selection');

    if ($form_state->getValue('invalidate_cache')) {
      $this->cacheTagsInvalidator->invalidateTags(['search_results:data']);
    }

    $settings = $this->configFactory->get('find_text.settings');
    $is_cache_enabled = $settings->get('enable_search_results_cache');
    $cid = 'search_results:' . md5($needle . $regexed . $render . $langcode);

    // Try to load from cache.
    if ($is_cache_enabled && $cache = $this->cacheManager->get($cid)) {
      $results = $cache->data['results'];
      $filtered_results = $cache->data['table'];
      $renderable_results = $cache->data['renderable_results'];
    }
    else {
      // Perform search.
      $results = $this->textSearchService->searchFields($needle, $regexed, $render, $langcode);

      // Apply filters.
      $workflow_filter = $form_state->getValue('workflow_selection');
      $publication_status_filter = $form_state->getValue('published_state');
      $bundle_filter = $form_state->getValue('node_type');

      $filtered_results = $this->filterResults($results, $workflow_filter, $publication_status_filter, $bundle_filter);
      $renderable_results = $this->renderResultsTable($filtered_results);

      // Cache the results.
      if ($is_cache_enabled) {
        $cache_expiry = time() + ($settings->get('search_results_cache_duration') ?? 3600);
        $this->cacheManager->set(
          $cid,
          [
            'results' => $results,
            'table' => $filtered_results,
            'renderable_results' => $renderable_results,
          ],
          $cache_expiry,
          ['search_results:data']
        );
      }
    }

    $rendered_results = $this->renderer->render($renderable_results);

    $form_state->set('results', $results);
    $form_state->set('filtered_results', $filtered_results);
    $form_state->set('results_markup', $rendered_results);
    $form_state->setRebuild(TRUE);
  }

  /**
   * Generate CSV file from search results.
   */
  public function generateCsv(array &$form, FormStateInterface $form_state): void {
    $filtered_results = $form_state->get('filtered_results');

    if (empty($filtered_results)) {
      $this->messenger()->addWarning($this->t('No results to export.'));
      return;
    }

    $response = $this->generateCsvFile($filtered_results);
    $form_state->setResponse($response);
    $form_state->setRebuild();
  }

  /**
   * Generate a CSV file from results.
   *
   * @param array $results
   *   The filtered results array.
   *
   * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
   *   Binary file response.
   */
  protected function generateCsvFile(array $results): BinaryFileResponse {
    $file_directory = $this->configFactory->get('system.file')->get('default_scheme') . '://';
    $file_name = 'search_results_' . date('Y-m-d_H-i-s') . '.csv';
    $csv_file_path = $file_directory . $file_name;
    $csv_file = fopen($csv_file_path, 'w');

    // Write header.
    $header = [
      'Entity ID',
      'Entity Type',
      'Title',
      'Content Type',
      'Node URL',
      'Edit URL',
      'Layout Builder URL',
      'Status',
      'Author',
      'Created',
      'Updated',
      'Field: Contents',
    ];
    fputcsv($csv_file, $header);

    // Write data rows.
    if (!empty($results['#rows'])) {
      foreach ($results['#rows'] as $row) {
        $matches = $row['matches'];
        foreach ($matches as $match) {
          $csv_row = [
            $row['id'],
            $row['entity_type'],
            $row['title'],
            $row['content_type'] ?? '',
            $row['node_url'] ?? '',
            $row['edit_url'] ?? '',
            $row['lb_edit_url'] ?? '',
            $row['status'] ?? '',
            $row['author'] ?? '',
            $row['created'] ?? '',
            $row['updated'] ?? '',
            strip_tags((string) $match),
          ];
          fputcsv($csv_file, $csv_row);
        }
      }
    }

    fclose($csv_file);

    $response = new BinaryFileResponse($csv_file_path);
    $response->headers->set('Content-Type', 'text/csv');
    $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, $file_name);

    return $response;
  }

  /**
   * Reset the form to default state.
   */
  public function resetButton(array &$form, FormStateInterface $form_state): void {
    $form_state->setValue('needle', '');
    $form_state->set('results', FALSE);
    $form_state->set('filtered_results', FALSE);
    $form_state->set('results_markup', '');
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();

    if ($triggering_element['#name'] === 'search') {
      $needle = $form_state->getValue('needle');
      if (strlen($needle) < 3) {
        $form_state->setErrorByName('needle', $this->t('The search target must contain 3 or more characters.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Form submission is handled by button-specific submit handlers.
  }

  /**
   * Filter results based on user selections.
   *
   * @param array $results
   *   Raw search results.
   * @param string|null $workflow_filter
   *   Workflow filter value.
   * @param string|null $publication_status_filter
   *   Publication status filter value.
   * @param string|null $bundle_filter
   *   Bundle filter value.
   *
   * @return array
   *   Filtered results table array.
   */
  protected function filterResults(array $results, string|null $workflow_filter = NULL, string|null $publication_status_filter = NULL, string|null $bundle_filter = NULL): array {
    if (empty($results)) {
      return [];
    }

    // Reorganize results by entity.
    $reorganized = [];
    foreach ($results as $type => $typed_results) {
      foreach ($typed_results as $entity_id => $matches) {
        $mod_key = $type . '-' . $entity_id;
        foreach ($matches as $match) {
          $reorganized[$mod_key][] = $match->value;
        }
      }
    }

    $data_rows = [];
    $node_storage = $this->entityTypeManager->getStorage('node');

    foreach ($reorganized as $mod_key => $matches) {
      [$type, $id] = explode('-', $mod_key, 2);

      $row_data = $this->buildResultRow($type, $id, $matches, $workflow_filter, $publication_status_filter, $bundle_filter, $node_storage);

      if ($row_data) {
        $data_rows[] = $row_data;
      }
    }

    if (empty($data_rows)) {
      return [];
    }

    return [
      '#type' => 'table',
      '#header' => [
        'id' => 'Entity ID',
        'entity_type' => 'Entity Type',
        'title' => 'Title',
        'type' => 'Type',
        'node_url' => 'Node URL',
        'edit_url' => 'Edit URL',
        'lb_edit_url' => 'Layout Builder URL',
        'status' => 'Status',
        'author' => 'Author',
        'created' => 'Created',
        'updated' => 'Updated',
        'matches' => 'Matches',
      ],
      '#rows' => $data_rows,
    ];
  }

  /**
   * Build a result row for a single entity.
   *
   * @param string $type
   *   Entity type.
   * @param mixed $id
   *   Entity ID.
   * @param array $matches
   *   Array of match values.
   * @param string|null $workflow_filter
   *   Workflow filter.
   * @param string|null $publication_status_filter
   *   Publication status filter.
   * @param string|null $bundle_filter
   *   Bundle filter.
   * @param object $node_storage
   *   Node storage object.
   *
   * @return array|null
   *   Row data or NULL if filtered out.
   */
  protected function buildResultRow(string $type, mixed $id, array $matches, ?string $workflow_filter, ?string $publication_status_filter, ?string $bundle_filter, object $node_storage): ?array {
    return match ($type) {
      'block_content', 'paragraph', 'node' => $this->buildNodeResultRow($id, $matches, $workflow_filter, $publication_status_filter, $bundle_filter, $node_storage),
      'menu_link_content' => $this->buildMenuResultRow($id, $matches, $publication_status_filter, $bundle_filter),
      'taxonomy_term' => $this->buildTaxonomyResultRow($id, $matches, $bundle_filter),
      default => NULL,
    };
  }

  /**
   * Build result row for node entities.
   *
   * @param mixed $id
   *   Entity ID.
   * @param array $matches
   *   Array of match values.
   * @param string|null $workflow_filter
   *   Workflow filter.
   * @param string|null $publication_status_filter
   *   Publication status filter.
   * @param string|null $bundle_filter
   *   Bundle filter.
   * @param object $node_storage
   *   Node storage object.
   *
   * @return array|null
   *   Row data or NULL if filtered out.
   */
  protected function buildNodeResultRow(mixed $id, array $matches, ?string $workflow_filter, ?string $publication_status_filter, ?string $bundle_filter, object $node_storage): ?array {
    try {
      $node = $node_storage->load($id);
      if (!$node) {
        return NULL;
      }

      $node_type_machine_name = $node->getType();

      if ($bundle_filter && $bundle_filter !== 'All' && $node_type_machine_name !== $bundle_filter) {
        return NULL;
      }

      // Determine status.
      $status = $node->isPublished() ? 'published' : 'unpublished';
      $to_compare = $status;
      $status_filter = $publication_status_filter;
      if ($node->hasField('moderation_state') && $node->get('moderation_state')->value) {
        $status = $node->get('moderation_state')->value;
        if ($workflow_filter && $workflow_filter !== 'All') {
          $status_filter = $workflow_filter;
          $to_compare = $status;
        }
      }

      if ($status_filter && $status_filter !== 'All' && $to_compare !== $status_filter) {
        return NULL;
      }

      $node_url = Url::fromRoute('entity.node.canonical', ['node' => $id])->toString();
      $edit_url = Url::fromRoute('entity.node.edit_form', ['node' => $id])->toString();
      $lb_edit_url = $node->hasField('layout_builder__layout') ? '/node/' . $id . '/layout' : '';

      return [
        'id' => $id,
        'entity_type' => 'node',
        'title' => $node->getTitle(),
        'content_type' => node_get_type_label($node),
        'node_url' => $node_url,
        'edit_url' => $edit_url,
        'lb_edit_url' => $lb_edit_url,
        'status' => $status,
        'author' => $node->getOwner()->getDisplayName(),
        'created' => date("m-d-Y", $node->getCreatedTime()),
        'updated' => date("m-d-Y", $node->getChangedTime()),
        'matches' => $matches,
      ];
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Build result row for menu entities.
   *
   * @param mixed $id
   *   Entity ID.
   * @param array $matches
   *   Array of match values.
   * @param string|null $publication_status_filter
   *   Publication status filter.
   * @param string|null $bundle_filter
   *   Bundle filter.
   *
   * @return array|null
   *   Row data or NULL if filtered out.
   */
  protected function buildMenuResultRow(mixed $id, array $matches, ?string $publication_status_filter, ?string $bundle_filter): ?array {
    if ($bundle_filter && !in_array($bundle_filter, ['All', 'menu'])) {
      return NULL;
    }

    try {
      $menu_link_storage = $this->entityTypeManager->getStorage('menu_link_content');
      $menu_link = $menu_link_storage->load($id);

      if (!$menu_link) {
        return NULL;
      }

      $status = $menu_link->isPublished() ? 'published' : 'unpublished';

      if ($publication_status_filter && $publication_status_filter !== 'All' && $publication_status_filter !== $status) {
        return NULL;
      }

      return [
        'id' => $id,
        'entity_type' => 'menu_link_content',
        'title' => $menu_link->getTitle(),
        'content_type' => 'Menu Link',
        'node_url' => '',
        'edit_url' => '/admin/structure/menu/item/' . $id . '/edit',
        'lb_edit_url' => '',
        'status' => $status,
        'author' => '',
        'created' => '',
        'updated' => '',
        'matches' => $matches,
      ];
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Build result row for taxonomy term entities.
   *
   * @param mixed $id
   *   Entity ID.
   * @param array $matches
   *   Array of match values.
   * @param string|null $bundle_filter
   *   Bundle filter.
   *
   * @return array|null
   *   Row data or NULL if filtered out.
   */
  protected function buildTaxonomyResultRow(mixed $id, array $matches, ?string $bundle_filter): ?array {
    if ($bundle_filter && !in_array($bundle_filter, ['All', 'taxonomy'])) {
      return NULL;
    }

    try {
      $term = Term::load($id);

      if (!$term) {
        return NULL;
      }

      return [
        'id' => $id,
        'entity_type' => 'taxonomy_term',
        'title' => $term->getName(),
        'content_type' => 'Taxonomy Term',
        'node_url' => '/taxonomy/term/' . $id,
        'edit_url' => '/taxonomy/term/' . $id . '/edit',
        'lb_edit_url' => '',
        'status' => '',
        'author' => '',
        'created' => '',
        'updated' => '',
        'matches' => $matches,
      ];
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Render results as a table for display.
   *
   * @param array $results
   *   Filtered results array.
   *
   * @return array
   *   Renderable array.
   */
  protected function renderResultsTable(array $results): array {
    if (empty($results)) {
      return [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('No results found.') . '</p>',
      ];
    }

    $render_rows = [];

    foreach ($results['#rows'] as $result) {
      $entity_value = $this->formatEntityDisplay($result);
      $matches = $result['matches'];

      // First row includes entity info.
      $render_rows[] = [
        'id' => [
          'data' => $entity_value,
          'rowspan' => count($matches),
        ],
        'field' => array_shift($matches),
      ];

      // Additional rows for remaining matches.
      foreach ($matches as $match) {
        $render_rows[] = [
          'field' => $match,
        ];
      }
    }

    return [
      '#type' => 'table',
      '#header' => [
        'id' => $this->t('Entity'),
        'field' => $this->t('Field: Contents'),
      ],
      '#rows' => $render_rows,
    ];
  }

  /**
   * Format entity display for results table.
   *
   * @param array $result
   *   Result row data.
   *
   * @return \Drupal\Component\Render\FormattableMarkup
   *   Formatted markup.
   */
  protected function formatEntityDisplay(array $result) {
    switch ($result['entity_type']) {
      case 'node':
        $template = '<a href="@nodeUrl">@title</a><br /><strong>Node:</strong> @nid (<a href="@editUrl">edit</a>)';
        $args = [
          '@nid' => $result['id'],
          '@nodeUrl' => $result['node_url'],
          '@editUrl' => $result['edit_url'],
          '@title' => $result['title'],
        ];

        if ($result['lb_edit_url']) {
          $template .= ' (<a href="@lbEditUrl">layout</a>)';
          $args['@lbEditUrl'] = $result['lb_edit_url'];
        }

        $template .= '<br /><strong>Type:</strong> @type<br /><strong>Status:</strong> @status<br /><strong>Author:</strong> @author<br /><strong>Created:</strong> @date<br /><strong>Updated:</strong> @update';
        $args = array_merge($args, [
          '@type' => $result['content_type'],
          '@status' => $result['status'],
          '@author' => $result['author'],
          '@date' => $result['created'],
          '@update' => $result['updated'],
        ]);
        return new FormattableMarkup($template, $args);

      case 'menu_link_content':
        return new FormattableMarkup('@title<br /><strong>Menu:</strong> @mid (<a href="@editUrl">edit</a>)', [
          '@mid' => $result['id'],
          '@title' => $result['title'],
          '@editUrl' => $result['edit_url'],
        ]);

      case 'taxonomy_term':
        return new FormattableMarkup('<a href="@termUrl">@title</a><br /><strong>Term:</strong> @tid (<a href="@editUrl">edit</a>)', [
          '@tid' => $result['id'],
          '@title' => $result['title'],
          '@termUrl' => $result['node_url'],
          '@editUrl' => $result['edit_url'],
        ]);

      default:
        return new FormattableMarkup('@title', ['@title' => $result['title']]);
    }
  }

}
