<?php

namespace Drupal\find_text;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Service for searching text across entity fields.
 */
class TextSearchService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a TextSearchService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(Connection $database, EntityFieldManagerInterface $entity_field_manager, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory, ModuleHandlerInterface $module_handler) {
    $this->database = $database;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Get field tables and value mappings for all text-based fields.
   *
   * @return array
   *   The field names => info mapping for all text-based fields.
   */
  public function getTextFieldTables() {
    $fields = [];
    $tables = [];
    $config = $this->configFactory->get('find_text.settings');
    $field_types = $config->get('field_types') ?? [];
    $allow_all_entities = $config->get('allow_all_entities') ?? TRUE;

    $allowed_entity_types = [];
    if (!$allow_all_entities) {
      $allowed_entity_types = $config->get('entity_types') ?? [];
      foreach ($allowed_entity_types as $entity_type_id => $info) {
        if (empty($info['allowed'])) {
          unset($allowed_entity_types[$entity_type_id]);
        }
      }
    }

    // Gather fields by type.
    foreach ($field_types as $field_type => $field_config_settings) {
      if (empty($field_config_settings['allowed'])) {
        continue;
      }

      $new_fields = $this->entityFieldManager->getFieldMapByFieldType($field_type);

      // Filter by allowed entity types if configured.
      if (!$allow_all_entities) {
        foreach ($new_fields as $entity_type => $fields_data) {
          if (!in_array($entity_type, array_keys($allowed_entity_types))) {
            unset($new_fields[$entity_type]);
          }
        }
      }

      // Merge fields by entity type.
      foreach ($new_fields as $entity_type => $field_info) {
        $fields[$entity_type] = isset($fields[$entity_type])
          ? array_merge($fields[$entity_type], $field_info)
          : $field_info;
      }
    }

    // Process fields to build table mapping.
    foreach ($fields as $entity_type_id => $field_info) {
      $this->processEntityTypeFields($entity_type_id, $field_info, $tables);
    }

    $this->cleanupTables($tables);
    return $tables;
  }

  /**
   * Process fields for a specific entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array $field_info
   *   Field information array.
   * @param array $tables
   *   Tables array to populate (passed by reference).
   */
  protected function processEntityTypeFields(string $entity_type_id, array $field_info, array &$tables): void {
    switch ($entity_type_id) {
      case 'block_content':
      case 'node':
      case 'paragraph':
        $this->processContentEntityFields($entity_type_id, $field_info, $tables);
        break;

      case 'menu_link_content':
        $tables['menu_link_content_data'] = [
          'values' => ['title', 'link__uri'],
          'parent' => ['id'],
          'type' => $entity_type_id,
          'field_name' => 'menu',
        ];
        break;

      case 'taxonomy_term':
        $tables['taxonomy_term_field_data'] = [
          'values' => ['name', 'description__value'],
          'parent' => ['tid'],
          'type' => $entity_type_id,
          'field_name' => 'term',
        ];
        break;
    }
  }

  /**
   * Process content entity fields (node, paragraph, block_content).
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param array $field_info
   *   Field information array.
   * @param array $tables
   *   Tables array to populate (passed by reference).
   */
  protected function processContentEntityFields(string $entity_type_id, array $field_info, array &$tables): void {
    try {
      $entity_storage = $this->entityTypeManager->getStorage($entity_type_id);
      $field_storage_definitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type_id);
      $table_mapping = $entity_storage->getTableMapping($field_storage_definitions);

      foreach (array_intersect_key($field_storage_definitions, $field_info) as $field_storage_definition) {
        $field_name = $field_storage_definition->getName();

        // Skip rabbit hole fields.
        if (str_starts_with($field_name, 'rh_')) {
          continue;
        }

        try {
          $table_name = $table_mapping->getFieldTableName($field_name);
          $value_columns = $this->getValueColumns($field_name, $field_storage_definition->getType());
          $parent_type = ['entity_id', 'revision_id'];

          // Handle title fields differently.
          if ($field_name === 'title') {
            $parent_type = ['nid'];
            $value_columns = ['title'];
          }

          $tables[$table_name] = [
            'values' => $value_columns,
            'parent' => $parent_type,
            'type' => $entity_type_id,
            'field_name' => $field_name,
          ];
        }
        catch (\Exception $e) {
          // Skip fields that can't be mapped to tables.
          continue;
        }
      }
    }
    catch (\Exception $e) {
      // Skip entity types that can't be processed.
      return;
    }
  }

  /**
   * Search all fields for a needle.
   *
   * @param string $needle
   *   The string for which to search.
   * @param bool $regexed
   *   Whether a regex condition should be used over base LIKE.
   * @param bool $render
   *   Whether the results should be returned as rendered HTML.
   * @param string|null $langcode
   *   The language code to filter by, or NULL for all languages.
   *
   * @return array
   *   The search results.
   */
  public function searchFields(string $needle, bool $regexed = FALSE, bool $render = FALSE, string|null $langcode = NULL): array {
    $results = [];
    $tables = $this->getTextFieldTables();

    foreach ($tables as $table => $details) {
      $this->searchTable($table, $details, $needle, $regexed, $render, $langcode, $results);
    }

    // Allow other modules to alter results.
    $this->moduleHandler->invokeAll('find_text_results', [&$results]);

    return $results;
  }

  /**
   * Search a single table for matches.
   *
   * @param string $table
   *   The table name.
   * @param array $details
   *   Table details (values, parent, type, field_name).
   * @param string $needle
   *   The search string.
   * @param bool $regexed
   *   Whether to use REGEXP instead of LIKE.
   * @param bool $render
   *   Whether to render HTML markup.
   * @param string|null $langcode
   *   Language code filter.
   * @param array $results
   *   Results array (passed by reference).
   */
  protected function searchTable(string $table, array $details, string $needle, bool $regexed, bool $render, string|null $langcode, array &$results): void {
    $value_columns = $details['values'];
    $parent = $details['parent'];

    foreach ($value_columns as $value_column) {
      $query = $this->database->select($table)
        ->fields($table, array_merge($parent, [$value_column]));

      if ($regexed) {
        $query->condition($table . '.' . $value_column, $needle, 'REGEXP');
      }
      else {
        // Use wildcards for LIKE search.
        $query->condition($table . '.' . $value_column, '%' . $this->database->escapeLike($needle) . '%', 'LIKE');
      }

      if ($langcode && $langcode !== "All") {
        $query->condition($table . '.langcode', $langcode);
      }

      $temp_results = $query->execute()->fetchAllAssoc($parent[0]);

      if (empty($temp_results)) {
        continue;
      }

      $this->processSearchResults($temp_results, $value_column, $needle, $render, $details, $results);
    }
  }

  /**
   * Process and format search results.
   *
   * @param array $temp_results
   *   Temporary results from query.
   * @param string $value_column
   *   The column name being searched.
   * @param string $needle
   *   The search string.
   * @param bool $render
   *   Whether to render HTML.
   * @param array $details
   *   Table details.
   * @param array $results
   *   Results array (passed by reference).
   */
  protected function processSearchResults(array $temp_results, string $value_column, string $needle, bool $render, array $details, array &$results): void {
    foreach ($temp_results as $id => $row) {
      // Format the result value.
      $formatted_value = $this->formatResultValue($row, $value_column, $needle, $render);
      $row->value = $formatted_value;

      // Fetch parent entity.
      $revision_id = $row->revision_id ?? NULL;
      $ids = [
        'id' => $id,
        'revision_id' => $revision_id,
      ];
      $new_id = $this->fetchParent($ids, $details['type']);

      unset($temp_results[$id]);

      if ($new_id) {
        $marker = 'id_' . $new_id['id'];
        if (!isset($temp_results[$marker])) {
          $temp_results[$marker] = [];
        }
        $temp_results[$marker][] = $row;
      }
    }

    // Organize results by entity type.
    foreach ($temp_results as $marker => $values) {
      $exploded = explode('_', $marker);
      $id = $exploded[1];

      $type = in_array($details['type'], ['node', 'block_content', 'paragraph'])
        ? 'node'
        : $details['type'];

      $results[$type][$id] = isset($results[$type][$id])
        ? array_merge($results[$type][$id], $values)
        : $values;
    }
  }

  /**
   * Format a result value with highlighting.
   *
   * @param object $row
   *   The result row.
   * @param string $value_column
   *   The column name.
   * @param string $needle
   *   The search string.
   * @param bool $render
   *   Whether to render HTML.
   *
   * @return \Drupal\Component\Render\FormattableMarkup
   *   The formatted value.
   */
  protected function formatResultValue($row, $value_column, $needle, $render): FormattableMarkup {
    $field_label = $this->prettifyMachineName($value_column);

    if ($render) {
      // Simple highlighting for rendered mode.
      $highlighted = preg_replace(
        '%(?![^<]*>)' . preg_quote($needle, '%') . '%i',
        '<span class="find-text-match">$0</span>',
        $row->$value_column
      );
      return new FormattableMarkup("<strong>@name</strong><br/>" . $highlighted, [
        '@name' => $field_label,
      ]);
    }

    // Non-rendered mode: escape HTML but highlight matches.
    $tokenized = preg_replace(
      '%' . preg_quote($needle, '%') . '%i',
      '[TOKEN]$0[TOKEN]',
      $row->$value_column
    );
    $exploded = explode('[TOKEN]', $tokenized);
    $counter = 0;
    $replacements = [];

    foreach ($exploded as &$part) {
      if (preg_match('%' . preg_quote($needle, '%') . '%i', $part)) {
        $part = '<span class="find-text-match">' . $part . '</span>';
      }
      else {
        $replacements['@match' . $counter] = $part;
        $part = '@match' . $counter++;
      }
    }

    $highlighted_string = implode('', $exploded);
    return new FormattableMarkup(
      "<strong>@name</strong><br/>" . $highlighted_string,
      array_merge(['@name' => $field_label], $replacements)
    );
  }

  /**
   * Fetch parent entity for paragraphs and blocks.
   *
   * @param array $ids
   *   Entity and revision IDs.
   * @param string $type
   *   Entity type.
   *
   * @return array|false|null
   *   Parent IDs or FALSE/NULL if not found.
   */
  public function fetchParent(array $ids, string $type): false|array|null {
    switch ($type) {
      case 'paragraph':
        return $this->fetchParagraphParent($ids);

      case 'block_content':
        return $this->fetchBlockParent($ids);

      case 'node':
      default:
        return $ids;
    }
  }

  /**
   * Fetch parent for a paragraph entity.
   *
   * @param array $ids
   *   Entity and revision IDs.
   *
   * @return array|false
   *   Parent IDs or FALSE.
   */
  protected function fetchParagraphParent(array $ids): array|false {
    try {
      $paragraph = $this->entityTypeManager
        ->getStorage('paragraph')
        ->loadRevision($ids['revision_id']);

      if ($paragraph) {
        $parent = $paragraph->getParentEntity();
        if ($parent) {
          return $this->fetchParent([
            'id' => $parent->id(),
            'revision_id' => $parent->getRevisionId(),
          ], $parent->getEntityType()->id());
        }
      }

      // Fallback to database query.
      $result = $this->database->select('entity_usage')
        ->fields('entity_usage', ['source_id', 'source_vid', 'source_type'])
        ->condition('target_type', 'paragraph')
        ->condition('target_id', $ids['id'])
        ->distinct()
        ->orderBy('source_vid', 'DESC')
        ->execute()
        ->fetchAll();

      if (!empty($result[0])) {
        return $this->fetchParent([
          'id' => $result[0]->source_id,
          'revision_id' => $result[0]->source_vid,
        ], $result[0]->source_type);
      }
    }
    catch (\Exception $e) {
      return FALSE;
    }

    return FALSE;
  }

  /**
   * Fetch parent for a block_content entity.
   *
   * @param array $ids
   *   Entity and revision IDs.
   *
   * @return array|null
   *   Parent IDs or NULL.
   */
  protected function fetchBlockParent(array $ids): array|null {
    if (!$this->moduleHandler->moduleExists('layout_builder')) {
      return NULL;
    }

    if (!$this->database->schema()->tableExists('node__layout_builder__layout')) {
      return NULL;
    }

    try {
      $result = $this->database->select('node__layout_builder__layout')
        ->fields('node__layout_builder__layout', ['entity_id', 'revision_id'])
        ->condition('layout_builder__layout_section', '%"block_revision_id";s:_:"' . $ids['revision_id'] . '";%', 'LIKE')
        ->execute()
        ->fetchAll();

      if (!empty($result[0])) {
        return [
          'id' => $result[0]->entity_id,
          'revision_id' => $result[0]->revision_id,
        ];
      }
    }
    catch (\Exception $e) {
      return NULL;
    }

    return NULL;
  }

  /**
   * Get value columns for a field.
   *
   * @param string $name
   *   Field name.
   * @param string $type
   *   Field type.
   *
   * @return array
   *   Value column names.
   */
  protected function getValueColumns(string $name, string $type): array {
    $config = $this->configFactory->get('find_text.settings');
    $suffixes = $config->get("field_types.{$type}.value_columns");

    if (empty($suffixes)) {
      return [$name . '_value'];
    }

    $value_columns = [];
    foreach ($suffixes as $suffix) {
      $value_columns[] = $name . $suffix;
    }

    return $value_columns;
  }

  /**
   * Remove tables that should be skipped.
   *
   * @param array $tables
   *   Tables array (passed by reference).
   */
  protected function cleanupTables(array &$tables): void {
    $config = $this->configFactory->get('find_text.settings');
    $tables_to_skip = $config->get('tables_to_skip') ?? [];

    foreach ($tables_to_skip as $to_remove) {
      if (isset($tables[$to_remove])) {
        unset($tables[$to_remove]);
      }
    }
  }

  /**
   * Convert machine name to human-readable format.
   *
   * @param string $machine_name
   *   The machine name.
   *
   * @return string
   *   Human-readable name.
   */
  protected function prettifyMachineName($machine_name): string {
    // Remove common prefixes and suffixes.
    $machine_name = preg_replace('/(field_)|(uiowa_)|(_value)/', '', $machine_name);
    // Replace underscores with spaces.
    $machine_name = str_replace('_', ' ', $machine_name);
    return ucwords($machine_name);
  }

}
