<?php

namespace Drupal\Tests\first_time_login\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\User;
use Drupal\Core\Database\Database;

/**
 * Functional tests for First Time Login module.
 *
 * @group first_time_login
 */
class FirstTimeLoginFunctionalTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'first_time_login',
    'user',
  ];

  /**
   * Admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Regular user for testing.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->database = Database::getConnection();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer users',
      'administer site configuration',
    ]);

    // Create a test user.
    $this->testUser = $this->drupalCreateUser();
  }

  /**
   * Tests the settings form functionality.
   */
  public function testSettingsForm() {
    $this->drupalLogin($this->adminUser);

    // Visit the settings page.
    $this->drupalGet('admin/config/people/first-time-login/settings');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('First Time Login');

    // Test form elements are present.
    $this->assertSession()->fieldExists('first_time_login_config_days');
    $this->assertSession()->fieldExists('first_time_login_new_user_message');
    $this->assertSession()->fieldExists('first_time_login_update_user_message');

    // Test form submission.
    $edit = [
      'first_time_login_config_days' => '90',
      'first_time_login_new_user_message' => 'Welcome @user to @site_name!',
      'first_time_login_update_user_message' => 'Please update your profile @user.',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify configuration was saved.
    $config = $this->config('first_time_login.settings');
    $this->assertEquals('90', $config->get('first_time_login_config_days'));
    $this->assertEquals('Welcome @user to @site_name!', $config->get('first_time_login_new_user_message'));
    $this->assertEquals('Please update your profile @user.', $config->get('first_time_login_update_user_message'));
  }

  /**
   * Tests form validation.
   */
  public function testSettingsFormValidation() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/people/first-time-login/settings');

    // Submit invalid data.
    $edit = [
      'first_time_login_config_days' => 'invalid',
      'first_time_login_new_user_message' => 'Welcome message',
      'first_time_login_update_user_message' => 'Update message',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Enter the number of days should be number.');
  }

  /**
   * Tests first time login redirect functionality.
   */
  public function testFirstTimeLoginRedirect() {
    // Set up configuration.
    $this->config('first_time_login.settings')
      ->set('first_time_login_config_days', 30)
      ->set('first_time_login_new_user_message', 'Welcome @user to @site_name! Your account was created on @created_date. Please update your account details.')
      ->save();

    // Create a user that hasn't logged in yet.
    $new_user = User::create([
      'name' => 'newuser',
      'mail' => 'newuser@example.com',
      'pass' => 'password',
      'status' => 1,
    ]);
    $new_user->save();

    // Manually set the user as first time login in the database.
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 1, 'updated_date' => 0])
      ->condition('uid', $new_user->id())
      ->execute();

    // Login as the new user.
    $this->drupalLogin($new_user);

    // Should be redirected to profile edit page.
    $this->assertSession()->addressEquals('/user/' . $new_user->id() . '/edit');
    $this->assertSession()->pageTextContains('Welcome newuser to');
  }

  /**
   * Tests that super user (uid=1) is not redirected.
   */
  public function testSuperUserExclusion() {
    // Create user with uid=1 (admin user).
    $super_user = User::create([
      'uid' => 1,
      'name' => 'admin',
      'mail' => 'admin@example.com',
      'pass' => 'password',
      'status' => 1,
    ]);
    $super_user->save();

    // Login as super user.
    $this->drupalLogin($super_user);

    // Should not be redirected (will be on user profile page).
    $this->assertSession()->addressEquals('/user/1');
  }

  /**
   * Tests redirect for users who need profile updates.
   */
  public function testProfileUpdateReminder() {
    // Set up configuration with short threshold.
    $this->config('first_time_login.settings')
      ->set('first_time_login_config_days', 1)
      ->set('first_time_login_update_user_message', 'Welcome @user to @site_name! Your account was updated on @updated_date. Please update your account details.')
      ->save();

    // Update user record to simulate old profile update.
    $old_timestamp = REQUEST_TIME - (2 * 24 * 60 * 60); // 2 days ago
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 0, 'updated_date' => $old_timestamp])
      ->condition('uid', $this->testUser->id())
      ->execute();

    // Login as test user.
    $this->drupalLogin($this->testUser);

    // Should be redirected to profile edit page.
    $this->assertSession()->addressEquals('/user/' . $this->testUser->id() . '/edit');
    $this->assertSession()->pageTextContains('Welcome ' . $this->testUser->getAccountName() . ' to');
  }

  /**
   * Tests that users with recent updates are not redirected.
   */
  public function testNoRedirectForRecentUpdates() {
    // Set up configuration.
    $this->config('first_time_login.settings')
      ->set('first_time_login_config_days', 30)
      ->save();

    // Update user record to simulate recent profile update.
    $recent_timestamp = REQUEST_TIME - (1 * 24 * 60 * 60); // 1 day ago
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 0, 'updated_date' => $recent_timestamp])
      ->condition('uid', $this->testUser->id())
      ->execute();

    // Login as test user.
    $this->drupalLogin($this->testUser);

    // Should not be redirected (will be on user profile page).
    $this->assertSession()->addressEquals('/user/' . $this->testUser->id());
  }

  /**
   * Tests the help page.
   */
  public function testHelpPage() {
    $this->drupalLogin($this->adminUser);

    // Visit the help page.
    $this->drupalGet('admin/help/first_time_login');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('About');
    $this->assertSession()->pageTextContains('Super user (user with UID = 1), will not be prompted for profile update.');
    $this->assertSession()->pageTextContains('The First time login module prompts user to reset their profile');
  }

  /**
   * Tests user profile update clears first time login flag.
   */
  public function testProfileUpdateClearsFlag() {
    // Set user as first time login.
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 1, 'updated_date' => 0])
      ->condition('uid', $this->testUser->id())
      ->execute();

    // Login and update profile.
    $this->drupalLogin($this->testUser);
    
    // Go to profile edit page.
    $this->drupalGet('/user/' . $this->testUser->id() . '/edit');
    
    // Update profile.
    $edit = [
      'mail' => 'updated@example.com',
    ];
    $this->submitForm($edit, 'Save');

    // Check that flag was cleared.
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u', ['first_time_login', 'updated_date'])
      ->condition('uid', $this->testUser->id())
      ->execute()
      ->fetchObject();

    $this->assertEquals(0, $result->first_time_login, 'First time login flag was cleared.');
    $this->assertEquals(REQUEST_TIME, $result->updated_date, 'Updated date was set to current time.');
  }

} 