<?php

namespace Drupal\Tests\first_time_login\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\User;
use Drupal\Core\Database\Database;

/**
 * Tests for First Time Login module installation.
 *
 * @group first_time_login
 */
class FirstTimeLoginInstallTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'user',
  ];

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->database = Database::getConnection();
  }

  /**
   * Tests module installation populates existing users correctly.
   */
  public function testModuleInstallation() {
    // Create some users before installing the module.
    $new_user = User::create([
      'name' => 'newuser',
      'mail' => 'newuser@example.com',
      'status' => 1,
    ]);
    $new_user->save();

    $old_user = User::create([
      'name' => 'olduser',
      'mail' => 'olduser@example.com',
      'status' => 1,
    ]);
    // Simulate a user who has logged in before.
    $old_user->set('login', REQUEST_TIME - 86400);
    $old_user->save();

    // Install the module.
    $this->container->get('module_installer')->install(['first_time_login']);

    // Verify that the database table was created.
    $this->assertTrue($this->database->schema()->tableExists('user_first_login'), 'user_first_login table was created.');

    // Verify that existing users were properly categorized.
    $new_user_record = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $new_user->id())
      ->execute()
      ->fetchObject();

    $this->assertNotEmpty($new_user_record, 'Record created for new user.');
    $this->assertEquals(1, $new_user_record->first_time_login, 'New user marked as first time login.');

    $old_user_record = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $old_user->id())
      ->execute()
      ->fetchObject();

    $this->assertNotEmpty($old_user_record, 'Record created for old user.');
    $this->assertEquals(0, $old_user_record->first_time_login, 'Old user not marked as first time login.');
  }

  /**
   * Tests that configuration is installed correctly.
   */
  public function testConfigurationInstallation() {
    // Install the module.
    $this->container->get('module_installer')->install(['first_time_login']);

    // Verify default configuration.
    $config = $this->config('first_time_login.settings');
    $this->assertEquals(120, $config->get('first_time_login_config_days'));
    $this->assertStringContainsString('Welcome @user to @site_name', $config->get('first_time_login_new_user_message'));
    $this->assertStringContainsString('Welcome @user to @site_name', $config->get('first_time_login_update_user_message'));
  }

  /**
   * Tests module uninstallation cleans up properly.
   */
  public function testModuleUninstallation() {
    // Install the module first.
    $this->container->get('module_installer')->install(['first_time_login']);

    // Create a user to populate the table.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Verify table and data exist.
    $this->assertTrue($this->database->schema()->tableExists('user_first_login'));
    $record_count = $this->database->select('user_first_login')->countQuery()->execute()->fetchField();
    $this->assertGreaterThan(0, $record_count, 'Records exist in user_first_login table.');

    // Uninstall the module.
    $this->container->get('module_installer')->uninstall(['first_time_login']);

    // Verify table was removed.
    $this->assertFalse($this->database->schema()->tableExists('user_first_login'), 'user_first_login table was removed.');

    // Verify configuration was removed.
    $config = $this->config('first_time_login.settings');
    $this->assertTrue($config->isNew(), 'Configuration was removed.');
  }

} 