<?php

namespace Drupal\Tests\first_time_login\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;
use Drupal\Core\Database\Database;

/**
 * Tests for First Time Login module configuration and edge cases.
 *
 * @group first_time_login
 */
class FirstTimeLoginConfigTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'first_time_login',
  ];

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('first_time_login', ['user_first_login']);
    $this->installConfig(['first_time_login']);

    $this->database = Database::getConnection();
  }

  /**
   * Tests default configuration values.
   */
  public function testDefaultConfiguration() {
    $config = $this->config('first_time_login.settings');
    
    $this->assertEquals(120, $config->get('first_time_login_config_days'));
    $this->assertIsString($config->get('first_time_login_new_user_message'));
    $this->assertIsString($config->get('first_time_login_update_user_message'));
    
    // Check that default messages contain expected tokens.
    $new_user_message = $config->get('first_time_login_new_user_message');
    $this->assertStringContainsString('@user', $new_user_message);
    $this->assertStringContainsString('@site_name', $new_user_message);
    $this->assertStringContainsString('@created_date', $new_user_message);
    
    $update_user_message = $config->get('first_time_login_update_user_message');
    $this->assertStringContainsString('@user', $update_user_message);
    $this->assertStringContainsString('@site_name', $update_user_message);
    $this->assertStringContainsString('@updated_date', $update_user_message);
  }

  /**
   * Tests configuration schema validation.
   */
  public function testConfigurationSchema() {
    $config = $this->config('first_time_login.settings');
    
    // Test that configuration follows schema.
    $this->assertIsInt($config->get('first_time_login_config_days'));
    $this->assertIsString($config->get('first_time_login_new_user_message'));
    $this->assertIsString($config->get('first_time_login_update_user_message'));
  }

  /**
   * Tests edge case with zero days configuration.
   */
  public function testZeroDaysConfiguration() {
    // Set configuration to 0 days.
    $this->config('first_time_login.settings')
      ->set('first_time_login_config_days', 0)
      ->save();

    // Create a user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Set user as non-first-time login with old update date.
    $old_timestamp = REQUEST_TIME - (10 * 24 * 60 * 60); // 10 days ago
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 0, 'updated_date' => $old_timestamp])
      ->condition('uid', $user->id())
      ->execute();

    // Test the logic in hook_user_login (simulated).
    $config = $this->config('first_time_login.settings');
    $current_time = REQUEST_TIME;
    $days_diff = ($current_time - $old_timestamp) / (60 * 60 * 24);
    $needs_update = ($days_diff > $config->get('first_time_login_config_days'));

    // With 0 days threshold, user should always need update.
    $this->assertTrue($needs_update, 'User should need update with 0 days threshold.');
  }

  /**
   * Tests large days configuration.
   */
  public function testLargeDaysConfiguration() {
    // Set configuration to a very large number.
    $this->config('first_time_login.settings')
      ->set('first_time_login_config_days', 36500) // ~100 years
      ->save();

    // Create a user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Set user as non-first-time login with old update date.
    $old_timestamp = REQUEST_TIME - (365 * 24 * 60 * 60); // 1 year ago
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 0, 'updated_date' => $old_timestamp])
      ->condition('uid', $user->id())
      ->execute();

    // Test the logic in hook_user_login (simulated).
    $config = $this->config('first_time_login.settings');
    $current_time = REQUEST_TIME;
    $days_diff = ($current_time - $old_timestamp) / (60 * 60 * 24);
    $needs_update = ($days_diff > $config->get('first_time_login_config_days'));

    // With large threshold, user should not need update.
    $this->assertFalse($needs_update, 'User should not need update with large days threshold.');
  }

  /**
   * Tests handling of users with zero updated_date.
   */
  public function testZeroUpdatedDate() {
    // Create a user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Set user with zero updated_date.
    $this->database->update('user_first_login')
      ->fields(['first_time_login' => 0, 'updated_date' => 0])
      ->condition('uid', $user->id())
      ->execute();

    // Test the logic in hook_user_login (simulated).
    $config = $this->config('first_time_login.settings');
    $updated_date = 0;
    $current_time = REQUEST_TIME;
    
    if ($updated_date == 0) {
      $days_diff = 0;
    } else {
      $days_diff = ($current_time - $updated_date) / (60 * 60 * 24);
    }
    
    $needs_update = ($days_diff > $config->get('first_time_login_config_days'));

    // With zero updated_date, days_diff should be 0, so no update needed.
    $this->assertEquals(0, $days_diff, 'Days diff should be 0 for zero updated_date.');
    $this->assertFalse($needs_update, 'User should not need update with zero updated_date.');
  }

} 