<?php

namespace Drupal\Tests\first_time_login\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;
use Drupal\Core\Database\Database;

/**
 * Tests for First Time Login module hooks.
 *
 * @group first_time_login
 */
class FirstTimeLoginHooksTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'first_time_login',
  ];

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('first_time_login', ['user_first_login']);
    $this->installConfig(['first_time_login']);

    $this->database = Database::getConnection();
  }

  /**
   * Tests hook_user_insert implementation.
   *
   * @covers ::first_time_login_user_insert
   */
  public function testUserInsertHook() {
    // Create a new user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Verify that a record was inserted into user_first_login table.
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $user->id())
      ->execute()
      ->fetchObject();

    $this->assertNotEmpty($result, 'Record was created in user_first_login table.');
    $this->assertEquals($user->id(), $result->uid);
    $this->assertEquals(1, $result->first_time_login);
    $this->assertEquals($user->getLastAccessedTime(), $result->updated_date);
  }

  /**
   * Tests hook_user_update implementation.
   *
   * @covers ::first_time_login_user_update
   */
  public function testUserUpdateHook() {
    // Create a new user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    // Set the current user to the test user.
    $this->container->get('current_user')->setAccount($user);

    // Update the user (simulating profile update).
    $user->set('name', 'updateduser');
    $user->save();

    // Verify that the record was updated in user_first_login table.
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $user->id())
      ->execute()
      ->fetchObject();

    $this->assertEquals(0, $result->first_time_login, 'First time login flag was set to 0.');
    $this->assertEquals(REQUEST_TIME, $result->updated_date, 'Updated date was set to current time.');
  }

  /**
   * Tests hook_user_delete implementation.
   *
   * @covers ::first_time_login_user_delete
   */
  public function testUserDeleteHook() {
    // Create a new user.
    $user = User::create([
      'name' => 'testuser',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();

    $uid = $user->id();

    // Verify record exists in user_first_login table.
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $uid)
      ->execute()
      ->fetchObject();
    $this->assertNotEmpty($result, 'Record exists before deletion.');

    // Delete the user.
    $user->delete();

    // Verify record was removed from user_first_login table.
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', $uid)
      ->execute()
      ->fetchObject();
    $this->assertEmpty($result, 'Record was deleted from user_first_login table.');
  }

  /**
   * Tests that super user (uid=1) is not affected.
   */
  public function testSuperUserExclusion() {
    // Create user with uid=1.
    $admin_user = User::create([
      'uid' => 1,
      'name' => 'admin',
      'mail' => 'admin@example.com',
      'status' => 1,
    ]);
    $admin_user->save();

    // Verify record was still created (hook_user_insert doesn't exclude uid=1).
    $result = $this->database->select('user_first_login', 'u')
      ->fields('u')
      ->condition('uid', 1)
      ->execute()
      ->fetchObject();
    $this->assertNotEmpty($result, 'Record was created for uid=1.');

    // But the login hook should exclude uid=1 (tested in functional tests).
  }

  /**
   * Tests database schema.
   */
  public function testDatabaseSchema() {
    $schema = $this->database->schema();
    
    // Test that the table exists.
    $this->assertTrue($schema->tableExists('user_first_login'), 'user_first_login table exists.');

    // Test that required fields exist.
    $this->assertTrue($schema->fieldExists('user_first_login', 'uid'), 'uid field exists.');
    $this->assertTrue($schema->fieldExists('user_first_login', 'first_time_login'), 'first_time_login field exists.');
    $this->assertTrue($schema->fieldExists('user_first_login', 'updated_date'), 'updated_date field exists.');

    // Test primary key.
    $this->assertTrue($schema->indexExists('user_first_login', 'PRIMARY'), 'Primary key exists.');
  }

} 