<?php

namespace Drupal\Tests\first_time_login\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\first_time_login\Form\FirstTimeLoginSettings;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\Config;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Unit tests for FirstTimeLoginSettings form.
 *
 * @coversDefaultClass \Drupal\first_time_login\Form\FirstTimeLoginSettings
 * @group first_time_login
 */
class FirstTimeLoginSettingsTest extends UnitTestCase {

  /**
   * The form under test.
   *
   * @var \Drupal\first_time_login\Form\FirstTimeLoginSettings
   */
  protected $form;

  /**
   * Mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactory|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock config object.
   *
   * @var \Drupal\Core\Config\Config|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->config = $this->createMock(Config::class);
    $this->configFactory = $this->createMock(ConfigFactory::class);
    $this->configFactory->method('getEditable')
      ->with('first_time_login.settings')
      ->willReturn($this->config);

    $this->form = new FirstTimeLoginSettings($this->configFactory);

    $string_translation = $this->createMock(TranslationInterface::class);
    $string_translation->method('translate')
      ->willReturnCallback(function ($string) {
        return $string;
      });
    $this->form->setStringTranslation($string_translation);
  }

  /**
   * Tests the getEditableConfigNames method.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames() {
    $expected = ['first_time_login.settings'];
    $this->assertEquals($expected, $this->form->getEditableConfigNames());
  }

  /**
   * Tests the getFormId method.
   *
   * @covers ::getFormId
   */
  public function testGetFormId() {
    $this->assertEquals('first_time_login.settings', $this->form->getFormId());
  }

  /**
   * Tests the buildForm method.
   *
   * @covers ::buildForm
   */
  public function testBuildForm() {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->config->method('get')
      ->willReturnMap([
        ['first_time_login_config_days', 120],
        ['first_time_login_new_user_message', 'Welcome new user'],
        ['first_time_login_update_user_message', 'Please update your profile'],
      ]);

    $form = $this->form->buildForm([], $form_state);

    // Check that all required form elements are present.
    $this->assertArrayHasKey('first_time_login_config_days', $form);
    $this->assertArrayHasKey('first_time_login_new_user_message', $form);
    $this->assertArrayHasKey('first_time_login_update_user_message', $form);

    // Check form element properties.
    $this->assertEquals('number', $form['first_time_login_config_days']['#type']);
    $this->assertEquals(0, $form['first_time_login_config_days']['#min']);
    $this->assertTrue($form['first_time_login_config_days']['#required']);
    $this->assertEquals(120, $form['first_time_login_config_days']['#default_value']);

    $this->assertEquals('textfield', $form['first_time_login_new_user_message']['#type']);
    $this->assertTrue($form['first_time_login_new_user_message']['#required']);
    $this->assertEquals('Welcome new user', $form['first_time_login_new_user_message']['#default_value']);

    $this->assertEquals('textfield', $form['first_time_login_update_user_message']['#type']);
    $this->assertTrue($form['first_time_login_update_user_message']['#required']);
    $this->assertEquals('Please update your profile', $form['first_time_login_update_user_message']['#default_value']);
  }

  /**
   * Tests form validation with valid numeric input.
   *
   * @covers ::validateForm
   */
  public function testValidateFormWithValidInput() {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->method('getValue')
      ->with('first_time_login_config_days')
      ->willReturn('30');

    $form_state->expects($this->never())
      ->method('setErrorByName');

    $this->form->validateForm($form, $form_state);
  }

  /**
   * Tests form validation with invalid non-numeric input.
   *
   * @covers ::validateForm
   */
  public function testValidateFormWithInvalidInput() {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->method('getValue')
      ->with('first_time_login_config_days')
      ->willReturn('invalid');

    $form_state->expects($this->once())
      ->method('setErrorByName')
      ->with('first_time_login_config_days', 'Enter the number of days should be number.');

    $this->form->validateForm($form, $form_state);
  }

  /**
   * Tests the submitForm method.
   *
   * @covers ::submitForm
   */
  public function testSubmitForm() {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->method('getValue')
      ->willReturnMap([
        ['first_time_login_config_days', '60'],
        ['first_time_login_new_user_message', 'New welcome message'],
        ['first_time_login_update_user_message', 'New update message'],
      ]);

    $this->config->expects($this->exactly(3))
      ->method('set')
      ->withConsecutive(
        ['first_time_login_config_days', '60'],
        ['first_time_login_new_user_message', 'New welcome message'],
        ['first_time_login_update_user_message', 'New update message']
      )
      ->willReturnSelf();

    $this->config->expects($this->once())
      ->method('save');

    $this->form->submitForm($form, $form_state);
  }

} 