<?php

namespace Drupal\Tests\first_time_login\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Unit tests for First Time Login help functionality.
 *
 * @group first_time_login
 */
class FirstTimeLoginHelpTest extends UnitTestCase {

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock the string translation service.
    $string_translation = $this->createMock(TranslationInterface::class);
    $string_translation->method('translate')
      ->willReturnCallback(function ($string, $args = [], $options = []) {
        // Return the string with token replacements if needed
        if (!empty($args)) {
          return strtr($string, $args);
        }
        return $string;
      });

    // Set up a container with the string translation service.
    $container = new ContainerBuilder();
    $container->set('string_translation', $string_translation);
    
    // Make the container available to Drupal's static methods
    \Drupal::setContainer($container);
  }

  /**
   * Tests the help hook with valid route.
   *
   * @covers ::first_time_login_help
   */
  public function testHelpWithValidRoute() {
    $route_match = $this->createMock(RouteMatchInterface::class);
    
    // Include the module file to access the hook function.
    require_once __DIR__ . '/../../../first_time_login.module';
    
    $help_output = first_time_login_help('help.page.first_time_login', $route_match);
    
    $this->assertIsString($help_output);
    $this->assertNotEmpty($help_output);
    
    // Check that the output contains the expected HTML structure
    $this->assertStringContainsString('<h3>', $help_output);
    $this->assertStringContainsString('</h3>', $help_output);
    $this->assertStringContainsString('<p>', $help_output);
    $this->assertStringContainsString('<strong>', $help_output);
    
    // The function should return a non-empty string when called with the correct route
    $this->assertGreaterThan(50, strlen($help_output), 'Help output should be substantial');
  }

  /**
   * Tests the help hook with invalid/unsupported route.
   *
   * @covers ::first_time_login_help
   */
  public function testHelpWithInvalidRoute() {
    $route_match = $this->createMock(RouteMatchInterface::class);
    
    // Include the module file to access the hook function.
    require_once __DIR__ . '/../../../first_time_login.module';
    
    $help_output = first_time_login_help('some.other.route', $route_match);
    
    $this->assertNull($help_output);
  }

  /**
   * Tests that help output is properly structured HTML.
   *
   * @covers ::first_time_login_help
   */
  public function testHelpOutputStructure() {
    $route_match = $this->createMock(RouteMatchInterface::class);
    
    // Include the module file to access the hook function.
    require_once __DIR__ . '/../../../first_time_login.module';
    
    $help_output = first_time_login_help('help.page.first_time_login', $route_match);
    
    // Check for proper HTML structure.
    $this->assertStringContainsString('<h3>', $help_output);
    $this->assertStringContainsString('</h3>', $help_output);
    $this->assertStringContainsString('<p>', $help_output);
    $this->assertStringContainsString('</p>', $help_output);
    $this->assertStringContainsString('<strong>', $help_output);
    $this->assertStringContainsString('</strong>', $help_output);
  }

} 