<?php

namespace Drupal\fix_anonymous_nodes\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Drupal\user\Entity\User;
use Drupal\Core\Database\Database;

/**
 *
 */
class FixAnonymousNodes extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'fix_anonymous_nodes_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['target_uid'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Reassign to user'),
      '#description' => $this->t('Select the user to assign as author for all orphaned nodes (currently anonymous).'),
      '#target_type' => 'user',
      '#required' => TRUE,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Fix anonymous nodes'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $target_uid = $form_state->getValue('target_uid');

    if (!$target_uid || !User::load($target_uid)) {
      $this->messenger()->addError($this->t('Invalid user selected.'));
      return;
    }

    $connection = Database::getConnection();

    // Find all nodes whose uid is 0 (anonymous) or belongs to deleted users.
    $uids = $connection->select('node_field_data', 'n')
      ->fields('n', ['uid'])
      ->distinct()
      ->execute()
      ->fetchCol();

    $existing_uids = $connection->select('users_field_data', 'u')
      ->fields('u', ['uid'])
      ->execute()
      ->fetchCol();

    $orphaned_uids = array_diff($uids, $existing_uids);
    $orphaned_uids[] = 0;

    if (empty($orphaned_uids)) {
      $this->messenger()->addStatus($this->t('No orphaned nodes found.'));
      return;
    }

    // Get all affected nodes.
    $nids = $connection->select('node_field_data', 'n')
      ->fields('n', ['nid'])
      ->condition('uid', $orphaned_uids, 'IN')
      ->execute()
      ->fetchCol();

    if (empty($nids)) {
      $this->messenger()->addStatus($this->t('No orphaned nodes found.'));
      return;
    }

    $count = 0;
    foreach ($nids as $nid) {
      $node = Node::load($nid);
      if ($node) {
        $node->setOwnerId($target_uid);
        $node->save();
        $count++;
      }
    }

    $this->messenger()->addStatus($this->t('Reassigned @count nodes to user ID @uid.', [
      '@count' => $count,
      '@uid' => $target_uid,
    ]));
  }

}
