<?php

namespace Drupal\flat_menus;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\system\Entity\Menu;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;

/**
 * Provides method to flatten a menu's links.
 *
 * @package Drupal\flat_menus
 */
class Flattener {

  /**
   * The entity manager.
   */
  protected EntityTypeManagerInterface $entityManager;

  /**
   * The menu link tree service.
   */
  protected MenuLinkTreeInterface $menuLinkTree;

  /**
   * Constructs a new Flattener instance.
   */
  public function __construct(EntityTypeManagerInterface $entity_manager, MenuLinkTreeInterface $menu_link_tree) {
    $this->entityManager = $entity_manager;
    $this->menuLinkTree = $menu_link_tree;
  }

  /**
   * Flatten an entire menu.
   */
  public function flatten(Menu $menu): void {
    $weight = 0;
    
    // Load all top-level menu links for this menu
    $menu_links = $this->entityManager->getStorage('menu_link_content')->loadByProperties([
      'menu_name' => $menu->id(),
    ]);
    
    // Get only top-level items (those with no parent or parent is the menu itself)
    $top_level_links = [];
    foreach ($menu_links as $menu_link) {
      $parent = $menu_link->getParentId();
      if (empty($parent) || $parent === $menu->id() . ':') {
        $top_level_links[] = $menu_link;
      }
    }
    
    // Flatten each top-level item and all its children
    foreach ($top_level_links as $menu_link) {
      $this->flattenSubtree($menu_link, $weight);
    }
  }

  /**
   * Flatten a menu subtree from the given menu link.
   */
  public function flattenSubtree(MenuLinkContent $menu_link, int &$weight): void {
    // Update the given menu link.
    $menu_link->set('weight', $weight++);
    $menu_link->set('parent', '');
    $menu_link->save();

    // Get child menu links
    $child_links = $this->entityManager->getStorage('menu_link_content')->loadByProperties([
      'menu_name' => $menu_link->getMenuName(),
      'parent' => 'menu_link_content:' . $menu_link->uuid(),
    ]);

    // Recursively flatten each child subtree.
    foreach ($child_links as $child_link) {
      $this->flattenSubtree($child_link, $weight);
    }
  }

}