# Flexus Theme

Flexus is a component-based Drupal theme inspired on Mercury core theme, providing a modern and flexible starting point for site owners to build scalable and efficient websites using [Drupal Canvas](/project/canvas).

## Customizing

### Fonts & colors

To change the fonts or colors in `my_theme`, edit the `theme.css` file. Changes to `theme.css` do not require a CSS rebuild, but you may need to clear the cache.

### Adding a color palette

Flexus supports predefined color palettes that can be selected in the theme settings. Each palette defines colors for both light and dark modes. To add a new color palette:

1. **Add the palette definition** to `src/ColorPaletteService.php` in the `getPalettes()` method:

```php
'my-palette' => [
  'label' => 'My Palette',
  'description' => 'Description of the palette',
  'light' => [
    'colors' => [
      'primary' => 'oklch(0.5 0.2 250)',
      'primary-foreground' => 'oklch(1 0 0)',
      'secondary' => 'oklch(0.7 0.15 200)',
      'secondary-foreground' => 'oklch(0.2 0.05 260)',
      'accent' => 'oklch(0.6 0.2 300)',
      'accent-foreground' => 'oklch(0.2 0.05 260)',
      'muted' => 'oklch(0.95 0.01 250)',
      'muted-foreground' => 'oklch(0.4 0.05 260)',
    ],
    'swatches' => [
      'oklch(0.5 0.2 250)', // primary
      'oklch(0.7 0.15 200)', // secondary
      'oklch(0.6 0.2 300)', // accent
      'oklch(0.95 0.01 250)', // muted
      'oklch(1 0 0)', // background
    ],
  ],
  'dark' => [
    'colors' => [
      'primary' => 'oklch(0.55 0.22 250)',
      'primary-foreground' => 'oklch(0.1 0.02 260)',
      'secondary' => 'oklch(0.65 0.18 200)',
      'secondary-foreground' => 'oklch(0.1 0.02 260)',
      'accent' => 'oklch(0.65 0.22 300)',
      'accent-foreground' => 'oklch(0.1 0.02 260)',
      'muted' => 'oklch(0.25 0.03 250)',
      'muted-foreground' => 'oklch(0.85 0.01 250)',
    ],
    'swatches' => [
      'oklch(0.55 0.22 250)', // primary
      'oklch(0.65 0.18 200)', // secondary
      'oklch(0.65 0.22 300)', // accent
      'oklch(0.25 0.03 250)', // muted
      'oklch(0.1 0.02 260)', // background
    ],
  ],
],
```

   **Note**: The `swatches` array should contain exactly 5 colors that represent the palette visually. These are displayed in the theme settings UI. The swatches should match the actual color variables (primary, secondary, accent, muted, and background) for consistency.

2. **Create the CSS file** at `src/palettes/my-palette.css`:

```css
/**
 * @file
 * My Palette color palette CSS variables.
 */

:root {
  --primary: oklch(0.5 0.2 250);
  --primary-foreground: oklch(1 0 0);
  --secondary: oklch(0.7 0.15 200);
  --secondary-foreground: oklch(0.2 0.05 260);
  --accent: oklch(0.6 0.2 300);
  --accent-foreground: oklch(0.2 0.05 260);
  --muted: oklch(0.95 0.01 250);
  --muted-foreground: oklch(0.4 0.05 260);
}

.dark {
  --primary: oklch(0.55 0.22 250);
  --primary-foreground: oklch(0.1 0.02 260);
  --secondary: oklch(0.65 0.18 200);
  --secondary-foreground: oklch(0.1 0.02 260);
  --accent: oklch(0.65 0.22 300);
  --accent-foreground: oklch(0.1 0.02 260);
  --muted: oklch(0.25 0.03 250);
  --muted-foreground: oklch(0.85 0.01 250);
}
```

3. **Add the library** to `flexus.libraries.yml`:

```yaml
color-palette-my-palette:
  version: VERSION
  css:
    theme:
      src/palettes/my-palette.css:
        preprocess: false
```

4. **Update the config schema** in `config/schema/flexus.schema.yml` to add your palette key to the allowed choices:

```yaml
color_palette:
  type: string
  label: Color palette
  constraints:
    Choice:
      - default
      - vibrant
      - warm
      - earth
      - ocean
      - navy
      - my-palette  # Add your palette key here
```

After completing these steps, clear the Drupal cache and your new palette will appear in the theme settings at `/admin/appearance/settings/flexus`.

**Color format**: Colors should be specified in `oklch()` format for better color consistency and perceptual uniformity. You can convert hex colors to oklch using online tools like [oklch.com](https://oklch.com) or [vis4.net/labs/monochromatic](https://vis4.net/labs/monochromatic).

### Custom components

Flexus uses [single-directory components](https://www.drupal.org/docs/develop/theming-drupal/using-single-directory-components) and comes with a variety of commonly used components. You can add new components and modify existing ones, but be sure to rebuild the CSS when you make changes.

## Building CSS

Flexus uses [Tailwind](https://tailwindcss.com) to simplify styling by using classes to compose designs directly in the markup.

If you want to customize the Tailwind-generated CSS, install the development tooling dependencies by running the following command in your theme's directory:

```shell
npm install
```

If you modify CSS files or classes in a Twig template, you need to rebuild the CSS:

```bash
npm run build
```

For development, you can watch for changes and automatically rebuild the CSS:

```bash
npm run dev
```

## Code Formatting

Flexus uses [Prettier](https://prettier.io) to automatically format code for consistency. The project is configured with plugins for Tailwind CSS and Twig templates.

For the best experience, [set up Prettier in your editor](https://prettier.io/docs/editors) to automatically format files on save.

To format all files in the project:

```bash
npm run format
```

To check if files are formatted correctly without making changes:

```bash
npm run format:check
```

**Note**: Some files are excluded from formatting via `.prettierignore`, such as Drupal's `html.html.twig` template, which contains placeholder tokens that break Prettier's HTML parsing.

## Component JavaScript

`lib/component.js` has two classes you can use to nicely encapsulate your component JS without pasting all the `Drupal.behaviors.componentName` boilerplate into every file. The steps are:

1. Extend the `ComponentInstance` class to a new class with the code for your component.
2. Create a new instance of the `ComponentType` class to automatically activate all the component instances on that page.

For example, here's a stub of `accordion.js`:

```js
import { ComponentType, ComponentInstance } from "../../lib/component.js";

// Make a new class with the code for our component.
//
// In every method of this class, `this.el` is an HTMLElement object of
// the component container, whose selector you provide below. You don't
// have an array of elements that you have to `.forEach()` over yourself;
// the ComponentType class handles all that for you.
class Accordion extends ComponentInstance {
  // Every subclass must have an `init` method to activate the component.
  init() {
    this.el.querySelector(".accordion--content").classList.toggle("visible");
    this.el.addClass("js");
  }

  // You may also implement a `remove()` method to clean up when a component is
  // about to be removed from the document. This will be invoked during the
  // `detach()` method of the Drupal behavior.

  // You can create as many other methods as you want; in all of them,
  // `this.el` represents the single instance of the component. Any other
  // properties you add to `this` will be isolated to that one instance
  // as well.
}

// Now we instantiate ComponentType to find the component elements and run
// our script.
new ComponentType(
  // First argument: The subclass of ComponentInstance we just created above.
  Accordion,
  // Second argument: A camel-case unique ID for the behavior (and for `once()`
  // if applicable).
  "accordion",
  // Third argument: A selector for `querySelectorAll()`. All matching elements
  // on the page get their own instance of the subclass you created, each of
  // which has `this.el` pointing to one of those matches.
  ".accordion",
);
```

This is all the code required to be in each component. The ComponentType instance handles finding the elements, running them through `once` if available, and adding them to `Drupal.behaviors`.

All the objects created this way will be stored in a global variable so you can do stuff with them later. Since the `namespace` variable at the top of component.js is `flexusComponents`, you would find the Accordion's ComponentType instance at `window.flexusComponents.accordion`.

Furthermore, `window.flexusComponents.accordion.instances` is an array of all the ComponentInstance objects, and `window.flexusComponents.accordion.elements` is an array of all the component container elements.

## Known issues

Canvas's code components are currently not compatible with Tailwind-based themes like Flexus, and creating a code component will break Flexus's styling. This will be fixed in [#3549628], but for now, here's how to work around it:

1. In Canvas's in-browser code editor, open the Global CSS tab.
2. Paste the contents of your custom theme's `theme.css` into the code editor. It must be at the top.
3. Paste the contents of your custom theme's `main.css` into the code editor, removing all the `@import` statements at the top first. It must come _after_ the contents of `theme.css`.
4. Save the global CSS.




