<?php

/**
 * @file
 * Helper class for managing color palettes.
 */

namespace Drupal\flexus;

/**
 * Helper class for managing color palettes.
 */
class ColorPaletteService {

  /**
   * Get all available color palettes.
   *
   * @return array
   *   Array of color palettes with their definitions.
   */
  public static function getPalettes(): array {
    $palettes = [
      'default' => [
        'label' => 'Default',
        'description' => 'The default color scheme',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.547 0.246 262.866)',
            'primary-foreground' => 'oklch(1 0 0)',
            'secondary' => 'oklch(0.9219 0 0)',
            'secondary-foreground' => 'oklch(0.28 0.041 260.329)',
            'accent' => 'oklch(0.932 0.032 255.585)',
            'accent-foreground' => 'oklch(0.38 0.145 265.465)',
            'muted' => 'oklch(0.967 0.003 264.542)',
            'muted-foreground' => 'oklch(0.373 0.031 259.733)',
          ],
          'swatches' => [
            'oklch(0.547 0.246 262.866)', // primary
            'oklch(0.9219 0 0)', // secondary
            'oklch(0.932 0.032 255.585)', // accent
            'oklch(0.967 0.003 264.542)', // muted
            'oklch(1 0 0)', // background/foreground
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.547 0.246 262.866)',
            'primary-foreground' => 'oklch(1 0 0)',
            'secondary' => 'oklch(0.28 0.041 260.329)',
            'secondary-foreground' => 'oklch(0.9219 0 0)',
            'accent' => 'oklch(0.932 0.032 255.585)',
            'accent-foreground' => 'oklch(0.13 0.043 265.132)',
            'muted' => 'oklch(0.373 0.031 259.733)',
            'muted-foreground' => 'oklch(0.967 0.003 264.542)',
          ],
          'swatches' => [
            'oklch(0.547 0.246 262.866)', // primary
            'oklch(0.28 0.041 260.329)', // secondary
            'oklch(0.932 0.032 255.585)', // accent
            'oklch(0.373 0.031 259.733)', // muted
            'oklch(0.13 0.043 265.132)', // background
          ],
        ],
      ],
      'vibrant' => [
        'label' => 'Vibrant',
        'description' => 'Bright and energetic colors',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.7 0.2 90)',
            'primary-foreground' => 'oklch(0.2 0.05 260)',
            'secondary' => 'oklch(0.75 0.2 30)',
            'secondary-foreground' => 'oklch(0.2 0.05 260)',
            'accent' => 'oklch(0.7 0.2 320)',
            'accent-foreground' => 'oklch(1 0 0)',
            'muted' => 'oklch(0.85 0.1 280)',
            'muted-foreground' => 'oklch(0.3 0.05 260)',
          ],
          'swatches' => [
            'oklch(0.7 0.2 90)', // primary
            'oklch(0.75 0.2 30)', // secondary
            'oklch(0.7 0.2 320)', // accent
            'oklch(0.85 0.1 280)', // muted
            'oklch(0.2 0.05 260)', // foreground
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.65 0.22 90)',
            'primary-foreground' => 'oklch(0.1 0.02 260)',
            'secondary' => 'oklch(0.7 0.22 30)',
            'secondary-foreground' => 'oklch(0.1 0.02 260)',
            'accent' => 'oklch(0.65 0.22 320)',
            'accent-foreground' => 'oklch(0.95 0 0)',
            'muted' => 'oklch(0.3 0.08 280)',
            'muted-foreground' => 'oklch(0.85 0.1 280)',
          ],
          'swatches' => [
            'oklch(0.65 0.22 90)', // primary
            'oklch(0.7 0.22 30)', // secondary
            'oklch(0.65 0.22 320)', // accent
            'oklch(0.3 0.08 280)', // muted
            'oklch(0.1 0.02 260)', // background
          ],
        ],
      ],
      'warm' => [
        'label' => 'Warm',
        'description' => 'Warm sunset-inspired colors',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.55 0.25 15)',
            'primary-foreground' => 'oklch(1 0 0)',
            'secondary' => 'oklch(0.7 0.2 50)',
            'secondary-foreground' => 'oklch(0.2 0.05 260)',
            'accent' => 'oklch(0.75 0.2 90)',
            'accent-foreground' => 'oklch(0.2 0.05 260)',
            'muted' => 'oklch(0.65 0.15 120)',
            'muted-foreground' => 'oklch(0.3 0.05 260)',
          ],
          'swatches' => [
            'oklch(0.55 0.25 15)', // primary
            'oklch(0.7 0.2 50)', // secondary
            'oklch(0.75 0.2 90)', // accent
            'oklch(0.65 0.15 120)', // muted
            'oklch(1 0 0)', // background
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.6 0.25 15)',
            'primary-foreground' => 'oklch(0.1 0.02 260)',
            'secondary' => 'oklch(0.65 0.22 50)',
            'secondary-foreground' => 'oklch(0.1 0.02 260)',
            'accent' => 'oklch(0.7 0.22 90)',
            'accent-foreground' => 'oklch(0.1 0.02 260)',
            'muted' => 'oklch(0.35 0.12 120)',
            'muted-foreground' => 'oklch(0.7 0.15 120)',
          ],
          'swatches' => [
            'oklch(0.6 0.25 15)', // primary
            'oklch(0.65 0.22 50)', // secondary
            'oklch(0.7 0.22 90)', // accent
            'oklch(0.35 0.12 120)', // muted
            'oklch(0.1 0.02 260)', // background
          ],
        ],
      ],
      'earth' => [
        'label' => 'Earth',
        'description' => 'Natural earth tones',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.5 0.15 50)',
            'primary-foreground' => 'oklch(0.95 0.01 50)',
            'secondary' => 'oklch(0.2 0.05 260)',
            'secondary-foreground' => 'oklch(0.95 0.01 50)',
            'accent' => 'oklch(0.6 0.1 150)',
            'accent-foreground' => 'oklch(0.2 0.05 260)',
            'muted' => 'oklch(0.85 0.05 60)',
            'muted-foreground' => 'oklch(0.4 0.05 260)',
          ],
          'swatches' => [
            'oklch(0.5 0.15 50)', // primary
            'oklch(0.2 0.05 260)', // secondary
            'oklch(0.6 0.1 150)', // accent
            'oklch(0.85 0.05 60)', // muted
            'oklch(0.95 0.01 50)', // background
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.55 0.18 50)',
            'primary-foreground' => 'oklch(0.1 0.02 260)',
            'secondary' => 'oklch(0.25 0.06 260)',
            'secondary-foreground' => 'oklch(0.9 0.02 50)',
            'accent' => 'oklch(0.55 0.12 150)',
            'accent-foreground' => 'oklch(0.1 0.02 260)',
            'muted' => 'oklch(0.3 0.05 60)',
            'muted-foreground' => 'oklch(0.8 0.05 60)',
          ],
          'swatches' => [
            'oklch(0.55 0.18 50)', // primary
            'oklch(0.25 0.06 260)', // secondary
            'oklch(0.55 0.12 150)', // accent
            'oklch(0.3 0.05 60)', // muted
            'oklch(0.1 0.02 260)', // background
          ],
        ],
      ],
      'ocean' => [
        'label' => 'Ocean',
        'description' => 'Cool ocean-inspired colors',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.45 0.15 200)',
            'primary-foreground' => 'oklch(1 0 0)',
            'secondary' => 'oklch(0.6 0.2 160)',
            'secondary-foreground' => 'oklch(0.2 0.05 260)',
            'accent' => 'oklch(0.7 0.15 140)',
            'accent-foreground' => 'oklch(0.2 0.05 260)',
            'muted' => 'oklch(0.95 0.01 90)',
            'muted-foreground' => 'oklch(0.4 0.05 260)',
          ],
          'swatches' => [
            'oklch(0.45 0.15 200)', // primary
            'oklch(0.6 0.2 160)', // secondary
            'oklch(0.7 0.15 140)', // accent
            'oklch(0.95 0.01 90)', // muted
            'oklch(1 0 0)', // background
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.5 0.18 200)',
            'primary-foreground' => 'oklch(0.1 0.02 260)',
            'secondary' => 'oklch(0.55 0.22 160)',
            'secondary-foreground' => 'oklch(0.1 0.02 260)',
            'accent' => 'oklch(0.65 0.18 140)',
            'accent-foreground' => 'oklch(0.1 0.02 260)',
            'muted' => 'oklch(0.25 0.05 200)',
            'muted-foreground' => 'oklch(0.7 0.15 140)',
          ],
          'swatches' => [
            'oklch(0.5 0.18 200)', // primary
            'oklch(0.55 0.22 160)', // secondary
            'oklch(0.65 0.18 140)', // accent
            'oklch(0.25 0.05 200)', // muted
            'oklch(0.1 0.02 260)', // background
          ],
        ],
      ],
      'navy' => [
        'label' => 'Navy',
        'description' => 'Professional navy blue with light background',
        'light' => [
          'colors' => [
            'primary' => 'oklch(0.18 0.08 260)',
            'primary-foreground' => 'oklch(0.97 0 0)',
            'secondary' => 'oklch(0.5 0.1 260)',
            'secondary-foreground' => 'oklch(0.97 0 0)',
            'accent' => 'oklch(0.55 0.15 250)',
            'accent-foreground' => 'oklch(0.18 0.08 260)',
            'muted' => 'oklch(0.92 0.002 260)',
            'muted-foreground' => 'oklch(0.35 0.02 260)',
          ],
          'swatches' => [
            'oklch(0.18 0.08 260)', // primary
            'oklch(0.5 0.1 260)', // secondary
            'oklch(0.55 0.15 250)', // accent
            'oklch(0.92 0.002 260)', // muted
            'oklch(0.965 0.002 0)', // background
          ],
        ],
        'dark' => [
          'colors' => [
            'primary' => 'oklch(0.6 0.12 260)',
            'primary-foreground' => 'oklch(0.1 0.02 260)',
            'secondary' => 'oklch(0.45 0.1 260)',
            'secondary-foreground' => 'oklch(0.95 0 0)',
            'accent' => 'oklch(0.65 0.18 250)',
            'accent-foreground' => 'oklch(0.1 0.02 260)',
            'muted' => 'oklch(0.25 0.03 260)',
            'muted-foreground' => 'oklch(0.85 0.01 260)',
          ],
          'swatches' => [
            'oklch(0.6 0.12 260)', // primary
            'oklch(0.45 0.1 260)', // secondary
            'oklch(0.65 0.18 250)', // accent
            'oklch(0.25 0.03 260)', // muted
            'oklch(0.1 0.02 260)', // background
          ],
        ],
      ],
    ];

    // Add custom palette from config if it exists.
    $config = \Drupal::config('flexus.settings');
    $custom_colors = $config->get('custom_palette_colors');
    if ($custom_colors) {
      $palettes['custom'] = self::buildCustomPalette($custom_colors);
    }
    else {
      // Provide default custom palette structure.
      $palettes['custom'] = self::getDefaultCustomPalette();
    }

    return $palettes;
  }

  /**
   * Get default custom palette structure.
   *
   * @return array
   *   Default custom palette structure.
   */
  public static function getDefaultCustomPalette(): array {
    return [
      'label' => 'Custom',
      'description' => 'Create your own color palette using HEX colors',
      'light' => [
        'colors' => [
          'primary' => '#3b82f6',
          'primary-foreground' => '#ffffff',
          'secondary' => '#f3f4f6',
          'secondary-foreground' => '#1f2937',
          'accent' => '#e0e7ff',
          'accent-foreground' => '#3730a3',
          'muted' => '#f9fafb',
          'muted-foreground' => '#6b7280',
        ],
        'swatches' => [
          '#3b82f6', // primary
          '#f3f4f6', // secondary
          '#e0e7ff', // accent
          '#f9fafb', // muted
          '#ffffff', // background
        ],
      ],
      'dark' => [
        'colors' => [
          'primary' => '#60a5fa',
          'primary-foreground' => '#1f2937',
          'secondary' => '#374151',
          'secondary-foreground' => '#f9fafb',
          'accent' => '#6366f1',
          'accent-foreground' => '#ffffff',
          'muted' => '#1f2937',
          'muted-foreground' => '#d1d5db',
        ],
        'swatches' => [
          '#60a5fa', // primary
          '#374151', // secondary
          '#6366f1', // accent
          '#1f2937', // muted
          '#111827', // background
        ],
      ],
    ];
  }

  /**
   * Build custom palette from stored colors.
   *
   * @param array $custom_colors
   *   Array of custom colors from config.
   *
   * @return array
   *   Custom palette structure.
   */
  protected static function buildCustomPalette(array $custom_colors): array {
    $light_colors = $custom_colors['light'] ?? [];
    $dark_colors = $custom_colors['dark'] ?? [];

    // Build swatches from colors.
    $light_swatches = [
      $light_colors['primary'] ?? '#3b82f6',
      $light_colors['secondary'] ?? '#f3f4f6',
      $light_colors['accent'] ?? '#e0e7ff',
      $light_colors['muted'] ?? '#f9fafb',
      $light_colors['primary-foreground'] ?? '#ffffff',
    ];

    $dark_swatches = [
      $dark_colors['primary'] ?? '#60a5fa',
      $dark_colors['secondary'] ?? '#374151',
      $dark_colors['accent'] ?? '#6366f1',
      $dark_colors['muted'] ?? '#1f2937',
      $dark_colors['primary-foreground'] ?? '#111827',
    ];

    return [
      'label' => 'Custom',
      'description' => 'Your custom color palette',
      'light' => [
        'colors' => $light_colors,
        'swatches' => $light_swatches,
      ],
      'dark' => [
        'colors' => $dark_colors,
        'swatches' => $dark_swatches,
      ],
    ];
  }

  /**
   * Get the currently selected palette.
   *
   * @return string
   *   The palette key.
   */
  public static function getSelectedPalette(): string {
    return \Drupal\Component\Utility\DeprecationHelper::backwardsCompatibleCall(
      \Drupal::VERSION,
      '11.3',
      fn () => \Drupal::service('Drupal\Core\Extension\ThemeSettingsProvider')->getSetting('color_palette'),
      fn () => theme_get_setting('color_palette'),
    ) ?? 'default';
  }

  /**
   * Get the colors for the selected palette.
   *
   * @param string|null $palette_key
   *   Optional palette key. If not provided, uses the selected palette.
   * @param string $scheme
   *   The color scheme: 'light' or 'dark'.
   *
   * @return array
   *   Array of color values for the palette.
   */
  public static function getPaletteColors(?string $palette_key = NULL, string $scheme = 'light'): array {
    $palette_key = $palette_key ?? self::getSelectedPalette();
    $palettes = self::getPalettes();
    $palette = $palettes[$palette_key] ?? $palettes['default'];
    return $palette[$scheme]['colors'] ?? $palette['light']['colors'];
  }

  /**
   * Get custom palette colors from theme settings.
   *
   * @return array
   *   Array of custom colors or empty array.
   */
  public static function getCustomPaletteColors(): array {
    return \Drupal\Component\Utility\DeprecationHelper::backwardsCompatibleCall(
      \Drupal::VERSION,
      '11.3',
      fn () => \Drupal::service('Drupal\Core\Extension\ThemeSettingsProvider')->getSetting('custom_palette_colors'),
      fn () => theme_get_setting('custom_palette_colors'),
    ) ?? [];
  }

  /**
   * Generate CSS for custom color palette.
   *
   * @param array $custom_colors
   *   Array of custom colors with 'light' and 'dark' keys.
   *
   * @return string
   *   Generated CSS string.
   */
  public static function generateCustomPaletteCss(array $custom_colors): string {
    $light_colors = $custom_colors['light'] ?? [];
    $dark_colors = $custom_colors['dark'] ?? [];

    $css = ":root {\n";
    foreach ($light_colors as $key => $value) {
      $css_key = '--' . str_replace('_', '-', $key);
      $css .= "  {$css_key}: {$value};\n";
    }
    $css .= "}\n\n";

    $css .= ".dark {\n";
    foreach ($dark_colors as $key => $value) {
      $css_key = '--' . str_replace('_', '-', $key);
      $css .= "  {$css_key}: {$value};\n";
    }
    $css .= "}\n";

    return $css;
  }

}

