<?php

/**
 * @file
 * Helper class for managing font families.
 */

namespace Drupal\flexus;

/**
 * Helper class for managing font families.
 */
class FontService {

  /**
   * Get all available font families.
   *
   * @return array
   *   Array of font families with their definitions.
   */
  public static function getFonts(): array {
    return [
      'outfit' => [
        'label' => 'Outfit',
        'description' => 'Modern sans-serif font with clean lines',
        'font-family' => '"Outfit", "Helvetica Neue", Arial, Helvetica, sans-serif',
        'type' => 'sans-serif',
      ],
      'source-serif' => [
        'label' => 'Source Serif 4',
        'description' => 'Elegant serif font for body text',
        'font-family' => '"SourceSerif4", ui-serif, Georgia, Cambria, "Times New Roman", Times, serif',
        'type' => 'serif',
      ],
    ];
  }

  /**
   * Get the currently selected font family.
   *
   * @return string
   *   The font key.
   */
  public static function getSelectedFont(): string {
    return \Drupal\Component\Utility\DeprecationHelper::backwardsCompatibleCall(
      \Drupal::VERSION,
      '11.3',
      fn () => \Drupal::service('Drupal\Core\Extension\ThemeSettingsProvider')->getSetting('font_family'),
      fn () => theme_get_setting('font_family'),
    ) ?? 'outfit';
  }

  /**
   * Get the font family CSS value for the selected font.
   *
   * @param string|null $font_key
   *   Optional font key. If not provided, uses the selected font.
   *
   * @return string
   *   The font-family CSS value.
   */
  public static function getFontFamily(?string $font_key = NULL): string {
    $font_key = $font_key ?? self::getSelectedFont();
    $fonts = self::getFonts();
    $font = $fonts[$font_key] ?? $fonts['outfit'];
    return $font['font-family'];
  }

  /**
   * Generate CSS for the selected font family.
   *
   * @param string|null $font_key
   *   Optional font key. If not provided, uses the selected font.
   *
   * @return string
   *   Generated CSS string.
   */
  public static function generateFontCss(?string $font_key = NULL): string {
    $font_key = $font_key ?? self::getSelectedFont();
    $font_family = self::getFontFamily($font_key);

    $css = ":root {\n";
    $css .= "  --font-sans: {$font_family};\n";
    $css .= "  --default-font-family: {$font_family};\n";
    $css .= "}\n\n";

    $css .= ".dark {\n";
    $css .= "  --font-sans: {$font_family};\n";
    $css .= "  --default-font-family: {$font_family};\n";
    $css .= "}\n";

    return $css;
  }

}

