<?php

declare(strict_types=1);

namespace Drupal\flexus;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;

/**
 * Implements internal callbacks to help with rendering certain SDCs.
 */
final class RenderCallbacks implements TrustedCallbackInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|null
   */
  private static ?EntityTypeManagerInterface $entityTypeManager = NULL;

  /**
   * The entity form builder.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface|null
   */
  private static ?EntityFormBuilderInterface $entityFormBuilder = NULL;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|null
   */
  private static ?ConfigFactoryInterface $configFactory = NULL;

  /**
   * Gets the entity type manager service.
   *
   * @return \Drupal\Core\Entity\EntityTypeManagerInterface
   *   The entity type manager.
   */
  private static function getEntityTypeManager(): EntityTypeManagerInterface {
    if (self::$entityTypeManager === NULL) {
      self::$entityTypeManager = \Drupal::service('entity_type.manager');
    }
    return self::$entityTypeManager;
  }

  /**
   * Gets the entity form builder service.
   *
   * @return \Drupal\Core\Entity\EntityFormBuilderInterface
   *   The entity form builder.
   */
  private static function getEntityFormBuilder(): EntityFormBuilderInterface {
    if (self::$entityFormBuilder === NULL) {
      self::$entityFormBuilder = \Drupal::service('entity.form_builder');
    }
    return self::$entityFormBuilder;
  }

  /**
   * Gets the config factory service.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   The config factory.
   */
  private static function getConfigFactory(): ConfigFactoryInterface {
    if (self::$configFactory === NULL) {
      self::$configFactory = \Drupal::service('config.factory');
    }
    return self::$configFactory;
  }

  /**
   * Builds the contact form wrapper classes based on component props.
   *
   * @param array $props
   *   The component props.
   *
   * @return string
   *   The class string.
   */
  private static function buildContactFormClasses(array $props): string {
    $classes = ['flexus-form'];

    // Input style.
    $input_style = $props['input_style'] ?? 'default';
    if ($input_style === 'boxed') {
      $classes[] = 'flexus-form--style-boxed';
    }
    else {
      $classes[] = 'flexus-form--style-default';
    }

    // Border radius.
    $border_radius = $props['input_border_radius'] ?? 'none';
    $classes[] = match ($border_radius) {
      'small' => 'flexus-form--radius-small',
      'medium' => 'flexus-form--radius-medium',
      'large' => 'flexus-form--radius-large',
      default => 'flexus-form--radius-none',
    };

    // Input background color.
    $input_bg = $props['input_background_color'] ?? '';
    if (!empty($input_bg) && in_array($input_bg, ['primary', 'secondary', 'accent', 'muted', 'white'])) {
      $classes[] = 'flexus-form--input-bg-' . $input_bg;
    }

    // Label color.
    $label_color = $props['label_color'] ?? '';
    if (!empty($label_color) && in_array($label_color, ['primary', 'secondary', 'accent', 'muted'])) {
      $classes[] = 'flexus-form--label-' . $label_color;
    }

    // Hide labels.
    $hide_labels = $props['hide_labels'] ?? FALSE;
    if ($hide_labels) {
      $classes[] = 'flexus-form--hide-labels';
    }

    return implode(' ', $classes);
  }

  /**
   * Pre-render callback for SDC elements.
   *
   * @param array $element
   *   The element being rendered.
   *
   * @return array
   *   The modified element.
   */
  public static function preRenderComponent(array $element): array {
    // In the hero-blog component, convert a UNIX timestamp into a string that
    // can be passed to strtotime() by Twig's date filter.
    if ($element['#component'] === 'flexus:hero-blog' && isset($element['#props']['date']) && is_int($element['#props']['date'])) {
      $element['#props']['date'] = date('Y-m-d', $element['#props']['date']);
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return ['preRenderComponent'];
  }

}
