/**
 * @file
 * JavaScript for the color palette visual selector.
 */

(function (Drupal, drupalSettings) {
  'use strict';

  /**
   * Initialize the color palette visual selector.
   */
  Drupal.behaviors.flexusColorPaletteSelector = {
    attach: function (context, settings) {
      const visualContainer = document.querySelector(
        '#flexus-color-palette-visual',
      );
      const selectElement = document.querySelector(
        'select.flexus-color-palette-select',
      );

      if (!visualContainer || !selectElement) {
        return;
      }

      const currentPalette =
        settings.flexus?.currentPalette || 'default';
      const palettes = settings.flexus?.colorPalettes || {};

      // Mark the current palette as selected.
      const currentOption = visualContainer.querySelector(
        `[data-palette-key="${currentPalette}"]`,
      );
      if (currentOption) {
        currentOption.classList.add('selected');
      }

      // Add click handlers to palette options.
      const paletteOptions = visualContainer.querySelectorAll(
        '.color-palette-option',
      );
      paletteOptions.forEach((option) => {
        option.addEventListener('click', function () {
          const paletteKey = this.getAttribute('data-palette-key');

          // Update select element.
          selectElement.value = paletteKey;

          // Update visual selection.
          paletteOptions.forEach((opt) => opt.classList.remove('selected'));
          this.classList.add('selected');

          // Trigger change event on select to ensure form submission works.
          const changeEvent = new Event('change', { bubbles: true });
          selectElement.dispatchEvent(changeEvent);
        });
      });

      // Sync select changes to visual selector.
      selectElement.addEventListener('change', function () {
        const selectedKey = this.value;
        paletteOptions.forEach((opt) => {
          if (opt.getAttribute('data-palette-key') === selectedKey) {
            opt.classList.add('selected');
          } else {
            opt.classList.remove('selected');
          }
        });

        // Show/hide custom palette settings.
        const customSettings = document.querySelector('.flexus-custom-palette-settings');
        if (customSettings) {
          if (selectedKey === 'custom') {
            customSettings.style.display = 'block';
          } else {
            customSettings.style.display = 'none';
          }
        }
      });

      // Initial state: show/hide custom palette settings.
      const customSettings = document.querySelector('.flexus-custom-palette-settings');
      if (customSettings) {
        if (currentPalette === 'custom') {
          customSettings.style.display = 'block';
        } else {
          customSettings.style.display = 'none';
        }
      }
    },
  };
})(Drupal, drupalSettings);

