/**
 * @file
 * Menu toggle functionality for multi-level menus.
 */

(function (Drupal, once) {
  'use strict';

  /**
   * Initialize menu toggle buttons for mobile navigation.
   */
  function initMenuToggles(context) {
    const menuToggles = once('flexusMenuToggle', '[data-menu-toggle]', context);

    menuToggles.forEach((toggle) => {
      const menuItem = toggle.closest('.menu-item--has-children');
      if (!menuItem) return;

      const submenu = menuItem.querySelector('ul[data-level]');
      const link = menuItem.querySelector('a[aria-expanded]');

      if (!submenu) return;

      toggle.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();

        const isOpen = menuItem.getAttribute('data-menu-open') === 'true';
        const newState = !isOpen;

        menuItem.setAttribute('data-menu-open', newState);
        if (link) {
          link.setAttribute('aria-expanded', newState);
        }
        toggle.setAttribute('aria-expanded', newState);
      });
    });
  }

  /**
   * Handle keyboard navigation for menu items.
   */
  function initKeyboardNavigation(context) {
    const menuItems = once('flexusMenuKeyboard', '.menu-item--has-children > div > a', context);

    menuItems.forEach((link) => {
      link.addEventListener('keydown', (e) => {
        // On Enter or Space, toggle submenu on mobile
        if ((e.key === 'Enter' || e.key === ' ') && window.innerWidth < 768) {
          const menuItem = link.closest('.menu-item--has-children');
          const toggle = menuItem?.querySelector('[data-menu-toggle]');
          if (toggle) {
            e.preventDefault();
            toggle.click();
          }
        }
      });
    });
  }

  /**
   * Drupal behavior for menu functionality.
   */
  Drupal.behaviors.flexusMenu = {
    attach: function (context) {
      initMenuToggles(context);
      initKeyboardNavigation(context);
    }
  };

})(Drupal, once);

