<?php

declare(strict_types=1);

namespace Drupal\flickr_integration_suite_filter\Plugin\Filter;

use Drupal\Component\Render\MarkupInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\flickr_integration_suite\FlickrIntegrationSuiteApiProvider;
use Drupal\flickr_integration_suite_filter\Helper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a filter to insert Flickr photos.
 *
 * @Filter(
 *   id = "flickr_integration_suite_filter",
 *   title = @Translation("Embed Flickr Photo"),
 *   description = @Translation("Allow users to embed a photo from Flickr website in an editable content area."),
 *   type = Drupal\filter\Plugin\FilterInterface::TYPE_MARKUP_LANGUAGE,
 *   settings = {
 *     "flickr_filter_imagesize" = 200,
 *   },
 * )
 */
final class FlickrIntegrationSuiteFilter extends FilterBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new FlickrIntegrationSuiteFilter instance.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly Helper $helper,
    private readonly FlickrIntegrationSuiteApiProvider $flickrIntegrationSuiteApiProvider,
    private readonly RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('flickr_integration_suite_filter.helper'),
      $container->get('flickr_integration_suite.api_provider'),
      $container->get('renderer'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $options = [];
    $sizes   = $this->helper->photoSizes();

    foreach ($sizes as $key => $size) {
      $options[$key] = $size['description']->render();
    }

    $form['flickr_filter_default_size'] = [
      '#type' => 'select',
      '#title' => $this->t('Default size for a single photo'),
      '#default_value' => $this->settings['flickr_filter_default_size'] ?? 's',
      '#options' => $options,
      '#description' => $this->t("A default Flickr size to use if no size is specified, for example [flickr-photo:id=3711935987].<br />TAKE CARE, the c size (800px) is missing on Flickr images uploaded before March 1, 2012!"),
    ];

    $form['flickr_filter_caption'] = [
      '#type' => 'select',
      '#title' => $this->t('Display captions for every Flickr photo'),
      '#required' => TRUE,
      '#default_value' => $this->settings['flickr_filter_caption'] ?? 0,
      '#description' => $this->t("If selected, flickr photos will display caption."),
      '#options' => [0 => 'No', 1 => 'Yes'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    $text = preg_replace_callback('/\[flickr-photo:(.+?)\]/', self::callbackPhoto(...), $text);
    $text = preg_replace_callback('/\[flickr-photoset:(.+?)\]/', self::callbackPhotosets(...), $text);
    return new FilterProcessResult($text);
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE): TranslatableMarkup {
    if ($long) {
      return $this->t(
        'Embed Flickr photos using the @embed syntax. Specifying image size is optional - if omitted, the %filter’s default dimensions will automatically apply.',
        [
          '@embed' => '[flickr-photo:id=<photo_id>, size=<imagesize>]',
          '%filter' => 'Embed Flickr Photo',
        ]
      );
    }
    else {
      return $this->t('Embed Flickr Photo using @embed.',
        [
          '@embed' => '[flickr-photo:id=<photo_id>, size=<imagesize>]',
        ]
      );
    }
  }

  /**
   * Filter callback for a photo.
   */
  public function callbackPhoto($matches): MarkupInterface|string {
    [$config, $attribs] = $this->helper->splitConfig($matches[1]);

    if (isset($config['id'])) {
      $photo = $this->flickrIntegrationSuiteApiProvider->photosGetInfo($config['id']);

      if ($photo['stat'] == 'ok') {
        $photo = $photo['photo'];

        if (!isset($config['size'])) {
          $config['size'] = $this->settings['flickr_filter_default_size'];
        }

        if (!isset($config['caption'])) {
          $config['caption'] = $this->settings['flickr_filter_caption'];
        }

        switch ($config['size']) {
          case "x":
          case "y":
            $this->messenger()->addError($this->t("Do not use a slideshow for a single image."));
            $config['size'] = $this->settings['flickr_filter_default_size'];
            break;
        }

        $photoimg = $this->helper->themePhoto($photo, $config['size'], (int) $config['caption']);
        return $this->renderer->render($photoimg);
      }
    }

    return '';
  }

  /**
   * Filter callback for a user or set.
   */
  public function callbackPhotosets($matches): MarkupInterface|string {
    [$config, $attribs] = $this->helper->splitConfig($matches[1]);

    // Class.
    // @todo Implement this.
    if (!isset($attribs['class'])) {
      $attribs['class'] = NULL;
    }

    // Style.
    // @todo Implement this.
    if (!isset($attribs['style'])) {
      $attribs['style'] = NULL;
    }

    // Size.
    if (!isset($config['size'])) {
      $config['size'] = NULL;
    }

    // Photo count.
    if (!isset($config['num'])) {
      $config['num'] = NULL;
    }

    // Media.
    if (!isset($config['media'])) {
      $config['media'] = 'photos';
    }

    // Tags.
    if (!isset($config['tags'])) {
      $config['tags'] = '';
    }
    else {
      $config['tags'] = str_replace("/", ",", $config['tags']);
    }

    // Location options.
    if (!isset($config['location'])) {
      $config['location'][0] = NULL;
      $config['location'][1] = NULL;
      $config['location'][2] = NULL;
    }
    else {
      $config['location'] = explode("/", $config['location']);
      if (!isset($config['location'][2])) {
        $config['location'][2] = NULL;
      }
    }

    // Date options.
    if (!isset($config['date'])) {
      $config['date'][0] = NULL;
      $config['date'][1] = NULL;
    }
    else {
      $config['date'] = explode("|", $config['date']);
      if (!isset($config['date'][1])) {
        $config['date'][1] = NULL;
      }
    }

    // Sort options.
    if (!isset($config['sort'])) {
      $config['sort'] = 'unsorted';
    }

    switch ($config['sort']) {
      case 'taken':
        $config['sort'] = 'date-taken-desc';
        break;

      case 'posted':
        $config['sort'] = 'date-posted-desc';
        break;
    }

    // Show Caption.
    if (!isset($config['caption'])) {
      $config['caption'] = $this->settings['flickr_filter_caption'];
    }

    // Tag Mode.
    if (!isset($config['tag_mode'])) {
      $config['tag_mode'] = 'context';
    }

    // Mintitle?
    // @todo Implement this.
    if (!isset($config['mintitle'])) {
      $config['mintitle'] = NULL;
    }

    // Minmetadata?
    // @todo Implement this.
    if (!isset($config['minmetadata'])) {
      $config['minmetadata'] = NULL;
    }

    // Filter.
    // @todo Implement this.
    if (!isset($config['filter'])) {
      $config['filter'] = NULL;
    }

    switch ($config['filter']) {
      case 'interesting':
        $config['filter'] = 'interestingness-desc';
        break;

      case 'relevant':
        $config['filter'] = 'relevance';
        break;
    }

    // Showtime.
    $photosetPhotos = $this->flickrIntegrationSuiteApiProvider->photosetsGetPhotos(
      $config['id'],
      (int) $config['num'],
      1,
      'photos',
    );

    if ($photosetPhotos['stat'] == 'ok') {
      $photos = $this->helper->themePhotos($photosetPhotos['photo'], $config['size'], $config['caption'], $photosetPhotos['id']);
      $photoset = $this->helper->themePhotoset($photos, $photosetPhotos['title']);
      return $this->renderer->render($photoset);
    }

    return '';
  }

}
