<?php

declare(strict_types=1);

namespace Drupal\flickr_integration_suite_filter;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\flickr_integration_suite\FlickrIntegrationSuiteApiProvider;

/**
 * Provides helper methods.
 */
final class Helper {

  use StringTranslationTrait;

  /**
   * Constructs a Helper object.
   */
  public function __construct(
    private readonly FlickrIntegrationSuiteApiProvider $flickrIntegrationSuiteApiProvider,
  ) {}

  /**
   * Returns the URL to $photo.
   *
   * With size $size using the correct image farm
   * from the $photo variable.
   *
   * @param array $photo
   *   Photo to which the url should point.
   * @param string|null $size
   *   Size of the photo.
   * @param string|null $format
   *   Format of the photo.
   *
   * @return string
   *   URL for $photo with the correct size and format.
   */
  public function photoImgUrl(array $photo, string|null $size = NULL, string|null $format = NULL): string {
    // Early images don't have a farm setting so default to 1.
    $farm = $photo['farm'] ?? 1;
    $server = $photo['server'];
    // photoset's use primary instead of id to specify the image.
    $id = $photo['primary'] ?? $photo['id'];
    $secret = $photo['secret'];
    $suffix = $size ? "_$size." : '.';
    $suffix = $size == '-' ? '.' : $suffix;
    $extension = $size == 'o' ? $format : 'jpg';
    return "https://farm{$farm}.static.flickr.com/{$server}/{$id}_{$secret}" . $suffix . $extension;
  }

  /**
   * A list of possible photo sizes with description and label.
   *
   * @return array
   *   An array of photo sizes.
   */
  public function photoSizes(): array {
    return [
      's' => [
        'label' => 'Square',
        'description' => $this->t('s: 75 px square'),
        'width' => 75,
        'height' => 75,
      ],
      't' => [
        'label' => 'Thumbnail',
        'description' => $this->t('t: 100px on longest side'),
        'width' => 100,
        'height' => 67,
      ],
      'q' => [
        'label' => 'Large Square',
        'description' => $this->t('q: 150px square'),
        'width' => 150,
        'height' => 150,
      ],
      'm' => [
        'label' => 'Small',
        'description' => $this->t('m: 240px on longest side'),
        'width' => 240,
        'height' => 240,
      ],
      'n' => [
        'label' => 'Small 320',
        'description' => $this->t('n: 320px on longest side'),
        'width' => 320,
        'height' => 320,
      ],
      '-' => [
        'label' => 'Medium',
        'description' => $this->t('-: 500px on longest side'),
        'width' => 500,
        'height' => 500,
      ],
      'z' => [
        'label' => 'Medium 640',
        'description' => $this->t('z: 640px on longest side'),
        'width' => 640,
        'height' => 640,
      ],
      'c' => [
        'label' => 'Medium 800',
        'description' => $this->t('c: 800px on longest side'),
        'width' => 800,
        'height' => 800,
      ],
      'b' => [
        'label' => 'Large',
        'description' => $this->t('b: 1024px on longest side'),
        'width' => 1024,
        'height' => 1024,
      ],
      'h' => [
        'label' => 'Large 1600',
        'description' => $this->t('h: 1600px on longest side'),
        'width' => 1600,
        'height' => 1600,
      ],
      'k' => [
        'label' => 'Large 2048',
        'description' => $this->t('k: 2048px on longest side'),
        'width' => 2048,
        'height' => 2048,
      ],
      'o' => [
        'label' => 'Original',
        'description' => $this->t('o: Original image'),
        'width' => 2048,
        'height' => 2048,
      ],
      'x' => [
        'label' => 'slideshow',
        'description' => $this->t('x: Full featured responsive slideshow (for group, set and user IDs only)'),
      ],
      'y' => [
        'label' => 'Simple slideshow',
        'description' => $this->t('y: Basic responsive slideshow (for set and user IDs only)'),
      ],
    ];
  }

  /**
   * Split the config.
   *
   * Parse parameters to the filter from a format like:
   * id=26159919@N00, size=m, num=9
   * into an associative array with two subarrays.
   * The first subarray is parameters for the request, and
   * the second are HTML attributes (class and style).
   *
   * @param string $string
   *   Param String.
   *
   * @return array
   *   Return an array.
   */
  public function splitConfig(string $string): array {
    $config = [];
    $attribs = [];

    // Put each setting on its own line.
    $string = str_replace(',', "\n", $string);

    // Break them up around the equal sign (=).
    preg_match_all('/([a-zA-Z_.]+)=([-@\/0-9a-zA-Z :;_.\|\%"\'&°]+)/', $string, $parts, PREG_SET_ORDER);

    foreach ($parts as $part) {
      // Normalize to the lowercase and remove extra spaces.
      $name = strtolower(trim($part[1]));
      $value = htmlspecialchars_decode(trim($part[2]));

      // Remove undesired but tolerated characters from the value.
      $value = str_replace(str_split('"\''), '', $value);

      if ($name == 'style' || $name == 'class') {
        $attribs[$name] = $value;
      }
      else {
        $config[$name] = $value;
      }
    }

    return [$config, $attribs];
  }

  /**
   * Theme Photos.
   *
   * @param array $photo
   *   Photo.
   * @param string $size
   *   Size.
   * @param int $caption
   *   Caption On Off.
   * @param string|null $parent
   *   Parent.
   *
   * @return array
   *   Return theme array.
   */
  public function themePhoto(array $photo, string $size, int $caption = 0, string|null $parent = NULL): array {
    $photoSize = $this->photoGetSize($photo['id'], $size);
    $photoSizeLarge = $this->photoGetSize($photo['id'], 'b');

    if ($photoSize) {
      $img = [
        '#theme' => 'image',
        '#style_name' => 'flickr-photo-' . $size . '-' . $photoSize['aspect'],
        '#uri' => $this->photoImgUrl($photo, $size),
        '#alt' => $photo['title']['_content'] . ' by ' . $photo['owner']['realname'],
        '#title' => $photo['title']['_content'] . ' by ' . $photo['owner']['realname'],
        '#attributes' => [
          'width' => $photoSize['width'],
          'height' => $photoSize['height'],
        ],
      ];

      $photoimg = [
        '#theme' => 'flickr_photo',
        '#photo' => $img,
        '#caption' => $caption,
        '#photo_page_url' => $photo['urls']['url'][0]['_content'],
        '#photo_image_large' => $photoSizeLarge['source'],
        '#parent' => $parent,
        '#style_name' => 'flickr-photo-' . $size . '-' . $photoSize['aspect'],
        '#width' => $photoSize['width'],
        '#height' => $photoSize['height'],
        '#attached' => [
          'library' => [
            'flickr_integration_suite_filter/flickr.styles',
          ],
        ],
      ];

      if ($caption == 1) {
        $photoimg['#caption_data'] = $this->themeCaption($photo, $size, $caption);
      }

      return $photoimg;
    }

    return [];
  }

  /**
   * Theme Photos.
   *
   * @param array $photos
   *   Photos.
   * @param string $size
   *   Size.
   * @param int $caption
   *   Caption On Off.
   * @param string|null $parent
   *   Parent.
   *
   * @return array
   *   Return theme array.
   */
  public function themePhotos(array $photos, string $size, int $caption = 0, string|null $parent = NULL): array {
    $themedPhotos = [];

    foreach ($photos as $photo) {
      $themedPhotos[] = $this->themePhoto(
        $this->flickrIntegrationSuiteApiProvider->photosGetInfo($photo['id']),
        $size,
        $caption,
        $parent
      );
    }

    return [
      '#theme' => 'flickr_photos',
      '#photos' => $themedPhotos,
      '#attached' => [
        'library' => [
          'flickr_integration_suite_filter/flickr.styles',
        ],
      ],
    ];
  }

  /**
   * Theme Caption.
   *
   * @param array $photo
   *   Photo.
   * @param string $size
   *   Size.
   * @param int $caption
   *   Caption.
   *
   * @return array
   *   Return theme array.
   */
  public function themeCaption(array $photo, string $size, int $caption): array {
    return [
      '#theme' => 'flickr_photo_caption',
      '#caption' => $caption,
      '#caption_realname' => $photo['owner']['realname'],
      '#caption_title' => $photo['title']['_content'],
      '#caption_description' => $photo['description']['_content'],
      '#caption_dateuploaded' => $photo['dateuploaded'],
      '#style_name' => 'flickr-photo-' . $size,
      '#photo_size' => $size,
    ];
  }

  /**
   * Get Photo Size.
   *
   * @param string $photoId
   *   Photo ID.
   * @param string $size
   *   Size.
   *
   * @return bool|array
   *   False or array.
   */
  public function photoGetSize(string $photoId, string $size): bool|array {
    $photoSizes = $this->flickrIntegrationSuiteApiProvider->photosGetSizes($photoId);

    if ($photoSizes) {
      $photoSizes = $photoSizes['sizes']['size'];
      $sizes = $this->photoSizes();
      $label = $sizes[$size]['label'];

      foreach ($photoSizes as $size) {
        if ($size['label'] == $label) {
          $size['width'] = (int) $size['width'];
          $size['height'] = (int) $size['height'];
          $size['aspect'] = $this->photoCalculateAspectRatio($size['width'], $size['height']);
          return $size;
        }
      }
    }

    return FALSE;
  }

  /**
   * Calculate aspect.
   *
   * @param int $width
   *   Width.
   * @param int $height
   *   Height.
   *
   * @return string
   *   Return aspect.
   */
  public function photoCalculateAspectRatio(int $width, int $height): string {
    $aspectRatio = (int) $width / (int) $height;

    if ($aspectRatio > 1) {
      // Image is Landscape.
      return 'landscape';
    }
    if ($aspectRatio < 1) {
      // Image is Portrait.
      return 'portrait';
    }
    else {
      // Image is Square.
      return 'square';
    }
  }

  /**
   * Theme Photos.
   *
   * @param array $photos
   *   Photos.
   * @param string $title
   *   Title.
   *
   * @return array
   *   Theme Array.
   */
  public function themePhotoset(array $photos, string $title): array {
    return [
      '#theme' => 'flickr_photoset',
      '#photos' => $photos,
      '#title' => $title,
      '#attached' => [
        'library' => [
          'flickr_integration_suite_filter/flickr.styles',
        ],
      ],
    ];
  }

}
