<?php

namespace Drupal\flipdown\Plugin\Field\FieldFormatter;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * @FieldFormatter(
 *   id = "flipdown",
 *   label = @Translation("FlipDown countdown"),
 *   field_types = {
 *     "datetime",
 *     "timestamp",
 *     "daterange"
 *   }
 * )
 */
class FlipDownFormatter extends FormatterBase {

  /* ------------------------------------------------------------------------
   * Default settings
   * --------------------------------------------------------------------- */
  public static function defaultSettings() {
    return [
        'end_format' => 'Ended!',
        'color_theme' => 'dark',
      ] + parent::defaultSettings();
  }

  /* ------------------------------------------------------------------------
   * Settings form
   * --------------------------------------------------------------------- */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $field_type = $this->fieldDefinition->getType();

    $form['end_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Text to show after countdown ends'),
      '#default_value' => $this->getSetting('end_format'),
      '#required' => TRUE,
    ];

    $form['color_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('FlipDown colour theme'),
      '#options' => [
        'dark' => $this->t('Dark'),
        'light' => $this->t('Light'),
      ],
      '#default_value' => $this->getSetting('color_theme'),
    ];

    // Only add extra controls when the field itself is a Date-range.
    if ($field_type === 'daterange') {
      $form['range_date'] = [
        '#type' => 'select',
        '#title' => $this->t('Select Start or End date from range for Count down'),
        '#options' => [
          'start' => $this->t('Start date'),
          'end' => $this->t('End date'),
        ],
        '#default_value' => $this->getSetting('range_date') ?? 'start',
      ];
    }

    return $form;
  }

  /* ------------------------------------------------------------------------
   * Field rendering
   * --------------------------------------------------------------------- */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $now = \Drupal::time()->getCurrentTime();
    $targets = [];
    $options = [];
    $ended = $this->getSetting('end_format');
    $theme = $this->getSetting('color_theme');
    $range_date = $this->getSetting('range_date') ?? 'start';

    foreach ($items as $delta => $item) {
      if ($item->isEmpty() || $item->value === NULL || $item->value === '') {
        continue;
      }

      $value = $item->value;
      if ($range_date == 'end') {
        $value = $item->end_value;
      }
      // Convert stored value to Unix seconds.
      if (is_numeric($value)) {
        $target_ts = (int) $value;
      }
      else {
        try {
          $target_ts = (new DrupalDateTime($value, new \DateTimeZone('UTC')))->getTimestamp();
        } catch (\Exception $e) {
          continue;
        }
      }

      $id = 'flipdown-' . $this->fieldDefinition->getName() . '-' . $delta . '-' . random_int(1000, 9999);

      $elements[$delta] = [
        '#markup' => '<div id="' . $id . '" class="flipdown"></div>',
        '#attached' => ['library' => ['flipdown/flipdown']],
      ];

      $targets[$id] = ($target_ts > $now) ? $target_ts : NULL;
      $options[$id] = ['color_theme' => $theme];
    }

    if ($targets) {
      $elements['#attached']['drupalSettings']['flipdown'] = [
        'targets' => $targets,
        'options' => $options,
        'endedText' => $ended,
      ];
    }

    return $elements;
  }

}
