<?php

namespace Drupal\flipdown\Plugin\views\field;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;

/**
 * Renders a FlipDown countdown in Views.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("flipdown_countdown")
 */
class FlipDownCountdown extends FieldPluginBase {

  /* ------------------------------------------------------------------------
   * Default handler options.
   * --------------------------------------------------------------------- */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['color_theme'] = ['default' => 'light'];
    $options['end_format'] = ['default' => 'Ended!'];
    return $options;
  }

  /* ------------------------------------------------------------------------
   * Options form shown in the Views UI.
   * --------------------------------------------------------------------- */
  public function buildOptionsForm(&$form, \Drupal\Core\Form\FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['color_theme'] = [
      '#type' => 'select',
      '#title' => $this->t('FlipDown colour theme'),
      '#options' => [
        'light' => $this->t('Light'),
        'dark' => $this->t('Dark'),
      ],
      '#default_value' => $this->options['color_theme'],
      '#weight' => -9,
    ];
    $form['end_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Text to show after countdown ends'),
      '#default_value' => $this->options['end_format'],
      '#weight' => -8,
    ];
  }

  /* ------------------------------------------------------------------------
   * Tell Views to pull the raw database value (no formatting).
   * --------------------------------------------------------------------- */
  public function query() {
    $this->ensureMyTable();
    // Grab the raw column value for this field.
    $this->addField($this->relationship, $this->realField);
  }

  /* ------------------------------------------------------------------------
   * Render each row.
   * --------------------------------------------------------------------- */
  public function render(ResultRow $values) {
    // Value straight from DB.
    $raw = $this->getValue($values);

    if ($raw === NULL || $raw === '') {
      return [];
    }

    // Convert to Unix seconds.
    if (is_numeric($raw)) {
      $ts = (int) $raw;
    }
    else {
      try {
        $ts = (new DrupalDateTime($raw))->getTimestamp();
      } catch (\Exception $e) {
        return [];
      }
    }

    // Unique ID for this row (view id + display id + position in result set).
    $id = 'flipdown-v-' . $this->view->storage->id() . '-' . $this->view->current_display . '-' . $this->view->row_index;

    // Build render array.
    $build = [
      '#markup' => '<div id="' . $id . '" class="flipdown"></div>',
      '#attached' => [
        'library' => ['flipdown/flipdown'],
        'drupalSettings' => [
          // Attach one target & option for this element.  Each handler instance
          // uses its own namespace so multiple FlipDown fields in the same
          // view don’t collide.
          'flipdown' => [
            'targets' => [
              $id => ($ts > \Drupal::time()
                  ->getCurrentTime()) ? $ts : NULL,
            ],
            'options' => [$id => ['color_theme' => $this->options['color_theme']]],
            'endedText' => $this->options['end_format'],
          ],
        ],
      ],
    ];

    return $build;
  }

}
