<?php

namespace Drupal\flood_alerts_england;

use Drupal\Core\Http\ClientFactory;
use Drupal\Core\Logger\LoggerChannel;
use Drupal\Core\Logger\LoggerChannelFactory;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Client for the Environment Agency Flood Monitoring API.
 */
class FloodApiClient {

  protected Client $httpClient;
  protected LoggerChannel $logger;

  protected string $base_url = 'https://environment.data.gov.uk/flood-monitoring/id/floods';

  public function __construct(ClientFactory $http_client_factory,  LoggerChannelFactory $logger) {
    $this->httpClient = $http_client_factory->fromOptions([]);
    $this->logger = $logger->get('flood_alerts_england');
  }

  /**
   * Fetch flood alerts (floods endpoint) with optional query params.
   *
   * @param array $query
   * @return array|null
   */
  public function getFloodAlerts(array $query = []): ?array {

    // Assemble query string.
    $options = [
      'query' => $query,
      'headers' => [
        'Accept' => 'application/json',
        'User-Agent' => 'flood_alerts_england/1.0 (+https://example.org)'
      ],
      'timeout' => 10,
    ];

    try {
      $response = $this->httpClient->get($this->base_url, $options);
      $status = $response->getStatusCode();
      if ($status === 200) {
        $body = (string) $response->getBody();
        return json_decode($body, TRUE);
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('Flood API request failed: @msg', ['@msg' => $e->getMessage()]);
    }

    return NULL;
  }
}
