<?php

namespace Drupal\flood_alerts_england\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\flood_alerts_england\FloodApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a 'Flood Alerts' Block.
 *
 * @Block(
 * id = "flood_alerts_england_block",
 * admin_label = @Translation("Flood Alerts"),
 * )
 */
class FloodAlertsBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected FloodApiClient $client;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, FloodApiClient $client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->client = $client;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('flood_alerts_england.flood_api_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      // How to filter alerts. Options: county | flood_area | proximity.
      'filter_mode' => 'county',
      // County name (used when filter_mode = county).
      'county' => '',
      // Flood area ID (used when filter_mode = flood_area).
      'flood_area_id' => '',
      // Proximity search (used when filter_mode = proximity).
      'lat' => '',
      'lon' => '',
      'radius' => '5',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, \Drupal\Core\Form\FormStateInterface $form_state) {

    $form['filter_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Filter method'),
      '#options' => [
        'county' => $this->t('County name'),
        'flood_area' => $this->t('Flood area ID'),
        'proximity' => $this->t('Latitude/Longitude with radius'),
      ],
      '#default_value' => $this->configuration['filter_mode'] ?? 'county',
      '#description' => $this->t('Choose how to filter flood alerts. See the Environment Agency flood monitoring API reference for supported filters.'),
    ];

    $form['county'] = [
      '#type' => 'select',
      '#title' => $this->t('County'),
      '#description' => $this->t('Filter alerts by county name.'),
      '#options' => $this->getCountyOptions(),
      '#empty_option' => $this->t('- Select -'),
      '#default_value' => $this->configuration['county'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'county'],
        ],
        'required' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'county'],
        ],
      ],
    ];

    $form['flood_area_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Flood area ID'),
      '#description' => $this->t('Filter by Environment Agency floodAreaID, see https://environment.data.gov.uk/flood-monitoring/id/floodAreas'),
      '#default_value' => $this->configuration['flood_area_id'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'flood_area'],
        ],
        'required' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'flood_area'],
        ],
      ],
    ];

    $form['lat'] = [
      '#type' => 'number',
      '#title' => $this->t('Latitude'),
      '#description' => $this->t('Latitude in decimal degrees (-90 to 90).'),
      '#default_value' => $this->configuration['lat'] ?? '',
      '#step' => 'any',
      '#states' => [
        'visible' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
        'required' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
      ],
    ];

    $form['lon'] = [
      '#type' => 'number',
      '#title' => $this->t('Longitude'),
      '#description' => $this->t('Longitude in decimal degrees (-180 to 180).'),
      '#default_value' => $this->configuration['lon'] ?? '',
      '#step' => 'any',
      '#states' => [
        'visible' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
        'required' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
      ],
    ];

    $form['radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Radius (km)'),
      '#description' => $this->t('Search radius in kilometres.'),
      '#default_value' => $this->configuration['radius'] ?? '5',
      '#min' => 0,
      '#step' => 'any',
      '#states' => [
        'visible' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
        'required' => [
          ':input[name="settings[filter_mode]"]' => ['value' => 'proximity'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['filter_mode'] = $form_state->getValue('filter_mode');
    $this->configuration['county'] = $form_state->getValue('county');
    $this->configuration['flood_area_id'] = $form_state->getValue('flood_area_id');
    $this->configuration['lat'] = $form_state->getValue('lat');
    $this->configuration['lon'] = $form_state->getValue('lon');
    $this->configuration['radius'] = $form_state->getValue('radius');
  }

  /**
   * {@inheritdoc}
   */
  public function blockValidate($form, FormStateInterface $form_state): void {
    $mode = $form_state->getValue('filter_mode');
    if ($mode === 'county') {
      $county = $form_state->getValue('county');
      if ($county === '' || $county === NULL) {
        $form_state->setErrorByName('county', $this->t('Please select a county.'));
      }
    }
    if ($mode === 'flood_area') {
      $fa = trim((string) $form_state->getValue('flood_area_id'));
      if ($fa === '') {
        $form_state->setErrorByName('flood_area_id', $this->t('Flood area ID is required.'));
      }
    }
    if ($mode === 'proximity') {
      $lat = $form_state->getValue('lat');
      $lon = $form_state->getValue('lon');
      $radius = $form_state->getValue('radius');
      if ($lat === '' || !is_numeric($lat) || $lat < -90 || $lat > 90) {
        $form_state->setErrorByName('lat', $this->t('Latitude must be a number between -90 and 90.'));
      }
      if ($lon === '' || !is_numeric($lon) || $lon < -180 || $lon > 180) {
        $form_state->setErrorByName('lon', $this->t('Longitude must be a number between -180 and 180.'));
      }
      if ($radius === '' || !is_numeric($radius) || $radius <= 0) {
        $form_state->setErrorByName('radius', $this->t('Radius must be a positive number.'));
      }
    }
  }

  public function build(): array {
    // Build query from block configuration.
    $query = [];
    $mode = $this->configuration['filter_mode'] ?? 'county';
    if ($mode === 'flood_area') {
      $flood_area_id = trim((string) ($this->configuration['flood_area_id'] ?? ''));
      if ($flood_area_id !== '') {
        // Property filter by floodAreaID.
        $query['floodAreaID'] = $flood_area_id;
      }
    }
    elseif ($mode === 'proximity') {
      $lat = $this->configuration['lat'] ?? '';
      $lon = $this->configuration['lon'] ?? '';
      $radius = $this->configuration['radius'] ?? '';
      if ($lat !== '' && $lon !== '' && $radius !== '') {
        $query['lat'] = $lat;
        $query['long'] = $lon;
        $query['dist'] = $radius;
      }
    }
    else {
      // Default to county name filter.
      $county = trim((string) ($this->configuration['county'] ?? ''));
      if ($county !== '') {
        $query['county'] = $county;
      }
    }

    // Fetch alerts.
    $data = $this->client->getFloodAlerts($query);

    $items = [];
    if (!empty($data) && isset($data['items']) && is_array($data['items'])) {
      foreach ($data['items'] as $item) {
        // Prepare timestamps as Unix time so Twig can use Drupal's format_date.
        $tmc_raw = $item['timeMessageChanged'] ?? ($item['timeUpdated'] ?? ($item['date'] ?? ''));
        $tr_raw = $item['timeRaised'] ?? NULL;
        $tsc_raw = $item['timeSeverityChanged'] ?? NULL;

        $tmc_ts = !empty($tmc_raw) ? strtotime($tmc_raw) : NULL;
        $tr_ts = !empty($tr_raw) ? strtotime($tr_raw) : NULL;
        $tsc_ts = !empty($tsc_raw) ? strtotime($tsc_raw) : NULL;

        $items[] = [
          // Use exact API key names as requested.
          'message' => $item['message'] ?? ($item['label'] ?? ($item['description'] ?? '')),
          'severity' => $item['severity'] ?? '',
          'severityLevel' => $item['severityLevel'] ?? NULL,
          'timeMessageChanged' => $tmc_ts,
          'timeRaised' => $tr_ts,
          'timeSeverityChanged' => $tsc_ts,

          // Description used for the title link as requested.
          'description' => $item['description'] ?? ($item['area'] ?? ($item['label'] ?? '')),

          // Additional helpful fields retained for display where available.
          'area' => $item['area'] ?? ($item['eaAreaName'] ?? ''),
          'floodAreaID' => $item['floodAreaID'] ?? NULL,

          // Back-compat keys (can be removed later if not needed by themes):
          'severity_level' => $item['severityLevel'] ?? NULL,
          'flood_area_id' => $item['floodAreaID'] ?? NULL,
          'updated' => $item['date'] ?? ($item['timeUpdated'] ?? ''),
        ];
      }
    }

    $build = [
      '#theme' => 'flood_alerts_england',
      '#alerts' => $items,
      '#cache' => [
        'max-age' => 900,
      ],
    ];

    return $build;
  }

  /**
   * Returns a list of English counties for selection.
   * The Environment Agency Flood Monitoring API covers England.
   */
  protected function getCountyOptions(): array {
    $counties = [
      'Bedfordshire', 'Berkshire', 'Bristol', 'Buckinghamshire', 'Cambridgeshire', 'Cheshire',
      'City of London', 'Cornwall', 'County Durham', 'Cumbria', 'Derbyshire', 'Devon',
      'Dorset', 'East Riding of Yorkshire', 'East Sussex', 'Essex', 'Gloucestershire',
      'Greater London', 'Greater Manchester', 'Hampshire', 'Herefordshire', 'Hertfordshire',
      'Isle of Wight', 'Kent', 'Lancashire', 'Leicestershire', 'Lincolnshire', 'Merseyside',
      'Norfolk', 'North Yorkshire', 'Northamptonshire', 'Northumberland', 'Nottinghamshire',
      'Oxfordshire', 'Rutland', 'Shropshire', 'Somerset', 'South Yorkshire', 'Staffordshire',
      'Suffolk', 'Surrey', 'Tyne and Wear', 'Warwickshire', 'West Midlands', 'West Sussex',
      'West Yorkshire', 'Wiltshire', 'Worcestershire',
    ];

    // Build an associative array of options.
    $options = [];
    foreach ($counties as $name) {
      $options[$name] = $name;
    }
    asort($options, SORT_NATURAL | SORT_FLAG_CASE);
    return $options;
  }
}
