<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_processor\Plugin\FlowDropNodeProcessor;

use Drupal\flowdrop\Attribute\FlowDropNodeProcessor;
use Drupal\flowdrop\DTO\ParameterBagInterface;
use Drupal\flowdrop\DTO\ValidationResult;
use Drupal\flowdrop\Plugin\FlowDropNodeProcessor\AbstractFlowDropNodeProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Executor for Regex Extractor nodes.
 */
#[FlowDropNodeProcessor(
  id: "regex_extractor",
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup("Regex Extractor"),
  description: "Extract data using regular expressions",
  version: "1.0.0"
)]
class RegexExtractor extends AbstractFlowDropNodeProcessor {

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function validateParams(array $params): ValidationResult {
    // Regex extractor nodes can accept any inputs or none.
    return ValidationResult::success();
  }

  /**
   * {@inheritdoc}
   */
  protected function process(ParameterBagInterface $params): array {
    $pattern = $params->get("pattern", "");
    $flags = $params->get("flags", "");
    $matchAll = $params->get("matchAll", FALSE);
    $text = $params->get("text") ?: "";

    $matches = [];
    if (!empty($pattern) && !empty($text)) {
      $regex = "/" . $pattern . "/" . $flags;

      if ($matchAll) {
        preg_match_all($regex, $text, $matches, PREG_SET_ORDER);
      }
      else {
        preg_match($regex, $text, $matches);
      }
    }

    return [
      "matches" => $matches,
      "pattern" => $pattern,
      "flags" => $flags,
      "match_all" => $matchAll,
      "total_matches" => count($matches),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getParameterSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "text" => [
          "type" => "string",
          "title" => "Text",
          "description" => "The text to extract from",
          "required" => FALSE,
        ],
        "pattern" => [
          "type" => "string",
          "title" => "Pattern",
          "description" => "The regex pattern to match",
          "default" => "",
        ],
        "flags" => [
          "type" => "string",
          "title" => "Flags",
          "description" => "Regex flags (i, m, s, x, etc.)",
          "default" => "",
        ],
        "matchAll" => [
          "type" => "boolean",
          "title" => "Match All",
          "description" => "Whether to match all occurrences",
          "default" => FALSE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getOutputSchema(): array {
    return [
      "type" => "object",
      "properties" => [
        "matches" => [
          "type" => "array",
          "description" => "The regex matches",
        ],
        "pattern" => [
          "type" => "string",
          "description" => "The regex pattern used",
        ],
        "flags" => [
          "type" => "string",
          "description" => "The regex flags used",
        ],
        "match_all" => [
          "type" => "boolean",
          "description" => "Whether to match all occurrences",
        ],
        "total_matches" => [
          "type" => "integer",
          "description" => "The total number of matches",
        ],
      ],
    ];
  }

}
