<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_type\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\flowdrop_node_type\FlowDropNodeTypeInterface;
use Drupal\flowdrop_node_type\FlowDropNodeTypeListBuilder;
use Drupal\flowdrop_node_type\Form\FlowDropNodeTypeForm;

/**
 * Defines the flowdrop node type entity type.
 */
#[ConfigEntityType(
  id: 'flowdrop_node_type',
  label: new TranslatableMarkup('FlowDrop Node Type'),
  label_collection: new TranslatableMarkup('FlowDrop Node Types'),
  label_singular: new TranslatableMarkup('FlowDrop Node Type'),
  label_plural: new TranslatableMarkup('FlowDrop Node Types'),
  config_prefix: 'flowdrop_node_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => FlowDropNodeTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
    'form' => [
      'add' => FlowDropNodeTypeForm::class,
      'edit' => FlowDropNodeTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
  ],
  links: [
    'collection' => '/admin/flowdrop/config/node-types',
    'add-form' => '/admin/flowdrop/config/node-types/add',
    'edit-form' => '/admin/flowdrop/config/node-types/{flowdrop_node_type}',
    'delete-form' => '/admin/flowdrop/config/node-types/{flowdrop_node_type}/delete',
  ],
  admin_permission: 'administer flowdrop_node_type',
  label_count: [
    'singular' => '@count flowdrop node type',
    'plural' => '@count flowdrop node types',
  ],
  config_export: [
    'id',
    'label',
    'uuid',
    'description',
    'category',
    'icon',
    'color',
    'plugin_version',
    'enabled',
    'parameters',
    'outputs',
    'tags',
    'executor_plugin',
    'visual_type',
    'supported_visual_types',
  ],
)]
final class FlowDropNodeType extends ConfigEntityBase implements FlowDropNodeTypeInterface {

  /**
   * The node type ID.
   */
  protected string $id;

  /**
   * The node type label.
   */
  protected string $label;

  /**
   * The node type description.
   */
  protected string $description = '';

  /**
   * The node category.
   */
  protected string $category = 'processing';

  /**
   * The node icon.
   */
  protected string $icon = 'mdi:cog';

  /**
   * The node color.
   */
  protected string $color = '#007cba';

  /**
   * The plugin version stored when this config was saved.
   *
   * This is automatically captured from the selected executor plugin's version
   * attribute and is used to detect if the plugin has been updated since this
   * config was last saved.
   */
  protected string $plugin_version = '';

  /**
   * Whether the node type is enabled.
   */
  protected bool $enabled = TRUE;

  /**
   * The unified parameter configuration.
   *
   * Structure:
   * [
   *   'paramName' => [
   *     'configurable' => true,
   *     'connectable' => false,
   *     'required' => false,
   *     'default' => 'value',
   *   ],
   * ]
   *
   * @var array<string, array<string, mixed>>
   */
  protected array $parameters = [];

  /**
   * The output configuration.
   *
   * Structure:
   * [
   *   'outputName' => [
   *     'exposed' => true,
   *   ],
   * ]
   *
   * @var array<string, array<string, mixed>>
   */
  protected array $outputs = [];

  /**
   * The node tags.
   *
   * @var array<string>
   */
  protected array $tags = [];

  /**
   * The executor plugin ID.
   */
  protected string $executor_plugin = '';

  /**
   * The visual type for rendering in the workflow editor.
   *
   * This determines how the node appears visually (e.g., "default", "gateway").
   * Site builders can choose the appropriate visual representation.
   */
  protected string $visual_type = 'default';

  /**
   * The supported visual types for this node.
   *
   * When multiple types are supported, users can change the visual type
   * per node instance in the workflow editor.
   *
   * @var array<string>
   */
  protected array $supported_visual_types = ['default'];

  /**
   * Get the label.
   *
   * @return string
   *   The label.
   */
  public function getLabel(): string {
    return $this->label;
  }

  /**
   * Set the label.
   *
   * @param string $label
   *   The label.
   *
   * @return static
   *   The node entity.
   */
  public function setLabel(string $label): static {
    $this->label = $label;
    return $this;
  }

  /**
   * Get the description.
   *
   * @return string
   *   The description.
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * Set the description.
   *
   * @param string $description
   *   The description.
   *
   * @return static
   *   The node entity.
   */
  public function setDescription(string $description): static {
    $this->description = $description;
    return $this;
  }

  /**
   * Get the category.
   *
   * @return string
   *   The category.
   */
  public function getCategory(): string {
    return $this->category;
  }

  /**
   * Set the category.
   *
   * @param string $category
   *   The category.
   *
   * @return static
   *   The node entity.
   */
  public function setCategory(string $category): static {
    $this->category = $category;
    return $this;
  }

  /**
   * Get the icon.
   *
   * @return string
   *   The icon.
   */
  public function getIcon(): string {
    return $this->icon;
  }

  /**
   * Set the icon.
   *
   * @param string $icon
   *   The icon.
   *
   * @return static
   *   The node entity.
   */
  public function setIcon(string $icon): static {
    $this->icon = $icon;
    return $this;
  }

  /**
   * Get the color.
   *
   * @return string
   *   The color.
   */
  public function getColor(): string {
    return $this->color;
  }

  /**
   * Set the color.
   *
   * @param string $color
   *   The color.
   *
   * @return static
   *   The node entity.
   */
  public function setColor(string $color): static {
    $this->color = $color;
    return $this;
  }

  /**
   * Get the plugin version stored when this config was saved.
   *
   * @return string
   *   The plugin version.
   */
  public function getPluginVersion(): string {
    return $this->plugin_version;
  }

  /**
   * Set the plugin version.
   *
   * This should be set automatically from the selected executor plugin's
   * version, not manually by users.
   *
   * @param string $plugin_version
   *   The plugin version.
   *
   * @return static
   *   The node entity.
   */
  public function setPluginVersion(string $plugin_version): static {
    $this->plugin_version = $plugin_version;
    return $this;
  }

  /**
   * Check if the node type is enabled.
   *
   * @return bool
   *   TRUE if enabled.
   */
  public function isEnabled(): bool {
    return $this->enabled;
  }

  /**
   * Set the enabled status.
   *
   * @param bool $enabled
   *   The enabled status.
   *
   * @return static
   *   The node entity.
   */
  public function setEnabled(bool $enabled): static {
    $this->enabled = $enabled;
    return $this;
  }

  /**
   * Get the unified parameter configuration.
   *
   * @return array<string, array<string, mixed>>
   *   The parameter configuration.
   */
  public function getParameters(): array {
    return $this->parameters;
  }

  /**
   * Set the unified parameter configuration.
   *
   * @param array<string, array<string, mixed>> $parameters
   *   The parameter configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setParameters(array $parameters): static {
    $this->parameters = $parameters;
    return $this;
  }

  /**
   * Get configuration for a specific parameter.
   *
   * @param string $name
   *   The parameter name.
   *
   * @return array<string, mixed>|null
   *   The parameter configuration, or NULL if not found.
   */
  public function getParameter(string $name): ?array {
    return $this->parameters[$name] ?? NULL;
  }

  /**
   * Set configuration for a specific parameter.
   *
   * @param string $name
   *   The parameter name.
   * @param array<string, mixed> $config
   *   The parameter configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setParameter(string $name, array $config): static {
    $this->parameters[$name] = $config;
    return $this;
  }

  /**
   * Get the output configuration.
   *
   * @return array<string, array<string, mixed>>
   *   The output configuration.
   */
  public function getOutputs(): array {
    return $this->outputs;
  }

  /**
   * Set the output configuration.
   *
   * @param array<string, array<string, mixed>> $outputs
   *   The output configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setOutputs(array $outputs): static {
    $this->outputs = $outputs;
    return $this;
  }

  /**
   * Get configuration for a specific output.
   *
   * @param string $name
   *   The output name.
   *
   * @return array<string, mixed>|null
   *   The output configuration, or NULL if not found.
   */
  public function getOutput(string $name): ?array {
    return $this->outputs[$name] ?? NULL;
  }

  /**
   * Set configuration for a specific output.
   *
   * @param string $name
   *   The output name.
   * @param array<string, mixed> $config
   *   The output configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setOutput(string $name, array $config): static {
    $this->outputs[$name] = $config;
    return $this;
  }

  /**
   * Get the tags.
   *
   * @return array
   *   The tags.
   */
  public function getTags(): array {
    return $this->tags;
  }

  /**
   * Set the tags.
   *
   * @param array $tags
   *   The tags.
   *
   * @return static
   *   The node entity.
   */
  public function setTags(array $tags): static {
    $this->tags = $tags;
    return $this;
  }

  /**
   * Get the executor plugin ID.
   *
   * @return string
   *   The executor plugin ID.
   */
  public function getExecutorPlugin(): string {
    return $this->executor_plugin;
  }

  /**
   * Set the executor plugin ID.
   *
   * @param string $executor_plugin
   *   The executor plugin ID.
   *
   * @return static
   *   The node entity.
   */
  public function setExecutorPlugin(string $executor_plugin): static {
    $this->executor_plugin = $executor_plugin;
    return $this;
  }

  /**
   * Get the visual type for this node.
   *
   * @return string
   *   The visual type (e.g., "default", "gateway", "simple", "square").
   */
  public function getVisualType(): string {
    return $this->visual_type;
  }

  /**
   * Set the visual type for this node.
   *
   * @param string $visual_type
   *   The visual type.
   *
   * @return static
   *   The node entity.
   */
  public function setVisualType(string $visual_type): static {
    $this->visual_type = $visual_type;
    return $this;
  }

  /**
   * Get the supported visual types for this node.
   *
   * @return array<string>
   *   Array of supported visual type keys.
   */
  public function getSupportedVisualTypes(): array {
    return $this->supported_visual_types;
  }

  /**
   * Set the supported visual types for this node.
   *
   * @param array<string> $supported_visual_types
   *   Array of supported visual type keys.
   *
   * @return static
   *   The node entity.
   */
  public function setSupportedVisualTypes(array $supported_visual_types): static {
    $this->supported_visual_types = $supported_visual_types;
    return $this;
  }

  /**
   * Transform singular category to plural form for API compatibility.
   *
   * @param string $category
   *   The singular category.
   *
   * @return string
   *   The plural category.
   */
  protected function transformCategoryToPlural(string $category): string {
    $category_mapping = [
      'input' => 'inputs',
      'output' => 'outputs',
      'model' => 'models',
      'prompt' => 'prompts',
      'processing' => 'processing',
      'logic' => 'logic',
      'data' => 'data',
      'helper' => 'helpers',
      'tool' => 'tools',
      'vectorstore' => 'vectorstores',
      'embedding' => 'embeddings',
      'memory' => 'memories',
      'agent' => 'agents',
      'bundle' => 'bundles',
    ];

    return $category_mapping[$category] ?? $category;
  }

  /**
   * Convert to node definition format.
   *
   * Produces a node definition array for API responses using the
   * Unified Parameter System.
   *
   * @return array<string, mixed>
   *   The node definition array.
   */
  public function toNodeDefinition(): array {
    return [
      "id" => $this->id,
      "name" => $this->label,
      "type" => $this->visual_type,
      "supportedTypes" => $this->supported_visual_types,
      "plugin_version" => $this->plugin_version,
      "description" => $this->description,
      "category" => $this->transformCategoryToPlural($this->category),
      "icon" => $this->icon,
      "color" => $this->color,
      "tags" => $this->tags,
      "enabled" => $this->enabled,
      "parameters" => $this->parameters,
      "outputs" => $this->outputs,
      "executor_plugin" => $this->executor_plugin,
    ];
  }

}
