<?php

declare(strict_types=1);

namespace Drupal\flowdrop_node_type;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Provides an interface defining a flowdrop node type entity type.
 *
 * This interface uses the Unified Parameter System for all node configuration.
 * Parameters define both connectable inputs and configurable settings.
 *
 * @see docs/development/flowdrop-node-processor.md
 */
interface FlowDropNodeTypeInterface extends ConfigEntityInterface {

  /**
   * Get the label.
   *
   * @return string
   *   The label.
   */
  public function getLabel(): string;

  /**
   * Set the label.
   *
   * @param string $label
   *   The label.
   *
   * @return static
   *   The node entity.
   */
  public function setLabel(string $label): static;

  /**
   * Get the description.
   *
   * @return string
   *   The description.
   */
  public function getDescription(): string;

  /**
   * Set the description.
   *
   * @param string $description
   *   The description.
   *
   * @return static
   *   The node entity.
   */
  public function setDescription(string $description): static;

  /**
   * Get the category.
   *
   * @return string
   *   The category.
   */
  public function getCategory(): string;

  /**
   * Set the category.
   *
   * @param string $category
   *   The category.
   *
   * @return static
   *   The node entity.
   */
  public function setCategory(string $category): static;

  /**
   * Get the icon.
   *
   * @return string
   *   The icon.
   */
  public function getIcon(): string;

  /**
   * Set the icon.
   *
   * @param string $icon
   *   The icon.
   *
   * @return static
   *   The node entity.
   */
  public function setIcon(string $icon): static;

  /**
   * Get the color.
   *
   * @return string
   *   The color.
   */
  public function getColor(): string;

  /**
   * Set the color.
   *
   * @param string $color
   *   The color.
   *
   * @return static
   *   The node entity.
   */
  public function setColor(string $color): static;

  /**
   * Get the plugin version stored when this config was saved.
   *
   * This version is automatically captured from the selected executor plugin
   * and is used to detect if the plugin has been updated since this config
   * was last saved.
   *
   * @return string
   *   The plugin version.
   */
  public function getPluginVersion(): string;

  /**
   * Set the plugin version.
   *
   * This should be set automatically from the selected executor plugin's
   * version, not manually by users.
   *
   * @param string $plugin_version
   *   The plugin version.
   *
   * @return static
   *   The node entity.
   */
  public function setPluginVersion(string $plugin_version): static;

  /**
   * Check if the node type is enabled.
   *
   * @return bool
   *   TRUE if enabled.
   */
  public function isEnabled(): bool;

  /**
   * Set the enabled status.
   *
   * @param bool $enabled
   *   The enabled status.
   *
   * @return static
   *   The node entity.
   */
  public function setEnabled(bool $enabled): static;

  /**
   * Get the unified parameter configuration.
   *
   * @return array<string, array<string, mixed>>
   *   The parameter configuration.
   */
  public function getParameters(): array;

  /**
   * Set the unified parameter configuration.
   *
   * @param array<string, array<string, mixed>> $parameters
   *   The parameter configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setParameters(array $parameters): static;

  /**
   * Get configuration for a specific parameter.
   *
   * @param string $name
   *   The parameter name.
   *
   * @return array<string, mixed>|null
   *   The parameter configuration, or NULL if not found.
   */
  public function getParameter(string $name): ?array;

  /**
   * Set configuration for a specific parameter.
   *
   * @param string $name
   *   The parameter name.
   * @param array<string, mixed> $config
   *   The parameter configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setParameter(string $name, array $config): static;

  /**
   * Get the output configuration.
   *
   * @return array<string, array<string, mixed>>
   *   The output configuration.
   */
  public function getOutputs(): array;

  /**
   * Set the output configuration.
   *
   * @param array<string, array<string, mixed>> $outputs
   *   The output configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setOutputs(array $outputs): static;

  /**
   * Get configuration for a specific output.
   *
   * @param string $name
   *   The output name.
   *
   * @return array<string, mixed>|null
   *   The output configuration, or NULL if not found.
   */
  public function getOutput(string $name): ?array;

  /**
   * Set configuration for a specific output.
   *
   * @param string $name
   *   The output name.
   * @param array<string, mixed> $config
   *   The output configuration.
   *
   * @return static
   *   The node entity.
   */
  public function setOutput(string $name, array $config): static;

  /**
   * Get the tags.
   *
   * @return array<string>
   *   The tags.
   */
  public function getTags(): array;

  /**
   * Set the tags.
   *
   * @param array<string> $tags
   *   The tags.
   *
   * @return static
   *   The node entity.
   */
  public function setTags(array $tags): static;

  /**
   * Get the executor plugin ID.
   *
   * @return string
   *   The executor plugin ID.
   */
  public function getExecutorPlugin(): string;

  /**
   * Set the executor plugin ID.
   *
   * @param string $executor_plugin
   *   The executor plugin ID.
   *
   * @return static
   *   The node entity.
   */
  public function setExecutorPlugin(string $executor_plugin): static;

  /**
   * Get the visual type for this node.
   *
   * The visual type determines how the node is rendered in the workflow editor.
   * This is configurable by site builders, not hardcoded in plugins.
   *
   * @return string
   *   The visual type (e.g., "default", "gateway", "simple", "square").
   */
  public function getVisualType(): string;

  /**
   * Set the visual type for this node.
   *
   * @param string $visual_type
   *   The visual type.
   *
   * @return static
   *   The node entity.
   */
  public function setVisualType(string $visual_type): static;

  /**
   * Get the supported visual types for this node.
   *
   * @return array<string>
   *   Array of supported visual type keys.
   */
  public function getSupportedVisualTypes(): array;

  /**
   * Set the supported visual types for this node.
   *
   * @param array<string> $supported_visual_types
   *   Array of supported visual type keys.
   *
   * @return static
   *   The node entity.
   */
  public function setSupportedVisualTypes(array $supported_visual_types): static;

  /**
   * Convert to node definition format.
   *
   * @return array<string, mixed>
   *   The node definition array.
   */
  public function toNodeDefinition(): array;

}
